﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/detail/fs_PriorityUtility.h>
#include <nn/fssrv/fssrv_RequestHook.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/sf/sf_Result.h>

namespace nn { namespace fssrv {

Result RequestHook::BeforeInvoke() NN_NOEXCEPT
{
    auto beforeInvoke = [&]() NN_NOEXCEPT -> Result
    {
        auto* pRequestHookContext = reinterpret_cast<RequestHookContext*>(fssystem::GetServiceContext()->GetRequestHookContext());

        pRequestHookContext->m_OldPriority = fssystem::GetServiceContext()->GetPriority();

        fs::PriorityRaw fsPriority;
        NN_RESULT_TRY(fs::detail::ConvertTlsIoPriorityToFsPriority(&fsPriority, nn::fs::detail::GetTlsIoPriority()))
            NN_RESULT_CATCH_ALL
            {
                fsPriority = fs::PriorityRaw_Normal;
            }
        NN_RESULT_END_TRY;
        fssystem::GetServiceContext()->SetPriority(fsPriority);

        pRequestHookContext->m_OldInlineContext = fs::detail::GetTlsIoValueForInheritance();
        if( !fs::detail::IsCurrentRequestRecursive() )
        {
            NN_RESULT_DO(m_pSessionResourceManager->Acquire(fsPriority));
        }
        fs::detail::SetCurrentRequestRecursive();

        pRequestHookContext->m_PriorityChanger.emplace(fssystem::ScopedThreadPriorityChangerByAccessPriority::AccessMode::Read);

        NN_RESULT_SUCCESS;
    };

    NN_RESULT_TRY(beforeInvoke())
        NN_RESULT_CATCH(nn::sf::ResultProcessDeferred)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_SDK_ASSERT(false);
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

void RequestHook::AfterInvoke() NN_NOEXCEPT
{
    auto* pRequestHookContext = reinterpret_cast<RequestHookContext*>(fssystem::GetServiceContext()->GetRequestHookContext());

    pRequestHookContext->m_PriorityChanger = util::nullopt;
    fs::detail::SetTlsIoValueForInheritance(pRequestHookContext->m_OldInlineContext);
    if( !fs::detail::IsCurrentRequestRecursive() )
    {
        m_pSessionResourceManager->Release();
    }
    fssystem::GetServiceContext()->SetPriority(pRequestHookContext->m_OldPriority);
}

}}
