﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/fssrv_SaveDataSharedFileStorage.h>
#include <nn/fssrv/fscreator/fssrv_SaveDataFileSystemCreator.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_ApplicationTemporaryFileSystem.h>
#include <nn/fssystem/fs_DirectorySaveDataFileSystem.h>
#include <nn/fssystem/fs_IMacGenerator.h>
#include <nn/fssystem/fs_ISaveDataFileSystemCacheManager.h>
#include <nn/fssystem/fs_SaveDataFileSystem.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheRegister.h>
#include <nn/fssystem/fs_SubdirectoryFileSystem.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/save/fs_SaveDataInternalStorageFileSystem.h>

#include "../detail/fssrv_SaveDataExtender.h"

using namespace nn::fs;
using namespace nn::fssystem;

namespace nn { namespace fssrv { namespace fscreator {

    namespace {
        class SaveDataFileSystemVisitor : public ISaveDataFileSystemVisitor
        {
        public:
            virtual ~SaveDataFileSystemVisitor() NN_NOEXCEPT NN_OVERRIDE {}

        public:
            virtual void Visit(fssystem::save::IInternalStorageFileSystem* pFileSystem) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
                m_pFileSystem = pFileSystem;
            }

            fssystem::save::IInternalStorageFileSystem* GetFileSystem() NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
                return m_pFileSystem;
            }

        private:
            fssystem::save::IInternalStorageFileSystem* m_pFileSystem;
        };

        class SaveDataInternalStorageFileSystem : public save::SaveDataInternalStorageFileSystem
        {
        public:
            SaveDataInternalStorageFileSystem() NN_NOEXCEPT
            {
            }

            virtual ~SaveDataInternalStorageFileSystem() NN_NOEXCEPT NN_OVERRIDE {}

        public:
            Result Initialize(
                std::shared_ptr<fssystem::SaveDataFileSystem>&& pSaveDataFileSystem
            ) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES(pSaveDataFileSystem);

                m_pSaveDataFileSystem = std::move(pSaveDataFileSystem);

                SaveDataFileSystemVisitor visitor;
                m_pSaveDataFileSystem->Accept(&visitor);
                NN_RESULT_DO(save::SaveDataInternalStorageFileSystem::Initialize(visitor.GetFileSystem()));

                NN_RESULT_SUCCESS;
            }

        private:
            std::shared_ptr<fssystem::SaveDataFileSystem> m_pSaveDataFileSystem;
        };
    }

    SaveDataFileSystemCreator::SaveDataFileSystemCreator(
        nn::MemoryResource* pAllocator,
        nn::fssystem::IBufferManager* pBufferManager,
        void(*pGenerateDeviceUniqueMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize),
        void(*pGenerateSeedUniqueMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize),
        void(*pGenerateRandom)(void* pHash, size_t sizeHash),
        uint32_t minimumVersion
    ) NN_NOEXCEPT
        : m_Allocator(pAllocator),
          m_pBufferManager(pBufferManager),
          m_DeviceUniqueMacGenerator(pGenerateDeviceUniqueMac),
          m_SeedUniqueMacGenerator(pGenerateSeedUniqueMac, &m_Seed),
          m_MinimumVersion(minimumVersion),
          m_GenerateRandom(pGenerateRandom)
    {
        memset(&m_Seed, 0, sizeof(m_Seed));
        nn::fssystem::save::JournalIntegritySaveDataFileSystem::SetGenerateRandomFunction(pGenerateRandom);
    }

    Result SaveDataFileSystemCreator::Format(
        fs::SubStorage baseStorage,
        size_t sizeBlock,
        int countExpandMax,
        const save::HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
        const save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
        uint32_t countDataBlock,
        uint32_t countReservedBlock,
        IBufferManager* pBufferManager,
        bool isDeviceUniqueMac,
        const nn::fs::SaveDataHashSalt& hashSalt
    ) NN_NOEXCEPT
    {
        return save::JournalIntegritySaveDataFileSystemDriver::Format(baseStorage, sizeBlock, countExpandMax, paramDuplex, paramIntegrity, countDataBlock, countReservedBlock, pBufferManager, GetMacGenerator(isDeviceUniqueMac), hashSalt);
    }

    Result SaveDataFileSystemCreator::FormatAsIntegritySaveData(
        fs::SubStorage baseStorage,
        size_t sizeBlock,
        const save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
        uint32_t countDataBlock,
        IBufferManager* pBufferManager,
        bool isDeviceUniqueMac
    ) NN_NOEXCEPT
    {
        return save::IntegritySaveDataFileSystemDriver::Format(baseStorage, sizeBlock, paramIntegrity, countDataBlock, pBufferManager, GetMacGenerator(isDeviceUniqueMac));
    }

    Result SaveDataFileSystemCreator::ExtractSaveDataParameters(
        fssystem::save::JournalIntegritySaveDataParameters* outParameters,
        fs::IStorage* pSaveDataStorage,
        bool isDeviceUniqueMac
    ) NN_NOEXCEPT
    {
        return fssystem::SaveDataFileSystem::ExtractParameters(
            outParameters,
            pSaveDataStorage,
            m_pBufferManager,
            GetMacGenerator(isDeviceUniqueMac),
            m_MinimumVersion);
    }

    Result SaveDataFileSystemCreator::ExtendSaveData(
        detail::SaveDataExtender* pExtender,
        fs::SubStorage saveDataStorage,
        fs::SubStorage logStorage,
        bool isDeviceUniqueMac
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pExtender);
        return pExtender->Extend(
            saveDataStorage,
            logStorage,
            m_pBufferManager,
            GetMacGenerator(isDeviceUniqueMac),
            m_MinimumVersion);
    }

    Result SaveDataFileSystemCreator::ReadExtraData(
        nn::fs::SaveDataExtraData* outData,
        fs::SubStorage baseStorage,
        IBufferManager* pBufferManager,
        bool isDeviceUniqueMac,
        bool isTemporaryFs
    ) NN_NOEXCEPT
    {
        if (isTemporaryFs)
        {
            return save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                       reinterpret_cast<fssystem::save::IntegritySaveDataFileSystem::ExtraData*>(outData),
                       baseStorage,
                       pBufferManager,
                       GetMacGenerator(isDeviceUniqueMac));
        }
        else
        {
            return save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                       reinterpret_cast<fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData*>(outData),
                       baseStorage,
                       pBufferManager,
                       GetMacGenerator(isDeviceUniqueMac),
                       m_MinimumVersion);
        }
    }

    void SaveDataFileSystemCreator::SetMacGenerationSeed(const void* pSeed, size_t seedSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(seedSize == MacGenerationSeed::Size);
        memcpy(m_Seed.value, pSeed, seedSize);
    }

    Result SaveDataFileSystemCreator::CreateRaw(std::shared_ptr<nn::fs::fsa::IFile>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fs::OpenMode mode) NN_NOEXCEPT
    {
        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        NN_RESULT_TRY(fileSystem->GetEntryType(&type, saveImageName))
            NN_RESULT_CATCH(ResultPathNotFound)
        {
            return ResultTargetNotFound();
        }
        NN_RESULT_END_TRY;

        if (type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
        {
            return ResultTargetNotFound();
        }
        else
        {
            std::unique_ptr<fs::fsa::IFile> file;
            NN_RESULT_DO(fileSystem->OpenFile(&file, saveImageName, mode));
            *outValue = std::move(file);
            NN_RESULT_SUCCESS;
        }
    }

    Result SaveDataFileSystemCreator::Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, fssystem::ISaveDataFileSystemCacheManager* pCacheManager, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isAllowedDirectorySaveData, bool isDeviceUniqueMac, nn::fs::SaveDataType saveDataType, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pCacheManager);

        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        NN_RESULT_TRY(fileSystem->GetEntryType(&type, saveImageName))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
                return ResultTargetNotFound();
            }
        NN_RESULT_END_TRY;

        if (type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
        {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            if( !isAllowedDirectorySaveData )
            {
                return ResultTargetNotFound();
            }
#else
            NN_UNUSED(isAllowedDirectorySaveData);
#endif
            std::unique_ptr<fssystem::SubdirectoryFileSystem> pBaseFs(new fssystem::SubdirectoryFileSystem(std::move(fileSystem), saveImageName));
            NN_RESULT_THROW_UNLESS(pBaseFs, nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCreatorA());

            std::shared_ptr<fssystem::DirectorySaveDataFileSystem> pFs = fssystem::AllocateShared<fssystem::DirectorySaveDataFileSystem>(std::move(pBaseFs), m_Allocator);
            NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCreatorB());
            NN_RESULT_DO(pFs->Initialize());

            *outValue = std::move(pFs);
        }
        else
        {
            std::shared_ptr<fs::IStorage> fileStorage;
            NN_RESULT_DO(OpenSaveDataStorage(&fileStorage, std::move(fileSystem), saveDataId, saveDataType, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write), fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Normal));

            if (saveDataType == nn::fs::SaveDataType::Temporary)
            {
                // TORIAEZU: TempFs はキャッシュしないが、Register のラップはする
                std::shared_ptr<fssystem::ApplicationTemporaryFileSystem> pFs = fssystem::AllocateShared<fssystem::ApplicationTemporaryFileSystem>();
                NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedAllocateShared());
                NN_RESULT_DO(pFs->Initialize(std::move(fileStorage), m_pBufferManager, GetMacGenerator(isDeviceUniqueMac)));
                *outExtraDataAccessor = pFs;

                auto pRegister = fssystem::AllocateShared<fssystem::ApplicationTemporaryFileSystemCacheRegister>(std::move(pFs), pCacheManager);
                NN_RESULT_THROW_UNLESS(pRegister.get(), nn::fs::ResultAllocationMemoryFailedAllocateShared());
                *outValue = pRegister;
            }
            else
            {
                std::shared_ptr<fssystem::SaveDataFileSystem> pFs = fssystem::AllocateShared<fssystem::SaveDataFileSystem>();
                NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCreatorD());
                NN_RESULT_DO(pFs->Initialize(std::move(fileStorage), m_pBufferManager, GetMacGenerator(isDeviceUniqueMac), timeStampGetter, m_GenerateRandom, m_MinimumVersion));
                *outExtraDataAccessor = pFs;

                auto pRegister = fssystem::AllocateShared<fssystem::SaveDataFileSystemCacheRegister>(std::move(pFs), pCacheManager);
                NN_RESULT_THROW_UNLESS(pRegister.get(), nn::fs::ResultAllocationMemoryFailedAllocateShared());
                *outValue = pRegister;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result SaveDataFileSystemCreator::CreateExtraDataAccessor(std::shared_ptr<fssystem::ISaveDataExtraDataAccessor>* outValue, std::shared_ptr<nn::fs::IStorage> storage, bool isDeviceUniqueMac, bool isTemporaryFs) NN_NOEXCEPT
    {
        if (isTemporaryFs)
        {
            std::shared_ptr<fssystem::ApplicationTemporaryFileSystem> pFs = fssystem::AllocateShared<fssystem::ApplicationTemporaryFileSystem>();
            NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCreatorE());
            NN_RESULT_DO(pFs->Initialize(std::move(storage), m_pBufferManager, GetMacGenerator(isDeviceUniqueMac)));
            *outValue = std::move(pFs);
        }
        else
        {
            std::shared_ptr<fssystem::SaveDataFileSystem> pFs = fssystem::AllocateShared<fssystem::SaveDataFileSystem>();
            NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCreatorE());
            NN_RESULT_DO(pFs->Initialize(std::move(storage), m_pBufferManager, GetMacGenerator(isDeviceUniqueMac), m_MinimumVersion));
            *outValue = std::move(pFs);
        }

        NN_RESULT_SUCCESS;
    }

    Result SaveDataFileSystemCreator::CreateInternalStorage(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, std::shared_ptr<nn::fssystem::SaveDataFileSystem>&& saveDataFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac, bool isTemporaryFs, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_SDK_REQUIRES_NOT_NULL(outExtraDataAccessor);
        NN_SDK_REQUIRES(fileSystem);

        std::shared_ptr<fssystem::SaveDataFileSystem> pFs;

        if( saveDataFileSystem )
        {
            pFs = std::move(saveDataFileSystem);
        }
        else
        {
            char saveImageName[1 + 16 + 1];
            util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

            DirectoryEntryType type;
            NN_RESULT_TRY(fileSystem->GetEntryType(&type, saveImageName))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    NN_RESULT_THROW(ResultTargetNotFound());
                }
            NN_RESULT_END_TRY;

            if( type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory )
            {
                NN_RESULT_THROW(ResultUnsupportedOperation());
            }
            else
            {
                std::shared_ptr<fs::IStorage> fileStorage;
                NN_RESULT_DO(OpenSaveDataStorage(&fileStorage, std::move(fileSystem), saveDataId, nn::fs::SaveDataType::Account, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write), fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Internal));

                if( isTemporaryFs )
                {
                    NN_RESULT_THROW(ResultUnsupportedOperation());
                }
                else
                {
                    pFs.reset(new fssystem::SaveDataFileSystem());
                    NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedNew());
                    NN_RESULT_DO(pFs->Initialize(std::move(fileStorage), m_pBufferManager, GetMacGenerator(isDeviceUniqueMac), timeStampGetter, m_GenerateRandom, m_MinimumVersion));
                }
            }
        }

        NN_SDK_ASSERT(pFs);
        *outExtraDataAccessor = pFs;

        std::shared_ptr<SaveDataInternalStorageFileSystem> pInternalStorage = fssystem::AllocateShared<SaveDataInternalStorageFileSystem>();
        NN_FSP_REQUIRES(pInternalStorage.get() != nullptr, nn::fs::ResultAllocationMemoryFailedAllocateShared());
        NN_RESULT_DO(pInternalStorage->Initialize(std::move(pFs)));

        *outValue = std::move(pInternalStorage);

        NN_RESULT_SUCCESS;
    }

    Result SaveDataFileSystemCreator::RecoverMasterHeader(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fssystem::IBufferManager* pBufferManager, bool isDeviceUniqueMac) NN_NOEXCEPT
    {
        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        NN_RESULT_TRY(fileSystem->GetEntryType(&type, saveImageName))
            NN_RESULT_CATCH(ResultPathNotFound)
        {
            return ResultTargetNotFound();
        }
        NN_RESULT_END_TRY;

        if( type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory )
        {
            // ディレクトリ版はマスターヘッダーなし
            NN_RESULT_SUCCESS;
        }
        else
        {
            std::shared_ptr<fs::FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<fs::FileStorageBasedFileSystem>();
            NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedAllocateShared());
            NN_RESULT_DO(fileStorage->Initialize(std::move(fileSystem), saveImageName, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

            int64_t size;
            NN_RESULT_DO(fileStorage->GetSize(&size));
            NN_RESULT_DO(nn::fssystem::save::JournalIntegritySaveDataFileSystem::RecoverMasterHeader(nn::fs::SubStorage(fileStorage.get(), 0, size), pBufferManager, GetMacGenerator(isDeviceUniqueMac), m_MinimumVersion));

            NN_RESULT_SUCCESS;
        }
    }

    Result SaveDataFileSystemCreator::UpdateMac(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac) NN_NOEXCEPT
    {
        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        DirectoryEntryType type;
        NN_RESULT_TRY(fileSystem->GetEntryType(&type, saveImageName))
            NN_RESULT_CATCH(ResultPathNotFound)
        {
            return ResultTargetNotFound();
        }
        NN_RESULT_END_TRY;

        if (type == nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
        {
            // ディレクトリ版は MAC なし
            NN_RESULT_SUCCESS;
        }
        else
        {
            std::shared_ptr<fs::FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<fs::FileStorageBasedFileSystem>();
            NN_RESULT_THROW_UNLESS(fileStorage, ResultAllocationMemoryFailedAllocateShared());
            NN_RESULT_DO(fileStorage->Initialize(std::move(fileSystem), saveImageName, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

            int64_t size;
            NN_RESULT_DO(fileStorage->GetSize(&size));
            NN_RESULT_DO(nn::fssystem::save::JournalIntegritySaveDataFileSystem::UpdateMac(nn::fs::SubStorage(fileStorage.get(), 0, size), GetMacGenerator(isDeviceUniqueMac), m_MinimumVersion));

            NN_RESULT_SUCCESS;
        }
    }

}}}
