﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssrv/fscreator/fssrv_MemoryStorageCreator.h>

using namespace nn::fssrv::detail;
using namespace nn::fssystem;
using namespace nn::fs;


namespace nn { namespace fssrv { namespace fscreator {

MemoryStorageCreator::MemoryStorageCreator() NN_NOEXCEPT
    : m_BufferArrayMutex(false)
{
}



Result MemoryStorageCreator::Create(std::shared_ptr<fs::IStorage>* outStorage, char** outRawBuffer, MemoryStorageId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_BufferArrayMutex);

    auto& buffer = m_BufferArray[static_cast<int>(id)];

    // TORIAEZU: 一度のみ確保可能
    NN_RESULT_THROW_UNLESS(!buffer.isAllocated,      fs::ResultAllocationMemoryFailed());
    NN_RESULT_THROW_UNLESS(buffer.buffer != nullptr, fs::ResultAllocationMemoryFailed());

    auto storage = fssystem::AllocateShared<fs::MemoryStorage>(buffer.buffer, buffer.bufferSize);
    NN_RESULT_THROW_UNLESS(storage != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    memset(buffer.buffer, 0x00, buffer.bufferSize);

    *outStorage = std::move(storage);
    *outRawBuffer = buffer.buffer;
    buffer.isAllocated = true;
    NN_RESULT_SUCCESS;
}


Result MemoryStorageCreator::RegisterMemory(MemoryStorageId id, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(id < MemoryStorageId::Count);
    NN_SDK_ASSERT(m_BufferArray[static_cast<int>(id)].buffer == nullptr);
    m_BufferArray[static_cast<int>(id)].buffer     = buffer;
    m_BufferArray[static_cast<int>(id)].bufferSize = bufferSize;
    NN_RESULT_SUCCESS;
}

MemoryStorageCreator::~MemoryStorageCreator() NN_NOEXCEPT
{
}

}}}
