﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Allocator.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fssrv/fscreator/fssrv_GameCardStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_GameCardFileSystemCreator.h>
#include <nn/gc/gc_Result.h>
#include "../detail/fssrv_GameCardManager.h"

namespace nn { namespace fssrv { namespace fscreator {


using namespace nn::fs;
using namespace nn::fssystem;
using namespace nn::fssrv::detail;

namespace {

    const char* GetPartitionPath(GameCardPartition partition)
    {
        switch(partition)
        {
        case GameCardPartition::Update:
            {
                return "update";
            }
        case GameCardPartition::Normal:
            {
                return "normal";
            }
        case GameCardPartition::Secure:
            {
                return "secure";
            }
        case GameCardPartition::Logo:
            {
                return "logo";
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

}

    class GameCardRootPartition : public nn::fs::detail::Newable
    {
    public:

        bool IsValid()
        {
            NN_SDK_ASSERT(m_pAlignRootStorage);

            return IsGameCardActivationValid(m_Handle);
        }

        GameCardRootPartition(nn::fs::GameCardHandle handle, std::shared_ptr<fs::IStorage> pBaseStorage, MemoryResource* pAllocator, IGameCardStorageCreator* pCreator, std::unique_ptr<Sha256PartitionFileSystemMeta>&& pMeta)
            : m_pCachedRootMeta(std::move(pMeta)),
            m_pAlignRootStorage(std::move(pBaseStorage)),
            m_Handle(handle),
            m_pAllocator(pAllocator),
            m_pHeaderBuffer(nullptr),
            m_pGameCardStorageCreator(pCreator),
            m_CachedLogoStorage(nullptr),
            m_CachedLogoStorageMutex(false)
        {
            m_HeaderSize = m_pCachedRootMeta->GetMetaDataSize();
        }

        ~GameCardRootPartition()
        {
            if( m_pHeaderBuffer != nullptr )
            {
                m_pAllocator->deallocate(m_pHeaderBuffer, m_HeaderSize);
            }
        }

        Result OpenPartition(std::shared_ptr<fs::IStorage>* pOutStorage, const Sha256PartitionFileSystemMeta::PartitionEntry** ppEntry, GameCardHandle handle, GameCardPartition partition)
        {
            switch(partition)
            {
            case GameCardPartition::Update:
            case GameCardPartition::Normal:
            case GameCardPartition::Secure:
            case GameCardPartition::Logo:
                break;
            default:
                return ResultInvalidArgument();
            }

            int index = m_pCachedRootMeta->GetEntryIndex(GetPartitionPath(partition));
            if( index < 0 )
            {
                return ResultPartitionNotFound();
            }

            auto pEntry = m_pCachedRootMeta->GetEntry(index);
            *ppEntry = pEntry;

            switch(partition)
            {
            case GameCardPartition::Update:
            case GameCardPartition::Normal:
                {
                    *pOutStorage = AllocateShared<SubStorage>(m_pAlignRootStorage, m_HeaderSize + pEntry->offset, pEntry->size);
                    NN_RESULT_THROW_UNLESS(*pOutStorage != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorA());
                    NN_RESULT_SUCCESS;
                }
            case GameCardPartition::Secure:
                {
                    // セキュアモードに移行するとロゴパーティションは読めなくなるので、ここで検証しキャッシュされていることを保証する
                    // 対策前のゲームカードにはロゴパーティションが存在しないため、ResultPartitionNotFound はスルーする
                    NN_RESULT_TRY(EnsureLogoDataCached())
                        NN_RESULT_CATCH(ResultPartitionNotFound)
                        {
                            // fall through
                        }
                    NN_RESULT_END_TRY;

                    std::shared_ptr<fs::IStorage> pSecureStorage;
                    NN_RESULT_DO(m_pGameCardStorageCreator->CreateSecureReadOnly(handle, &pSecureStorage));

                    auto pAlignStorage = AllocateShared<AlignmentMatchingStorageInBulkRead<1>>(std::move(pSecureStorage), 512);
                    NN_RESULT_THROW_UNLESS(pAlignStorage != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorB());

                    *pOutStorage = std::move(pAlignStorage);
                    NN_RESULT_SUCCESS;
                }
            case GameCardPartition::Logo:
                {
                    NN_RESULT_DO(EnsureLogoDataCached());
                    *pOutStorage = m_CachedLogoStorage;
                    NN_RESULT_SUCCESS;
                }
            default:
                return ResultPartitionNotFound();
            }
        }

        Result EnsureLogoDataCached()
        {
            int index = m_pCachedRootMeta->GetEntryIndex(GetPartitionPath(GameCardPartition::Logo));
            if( index < 0 )
            {
                return ResultPartitionNotFound();
            }

            std::lock_guard<os::Mutex> scopedLock(m_CachedLogoStorageMutex);
            if( !m_CachedLogoStorage )
            {
                auto pEntry = m_pCachedRootMeta->GetEntry(index);

                if( pEntry->size > LogoStorageSize )
                {
                    return ResultGameCardLogoDataTooLarge();
                }

                // ルートパーティションを SHA256PartitionFileSystem として読むことで検証する
                Sha256PartitionFileSystem rootPartitionFs;
                NN_RESULT_DO(rootPartitionFs.Initialize(m_pCachedRootMeta.get(), m_pAlignRootStorage));

                std::unique_ptr<fs::fsa::IFile> file;
                NN_RESULT_DO(rootPartitionFs.OpenFile(&file, "/logo", fs::OpenMode::OpenMode_Read));

                // ロゴパーティションはヘッダ・データの全域が検証対象となっているため、この Read のみで検証完了となる
                size_t readSize = 0;
                auto result = file->Read(&readSize, 0, m_LogoStorageDataCache, LogoStorageSize, fs::ReadOption::MakeValue(0));
                if( ResultDataCorrupted::Includes(result) )
                {
                    // 意図的にロゴの改竄を行った場合に専用のエラーを返すため変換
                    return ResultGameCardLogoDataCorrupted();
                }
                else if( result.IsFailure() )
                {
                    // その他のエラーはセキュアモードへの移行ができなければよいのでエラーをそのまま返す
                    return result;
                }

                if (static_cast<uint64_t>(readSize) != pEntry->size)
                {
                    return ResultGameCardLogoDataSizeInvalid();
                }

                m_CachedLogoStorage = AllocateShared<MemoryStorage>(m_LogoStorageDataCache, pEntry->size);
            }

            NN_RESULT_SUCCESS;
        }

    private:
        std::unique_ptr<Sha256PartitionFileSystemMeta> m_pCachedRootMeta;
        std::shared_ptr<fs::IStorage>                  m_pAlignRootStorage;
        nn::fs::GameCardHandle m_Handle;

        nn::MemoryResource* m_pAllocator;
        void* m_pHeaderBuffer;
        size_t m_HeaderSize;

        IGameCardStorageCreator* m_pGameCardStorageCreator;

        static const size_t LogoStorageSize = 72 * 1024; // Roundup(72704, 4 * 1024)
        char m_LogoStorageDataCache[LogoStorageSize];
        std::shared_ptr<fs::IStorage> m_CachedLogoStorage;
        os::Mutex m_CachedLogoStorageMutex;
    };

GameCardFileSystemCreator::GameCardFileSystemCreator(nn::MemoryResource* pAllocator, GameCardStorageCreator* pGameCardStorageCreator) NN_NOEXCEPT
    : m_pAllocator(pAllocator),
    m_pGameCardStorageCreator(pGameCardStorageCreator),
    m_CachedRootMutex(false)
{
}

GameCardFileSystemCreator::~GameCardFileSystemCreator() NN_NOEXCEPT
{
}

Result GameCardFileSystemCreator::Create(std::shared_ptr<fsa::IFileSystem>* outValue, GameCardHandle handle, GameCardPartition partition) NN_NOEXCEPT
{

    {
        std::lock_guard<os::Mutex> scopedLock(m_CachedRootMutex);

        if( m_CachedRoot == nullptr || !m_CachedRoot->IsValid() )
        {
            std::shared_ptr<fs::IStorage> rootStorage;
            NN_RESULT_DO(m_pGameCardStorageCreator->CreateReadOnly(handle, &rootStorage));

            auto pAlignRootStorage = AllocateShared<AlignmentMatchingStorageInBulkRead<1>>(std::move(rootStorage), 512);
            NN_RESULT_THROW_UNLESS(pAlignRootStorage != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorC());

            gc::GameCardStatus status;
            NN_RESULT_DO(GetGameCardStatus(&status, handle));



            int64_t updateAndNormalPartitionSize = status.normalAreaSize - status.partitionFsHeaderAddress;

            auto pRootFsStorage = AllocateShared<SubStorage>(std::move(pAlignRootStorage), status.partitionFsHeaderAddress, updateAndNormalPartitionSize);
            NN_RESULT_THROW_UNLESS(pRootFsStorage != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorD());

            std::unique_ptr<Sha256PartitionFileSystemMeta> rootPartitionFsMeta(new Sha256PartitionFileSystemMeta());
            NN_RESULT_THROW_UNLESS(rootPartitionFsMeta != nullptr, nn::fs::ResultAllocationMemoryFailedInGameCardFileSystemCreatorG());

            NN_RESULT_DO(rootPartitionFsMeta->Initialize(pRootFsStorage.get(), m_pAllocator, status.partitionFsHeaderHash, sizeof(status.partitionFsHeaderHash)));
            m_CachedRoot.reset(new GameCardRootPartition(handle, std::move(pRootFsStorage), m_pAllocator, m_pGameCardStorageCreator, std::move(rootPartitionFsMeta)));
            NN_RESULT_THROW_UNLESS(m_CachedRoot != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorE());
        }
    }

    {
        std::shared_ptr<fs::IStorage> pPartitionStorage;
        const Sha256PartitionFileSystemMeta::PartitionEntry* pEntry;
        NN_RESULT_DO(m_CachedRoot->OpenPartition(&pPartitionStorage, &pEntry, handle, partition));

        std::unique_ptr<Sha256PartitionFileSystemMeta> partitionFsMeta(new Sha256PartitionFileSystemMeta());
        NN_RESULT_THROW_UNLESS(partitionFsMeta != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorH());

        if (partition == GameCardPartition::Logo)
        {
            // ロゴパーティションの SHA256PartitionFsHeader は検証済み
            NN_RESULT_DO(partitionFsMeta->Initialize(pPartitionStorage.get(), m_pAllocator));
        }
        else
        {
            NN_RESULT_DO(partitionFsMeta->Initialize(pPartitionStorage.get(), m_pAllocator, pEntry->hash, sizeof(pEntry->hash)));
        }

        size_t headerSize;
        NN_RESULT_DO(Sha256PartitionFileSystemMeta::QueryMetaDataSize(&headerSize, pPartitionStorage.get()));

        auto pFs = AllocateShared<Sha256PartitionFileSystem>();
        NN_RESULT_THROW_UNLESS(pFs != nullptr, ResultAllocationMemoryFailedInGameCardFileSystemCreatorF());
        NN_RESULT_DO(pFs->Initialize(std::move(partitionFsMeta), std::move(pPartitionStorage)));

        *outValue = std::move(pFs);
    }

    NN_RESULT_SUCCESS;
}

}}}
