﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include "fssrv_Utility.h"
#include "fssrv_FatFileSystemCacheManager.h"
#include <nn/fssystem/fs_SubdirectoryFileSystem.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>

namespace nn { namespace fssrv { namespace detail {

FatDriveId GetFatDriveId(fs::BisPartitionId id)
{
    switch (id)
    {
        case fs::BisPartitionId::CalibrationFile:
            return FatDriveId_BisCalibration;
        case fs::BisPartitionId::SafeMode:
            return FatDriveId_BisSafeMode;
        case fs::BisPartitionId::System:
            return FatDriveId_BisSystem;
        case fs::BisPartitionId::User:
            return FatDriveId_BisUser;
        case fs::BisPartitionId::SystemProperPartition:
            return FatDriveId_BisSystemProperPartition;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result GetResultForInvalidFatFormat(fs::BisPartitionId id)
{
    switch (id)
    {
        case fs::BisPartitionId::CalibrationFile:
            return fs::ResultInvalidFatFormatBisCalibration();
        case fs::BisPartitionId::SafeMode:
            return fs::ResultInvalidFatFormatBisSafe();
        case fs::BisPartitionId::System:
            return fs::ResultInvalidFatFormatBisSystem();
        case fs::BisPartitionId::User:
            return fs::ResultInvalidFatFormatBisUser();
        case fs::BisPartitionId::SystemProperPartition:
            return fs::ResultInvalidFatFormatBisSafe();
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result GetResultForUsableSpaceNotEnough(fs::BisPartitionId id)
{
    switch (id)
    {
        case fs::BisPartitionId::CalibrationFile:
            return fs::ResultUsableSpaceNotEnoughMmcCalibration();
        case fs::BisPartitionId::SafeMode:
            return fs::ResultUsableSpaceNotEnoughMmcSafe();
        case fs::BisPartitionId::System:
            return fs::ResultUsableSpaceNotEnoughMmcSystem();
        case fs::BisPartitionId::User:
            return fs::ResultUsableSpaceNotEnoughMmcUser();
        case fs::BisPartitionId::SystemProperPartition:
            return fs::ResultUsableSpaceNotEnoughMmcSafe();
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

namespace {
    Result EnsureDirectoryImpl(nn::fs::fsa::IFileSystem* pFileSystem, char* path) NN_NOEXCEPT;
    Result EnsureParentDirectoryImpl(nn::fs::fsa::IFileSystem* pFileSystem, char* path) NN_NOEXCEPT;

    Result EnsureParentDirectoryImpl(nn::fs::fsa::IFileSystem* pFileSystem, char* path) NN_NOEXCEPT
    {
        int pathLength = static_cast<int>(strnlen(path, nn::fs::EntryNameLengthMax));
        NN_SDK_REQUIRES(path[pathLength - 1] != '/');

        int i = pathLength - 1;
        while (NN_STATIC_CONDITION(true))
        {
            if (i == 0)
            {
                NN_RESULT_SUCCESS;
            }

            if (path[i] == '/')
            {
                NN_RESULT_THROW_UNLESS(static_cast<size_t>(i) < (nn::fs::EntryNameLengthMax + 1)
                    , nn::fs::ResultTooLongPath());

                // 親ディレクトリを作成
                path[i] = '\0';
                NN_RESULT_DO(EnsureDirectoryImpl(pFileSystem, path));

                // パスを元に戻す
                path[i] = '/';

                break;
            }

            --i;
        }

        NN_RESULT_SUCCESS;
    }

    Result EnsureDirectoryImpl(nn::fs::fsa::IFileSystem* pFileSystem, char* path) NN_NOEXCEPT
    {
        int pathLength = static_cast<int>(strnlen(path, nn::fs::EntryNameLengthMax));
        NN_UNUSED(pathLength);
        // 前段で末尾の '/' は取り除かれているはず
        NN_SDK_REQUIRES(path[pathLength - 1] != '/');

        fs::DirectoryEntryType type;
        NN_RESULT_TRY(pFileSystem->GetEntryType(&type, path))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_DO(EnsureParentDirectoryImpl(pFileSystem, path));
            Result result = pFileSystem->CreateDirectory(path);
            if (result.IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }
            if (nn::fs::ResultPathAlreadyExists::Includes(result) == false)
            {
                return result;
            }
            NN_RESULT_DO(pFileSystem->GetEntryType(&type, path));
        }
        NN_RESULT_END_TRY;

        if (type != fs::DirectoryEntryType_Directory)
        {
            return fs::ResultPathAlreadyExists();
        }

        NN_RESULT_SUCCESS;
    }
}

Result EnsureParentDirectory(nn::fs::fsa::IFileSystem* pFileSystem, const char* path) NN_NOEXCEPT
{
    // パス操作用のバッファ確保
    char pathBuffer[nn::fs::EntryNameLengthMax + 1];
    int pathLength = static_cast<int>(strnlen(path, nn::fs::EntryNameLengthMax));
    memcpy(pathBuffer, path, pathLength);
    pathBuffer[pathLength] = '\0';

    // 末尾が '/' のパスは受け付けない
    NN_SDK_REQUIRES(path[pathLength - 1] != '/');

    return EnsureParentDirectoryImpl(pFileSystem, pathBuffer);
}

Result EnsureDirectory(nn::fs::fsa::IFileSystem* pFileSystem, const char* path) NN_NOEXCEPT
{
    // パス操作用のバッファ確保
    char pathBuffer[nn::fs::EntryNameLengthMax + 1];
    int pathLength = static_cast<int>(strnlen(path, nn::fs::EntryNameLengthMax));
    memcpy(pathBuffer, path, pathLength);
    pathBuffer[pathLength] = '\0';

    // 末尾の '/' は除去
    if (pathBuffer[pathLength - 1] == '/')
    {
        pathBuffer[pathLength - 1] = '\0';
    }

    return EnsureDirectoryImpl(pFileSystem, pathBuffer);
}

nn::Result CreateSubDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, const char* path) NN_NOEXCEPT
{
    return CreateSubDirectoryFileSystem(outValue, fileSystem, path, false);
}

nn::Result CreateSubDirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, const char* path, bool isUncPreserved) NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::fsa::IDirectory> dir;
    NN_RESULT_DO(fileSystem->OpenDirectory(&dir, path, nn::fs::OpenDirectoryMode_Directory));
    dir.reset(nullptr);

    std::shared_ptr<nn::fssystem::SubdirectoryFileSystem> pFs = nn::fssystem::AllocateShared<nn::fssystem::SubdirectoryFileSystem>(std::move(fileSystem), path, isUncPreserved);
    NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInSubDirectoryFileSystemCreatorA());
    *outValue = std::move(pFs);
    NN_RESULT_SUCCESS;
}

Result WrapSubDirectory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* rootDirectoryPath) NN_NOEXCEPT
{
    return WrapSubDirectory(pOutFileSystem, pBaseFileSystem, rootDirectoryPath, true);
}

Result WrapSubDirectory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* rootDirectoryPath, bool isDirectoryCreate) NN_NOEXCEPT
{
    if (!isDirectoryCreate)
    {
        // isDirectoryCreate が false の場合、ルートディレクトリが無かったら終了
        fs::DirectoryEntryType type;
        NN_RESULT_DO(pBaseFileSystem.get()->GetEntryType(&type, rootDirectoryPath));
    }
    NN_RESULT_DO(EnsureDirectory(pBaseFileSystem.get(), rootDirectoryPath));
    std::shared_ptr<nn::fs::fsa::IFileSystem> subDirFs;
    NN_RESULT_DO(CreateSubDirectoryFileSystem(&subDirFs, std::move(pBaseFileSystem), rootDirectoryPath));
    *pOutFileSystem = std::move(subDirFs);
    NN_RESULT_SUCCESS;
}

const char* GetExecutionDirectoryPath()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    static const fssystem::PathOnExecutionDirectory s_ExecutionDirectoryPath("");
    return s_ExecutionDirectoryPath.Get();
#else
    // TODO: 実機のデフォルトパス的なものを返す？
    NN_ABORT();
#endif
}


}}}
