﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fssystem/fs_Assert.h>

namespace nn { namespace fssrv { namespace detail {

    class UpdatePartitionPath
    {
    public:
        UpdatePartitionPath() NN_NOEXCEPT
            : m_Mutex(false)
        {
            memset(m_UpdatePartitionPath, 0x00, sizeof(m_UpdatePartitionPath));
        }

        Result Set(nn::Bit64 updaterProgramId, const char* path) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(path != nullptr, nn::fs::ResultNullptrArgument());

            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            if (path[0] == '\0')
            {
                m_UpdatePartitionPath[0] = '\0';
                NN_RESULT_SUCCESS;
            }

            auto length = strnlen(path, nn::fs::EntryNameLengthMax);
            NN_FSP_REQUIRES(length <= nn::fs::EntryNameLengthMax, nn::fs::ResultTooLongPath());
            NN_FSP_REQUIRES(length > 0, nn::fs::ResultInvalidPath());

            // UpdatePartition 内のパスは登録不可
            NN_FSP_REQUIRES(strncmp(path, const_cast<const char*>(nn::fs::detail::RegisteredUpdatePartitionMountName), strnlen(nn::fs::detail::RegisteredUpdatePartitionMountName, nn::fs::MountNameLengthMax)) != 0, nn::fs::ResultInvalidPath());

            strncpy(m_UpdatePartitionPath, path, sizeof(m_UpdatePartitionPath) - 1);
            m_UpdaterProgramId = updaterProgramId;
            NN_RESULT_SUCCESS;
        }

        const char* Get(nn::Bit64 *outValue) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            *outValue = m_UpdaterProgramId;
            return m_UpdatePartitionPath;
        }

    private:
        char m_UpdatePartitionPath[nn::fs::EntryNameLengthMax + 1];
        nn::Bit64 m_UpdaterProgramId;
        os::Mutex m_Mutex;
    };

}}}
