﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

namespace nn { namespace fs {
    class IStorage;
}}

namespace nn { namespace fssrv { namespace detail {

    typedef uint32_t SdCardHandle;
    const SdCardHandle InvalidSdCardHandle = 0;

    const size_t DataCacheLineSize = 64; // TODO: nn::dd などで定義されるべき

    inline bool IsBufferCapable(const void* buffer) NN_NOEXCEPT
    {
        return nn::fssystem::IsPooledBuffer(buffer)
               && nn::util::is_aligned(
                      reinterpret_cast<const uintptr_t>(buffer),
                      DataCacheLineSize
                  );
    }

    /**
        @brief pcv 経由のクロック制御に切り替えます。
     */
    void SwitchToPcvClockResetControl();

    /**
        @brief nand, sd 領域へのアクセサを取得します。// TODO: fscreator に分離
     */
    Result OpenSdStorage(fs::IStorage** outValue) NN_NOEXCEPT;
    Result OpenMmcStorage(fs::IStorage** outValue, nn::fs::MmcPartition id) NN_NOEXCEPT;

    /**
        @brief sd 領域へのアクセサを破棄します。
     */
    void CloseSdStorage() NN_NOEXCEPT;

    /**
        @brief SD カード向け API
     */
    bool IsSdCardRemovedAfterOpen() NN_NOEXCEPT;
    Result GetSdCardHandle(SdCardHandle* pOutValue) NN_NOEXCEPT;
    bool IsSdCardValid(SdCardHandle handle) NN_NOEXCEPT;
    void PutSdCardToSleep() NN_NOEXCEPT;
    void AwakenSdCard() NN_NOEXCEPT;
    void ShutdownSdCard() NN_NOEXCEPT;
    typedef void (*DeviceDetectionEventCallback)(void* pParameter);
    void RegisterSdCardDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT;
    void UnregisterSdCardDetectionEventCallback() NN_NOEXCEPT;
    bool IsSdCardInserted() NN_NOEXCEPT;
    Result GetSdCardSpeedMode(nn::fs::SdCardSpeedMode* pOutValue) NN_NOEXCEPT;
    Result GetSdCardCid(void* pOutBuffer, size_t size) NN_NOEXCEPT;
    Result GetSdCardUserAreaNumSectors(uint32_t* outValue) NN_NOEXCEPT;
    Result GetSdCardUserAreaSize(int64_t* outValue) NN_NOEXCEPT;
    Result GetSdCardProtectedAreaNumSectors(uint32_t* outValue) NN_NOEXCEPT;
    Result GetSdCardProtectedAreaSize(int64_t* outValue) NN_NOEXCEPT;
    Result GetAndClearSdCardErrorInfo(nn::fs::StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT;

    /**
        @brief MMC 向け API
     */
    void PutMmcToSleep() NN_NOEXCEPT;
    void AwakenMmc() NN_NOEXCEPT;
    void ShutdownMmc() NN_NOEXCEPT;
    Result GetMmcSpeedMode(nn::fs::MmcSpeedMode* pOutValue) NN_NOEXCEPT;
    Result GetMmcCid(void* pOutBuffer, size_t size) NN_NOEXCEPT;
    Result EraseMmc(nn::fs::MmcPartition id) NN_NOEXCEPT;
    Result GetMmcPartitionSize(int64_t* outValue, nn::fs::MmcPartition id) NN_NOEXCEPT;
    Result GetMmcPatrolCount(uint32_t* outValue) NN_NOEXCEPT;
    Result GetAndClearMmcErrorInfo(nn::fs::StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT;
    Result GetMmcExtendedCsd(void* pOutBuffer, size_t size) NN_NOEXCEPT;
    Result SuspendMmcPatrol() NN_NOEXCEPT;
    Result ResumeMmcPatrol() NN_NOEXCEPT;

    /**
        @brief MemoryReport 向け API
     */
    void GetAndClearPatrolReadAllocateBufferCount(uint64_t* outSuccessCount, uint64_t* outFailureCount) NN_NOEXCEPT;

    /**
        @brief SDMMC 制御向け API
     */
    Result GetSdmmcConnectionStatus(nn::fs::SdmmcSpeedMode* pOutSdmmcSpeedMode, nn::fs::SdmmcBusWidth* pOutSdmmcBusWidth, nn::fs::SdmmcPort sdmmcPort) NN_NOEXCEPT;
    void SuspendSdmmcControl() NN_NOEXCEPT;
    void ResumeSdmmcControl() NN_NOEXCEPT;

}}}

