﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>

#include "fssrv_AllocatorForServiceFrameWork.h"
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include "fssrv_SaveDataTransferVersion2.h"
#include "fssrv_SaveDataTransferStream.h"
#include "fssrv_SaveDataTransferPorterCore.h"
#include "fssrv_SaveDataTransferChunkPorter.h"
#include "fssrv_SaveDataTransferPorterManager.h"
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/fs_ConcatenationFile.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fssrv { namespace detail {

namespace {
    typedef nn::sf::ObjectFactory<AllocatorForServiceFrameWork::Policy> FileSystemFactory;

    class SaveDataExporterVersion2 : public Prohibitee
    {
    public:
        SaveDataExporterVersion2(
            const fs::SaveDataExtraData& extraData,
            const SaveDataTransferCryptoConfiguration& configuration,
            const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage,
            SaveDataPorterManager* pManager,
            bool diffImport
        ) NN_NOEXCEPT
            : Prohibitee(pManager)
            , m_Configuration(configuration)
            , m_KeySeedPackage(keySeedPackage)
            , m_ApplicationId(ncm::ApplicationId{extraData.attribute.programId.value})
            , m_IsDiffExport(diffImport)
        {
        }

        virtual ~SaveDataExporterVersion2() NN_OVERRIDE
        {
        }

        Result Initialize(std::shared_ptr<fs::IStorage> concatenatedStorage, const fs::SaveDataExtraData& extraData, std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs) NN_NOEXCEPT
        {
            NN_RESULT_DO(SaveDataPorterCore::CreateForExport(&m_Porter, std::move(concatenatedStorage), extraData, std::move(initialDataTheirs)));
            NN_RESULT_DO(Prohibitee::Initialize());
            NN_RESULT_SUCCESS;
        }

        Result SetDivisionCount(int count) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_DO(m_Porter->SetDivisionCount(count));
            NN_RESULT_SUCCESS;
        }

        Result OpenSaveDataDiffChunkIterator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkIterator>> outValue) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            return m_Porter->OpenSaveDataDiffChunkIterator(outValue, true);
        }

        Result OpenSaveDataChunkExporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkExporter>> outValue, uint32_t chunkId) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            // todo: IDの正当性チェック

            std::shared_ptr<fs::IStorage> storage;
            NN_RESULT_DO(m_Porter->OpenSaveDataChunkStorage(&storage, static_cast<uint16_t>(chunkId)));

            auto chunkExporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataChunkExporter, SaveDataChunkExporter>(m_Porter.get(), static_cast<fs::SaveDataChunkId>(chunkId), m_Configuration, m_KeySeedPackage);
            NN_RESULT_THROW_UNLESS(chunkExporter != nullptr, nn::fs::ResultAllocationMemoryFailed());
            NN_RESULT_DO(chunkExporter.GetImpl().Initialize(std::move(storage), chunkId != fs::SaveDataChunkIdForInitialData));
            outValue.Set(std::move(chunkExporter));

            NN_RESULT_SUCCESS;
        }

        void Invalidate() NN_NOEXCEPT NN_OVERRIDE
        {
            if (m_Porter != nullptr)
            {
                m_Porter->Invalidate();
            }
        }

        nn::ncm::ApplicationId GetApplicationId() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationId;
        }

        Result FinalizeFullExport(nn::sf::Out<fs::ISaveDataDivisionExporter::KeySeed> outKeySeed, nn::sf::Out<fs::ISaveDataDivisionExporter::InitialDataMac> outInitialDataMac) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(!m_IsDiffExport, fs::ResultPreconditionViolation());

            fs::ISaveDataDivisionExporter::KeySeed keySeed;
            memcpy(keySeed.data, m_KeySeedPackage.keySeed, keySeed.Size);

            InitialDataVersion2Detail::Content initialDataOurs;
            NN_RESULT_DO(m_Porter->FinalizeExport(&initialDataOurs));
            auto initialDataMac = CalculateInitialDataMac(initialDataOurs);

            outKeySeed.Set(keySeed);
            outInitialDataMac.Set(initialDataMac);

            Invalidate();
            m_IsInvalidated = true;
            NN_RESULT_SUCCESS;
        }

        Result FinalizeDiffExport(nn::sf::Out<fs::ISaveDataDivisionExporter::InitialDataMac> outInitialDataMac) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(m_IsDiffExport, fs::ResultPreconditionViolation());
            InitialDataVersion2Detail::Content initialDataOurs;
            NN_RESULT_DO(m_Porter->FinalizeExport(&initialDataOurs));
            auto initialDataMac = CalculateInitialDataMac(initialDataOurs);
            outInitialDataMac.Set(initialDataMac);

            Invalidate();
            m_IsInvalidated = true;
            NN_RESULT_SUCCESS;
        }

        Result CancelExport() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            Invalidate();
            m_IsInvalidated = true;
            NN_RESULT_SUCCESS;
        }

        Result GetImportInitialDataAad(nn::sf::Out<fs::ISaveDataDivisionExporter::InitialDataAad> outInitialDataAad) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            // 初期化時に与えられた InitialData の中から Aad を返す
            NN_RESULT_DO(m_Porter->GetInitialDataAadTheirs(outInitialDataAad.GetPointer()));
            NN_RESULT_SUCCESS;
        }

        Result SetExportInitialDataAad(const fs::ISaveDataDivisionExporter::InitialDataAad& aad) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            // Aad を設定する
            NN_RESULT_DO(m_Porter->SetInitialDataAad(aad));
            NN_RESULT_SUCCESS;
        }

    private:
        fs::ISaveDataDivisionExporter::InitialDataMac CalculateInitialDataMac(const InitialDataVersion2Detail::Content& initialData)
        {
            // TODO: 鍵を正式なものに変更
            char key[crypto::Aes128CmacGenerator::MacSize];
            m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken, nullptr, 0);

            fs::ISaveDataDivisionExporter::InitialDataMac mac;
            crypto::GenerateAes128Cmac(&mac.data, mac.Size, &initialData.singleSaveData, sizeof(initialData.singleSaveData), key, sizeof(key));
            return mac;
        }


    private:
        std::shared_ptr<SaveDataPorterCore> m_Porter;

        // TODO: 必要なものだけ渡す？
        const SaveDataTransferCryptoConfiguration& m_Configuration;
        const SaveDataTransferManagerVersion2::KeySeedPackage::Content& m_KeySeedPackage;
        nn::ncm::ApplicationId m_ApplicationId;
        bool m_IsInvalidated = false;

        const bool m_IsDiffExport;
    };

    // storage 間のコピーを行うヘルパ
    class StorageDuplicator
    {
    public:
        static const size_t BufferSize = 32 * 1024;

    public:
        StorageDuplicator(
            std::shared_ptr<fs::IStorage> srcStorage,
            std::shared_ptr<fs::IStorage> dstStorage
        )
            : m_SrcStorage(std::move(srcStorage))
            , m_DstStorage(std::move(dstStorage))
        {
        }

        Result Initialize()
        {
            int64_t srcSize;
            NN_RESULT_DO(m_SrcStorage->GetSize(&srcSize));

            int64_t dstSize;
            NN_RESULT_DO(m_DstStorage->GetSize(&dstSize));

            NN_RESULT_THROW_UNLESS(srcSize == dstSize, fs::ResultPreconditionViolation());

            m_RestSize = dstSize;
            NN_RESULT_SUCCESS;
        }

        Result ProcessDuplication(int64_t* pOutRestSize, int64_t size)
        {
            NN_SDK_ASSERT(m_RestSize >= 0);

            fssystem::PooledBuffer buffer;
            buffer.Allocate(BufferSize, BufferSize);

            int64_t restSizePerProcess = std::min(size, m_RestSize);

            while(restSizePerProcess > 0)
            {
                auto processSize = static_cast<size_t>(std::min(static_cast<int64_t>(buffer.GetSize()), std::min(size, restSizePerProcess)));

                NN_RESULT_DO(m_SrcStorage->Read(m_Offset, buffer.GetBuffer(), processSize));
                NN_RESULT_DO(m_DstStorage->Write(m_Offset, buffer.GetBuffer(), processSize));

                m_Offset           += processSize;
                restSizePerProcess -= processSize;
                m_RestSize         -= processSize;
            }

            *pOutRestSize = m_RestSize;
            NN_RESULT_SUCCESS;
        }

    private:
        std::shared_ptr<fs::IStorage> m_SrcStorage;
        std::shared_ptr<fs::IStorage> m_DstStorage;

        int64_t m_Offset = 0;
        int64_t m_RestSize = -1;
    };


    // pFs を保持するヘルパ
    // pFileArray の破棄後に pFs を破棄する
    template <int N>
    class ConcatenationFileWithParentFs : public fssystem::ConcatenationFile<N>
    {
    public:
        ConcatenationFileWithParentFs(std::array<std::shared_ptr<fs::fsa::IFile>, N> pFileArray, std::shared_ptr<fs::fsa::IFileSystem> pFs)
            : fssystem::ConcatenationFile<N>(std::move(pFileArray))
            , m_Fs(std::move(pFs))
        {}

        ~ConcatenationFileWithParentFs()
        {
            fssystem::ConcatenationFile<N>::m_pFileArray.fill(nullptr);
            m_Fs.reset();
        }

    private:
        std::shared_ptr<fs::fsa::IFileSystem> m_Fs;
    };


    // Thumbnail, AllocationTableControlArea, AllocationTableMeta, AllocationTableDataWithZeroFree を連結した IStorage を返す
    Result OpenConcatenatedSaveDataStorage(std::shared_ptr<fs::IStorage>* outValueStorage, std::shared_ptr<fs::fsa::IFileSystem>* outValueFileSystem, ISaveDataTransferCoreInterface* pTransferCore, fs::SaveDataSpaceId spaceId, fs::SaveDataId saveDataId)
    {
        std::shared_ptr<fs::IStorage> storage;

        std::shared_ptr<fs::fsa::IFileSystem> saveDataInternalFs;
        NN_RESULT_DO(pTransferCore->OpenSaveDataInternalStorageFileSystemCore(&saveDataInternalFs, spaceId, saveDataId));

        static const char* const Paths[] =
        {
            "/AllocationTableControlArea",
            "/AllocationTableMeta",
            "/AllocationTableDataWithZeroFree",
        };

        std::shared_ptr<fs::fsa::IFile> files[3];
        // TORIAEZU: 単純連結
        for (int i = 0; i < 3; i++)
        {
            std::unique_ptr<fs::fsa::IFile> file;
            NN_RESULT_DO(saveDataInternalFs->OpenFile(&file, Paths[i], static_cast<fs::OpenMode>(fs::OpenMode_Write | fs::OpenMode_Read)));
            files[i] = std::move(file);
        }

        // TODO: ユーザアカウント毎セーブであること（サムネ持ちセーブであること）の確認
        // Thumbnail
        std::shared_ptr<fs::fsa::IFile> thumbnailFile;
        NN_RESULT_DO(pTransferCore->OpenSaveDataMetaFileRaw(&thumbnailFile, spaceId, saveDataId, fs::SaveDataMetaType::Thumbnail, static_cast<fs::OpenMode>(fs::OpenMode_Write | fs::OpenMode_Read)));

        std::array<std::shared_ptr<fs::fsa::IFile>, 4> fileArray{
            {
                thumbnailFile,
                files[0],
                files[1],
                files[2],
            }
        };

        auto concatenatedFile = fssystem::AllocateShared<ConcatenationFileWithParentFs<4>>(std::move(fileArray), saveDataInternalFs);
        NN_RESULT_THROW_UNLESS(concatenatedFile != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

        storage = fssystem::AllocateShared<fs::FileStorage>(std::move(concatenatedFile));
        NN_RESULT_THROW_UNLESS(storage != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

        *outValueStorage = std::move(storage);
        if (outValueFileSystem != nullptr)
        {
            *outValueFileSystem = std::move(saveDataInternalFs);
        }
        NN_RESULT_SUCCESS;
    }

    Result OpenConcatenatedSaveDataStorage(std::shared_ptr<fs::IStorage>* outValueStorage, ISaveDataTransferCoreInterface* pTransferCore, fs::SaveDataSpaceId spaceId, fs::SaveDataId saveDataId)
    {
        return OpenConcatenatedSaveDataStorage(outValueStorage, nullptr, pTransferCore, spaceId, saveDataId);
    }


    class SaveDataImporterVersion2 : public Prohibitee
    {
    private:
        enum class Mode{
            Full,
            FullSwap,
            Diff,
            DiffDuplicateSwap,
        };

        bool IsDiffMode()
        {
            return m_Mode == Mode::Diff || m_Mode == Mode::DiffDuplicateSwap;
        }
        bool IsSwapMode()
        {
            return m_Mode == Mode::FullSwap || m_Mode == Mode::DiffDuplicateSwap;
        }

    public:

        // 新規インポート用
        SaveDataImporterVersion2(
            ISaveDataTransferCoreInterface* pSaveDataTransferCoreInterface,
            std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs,
            const SaveDataTransferCryptoConfiguration& configuration,
            const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage,
            fs::SaveDataSpaceId spaceId,
            fs::UserId userId,
            SaveDataPorterManager* pManager
        ) NN_NOEXCEPT
            : Prohibitee(pManager)
            , m_pTransferCore(pSaveDataTransferCoreInterface)
            , m_InitialDataTheirs(std::move(initialDataTheirs))
            , m_Configuration(configuration)
            , m_KeySeedPackage(keySeedPackage)
            , m_SpaceId(spaceId)
            , m_UserId(userId)
            , m_SrcSaveDataId(0xFFFFFFFFFFFFFFFFULL) // unused
            , m_Mode(Mode::Full)
        {
            m_DivisionCount = m_InitialDataTheirs->singleSaveData.divisionCount;
            m_ApplicationId = ncm::ApplicationId{ m_InitialDataTheirs->singleSaveData.extraData.attribute.programId.value };
        }

        // 差分インポート用
        SaveDataImporterVersion2(
            ISaveDataTransferCoreInterface* pSaveDataTransferCoreInterface,
            std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs,
            const SaveDataTransferCryptoConfiguration& configuration,
            const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage,
            fs::SaveDataSpaceId spaceId,
            fs::SaveDataId saveDataId,
            SaveDataPorterManager* pManager
        ) NN_NOEXCEPT
            : Prohibitee(pManager)
            , m_pTransferCore(pSaveDataTransferCoreInterface)
            , m_InitialDataTheirs(std::move(initialDataTheirs))
            , m_Configuration(configuration)
            , m_KeySeedPackage(keySeedPackage)
            , m_SpaceId(spaceId)
            , m_SrcSaveDataId(saveDataId)
            , m_DstSaveDataId(saveDataId)
            , m_Mode(Mode::Diff)
        {
            m_DivisionCount = m_InitialDataTheirs->singleSaveData.divisionCount;
            m_ApplicationId = ncm::ApplicationId{ m_InitialDataTheirs->singleSaveData.extraData.attribute.programId.value };
        }

        // 差分複製インポート用（仮）
        SaveDataImporterVersion2(
            ISaveDataTransferCoreInterface* pSaveDataTransferCoreInterface,
            std::unique_ptr<InitialDataVersion2Detail::Content>&& initialDataTheirs,
            const SaveDataTransferCryptoConfiguration& configuration,
            const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage,
            fs::SaveDataSpaceId spaceId,
            fs::SaveDataId saveDataId,
            SaveDataPorterManager* pManager,
            bool duplicate
        ) NN_NOEXCEPT
            : Prohibitee(pManager)
            , m_pTransferCore(pSaveDataTransferCoreInterface)
            , m_InitialDataTheirs(std::move(initialDataTheirs))
            , m_Configuration(configuration)
            , m_KeySeedPackage(keySeedPackage)
            , m_SpaceId(spaceId)
            , m_SrcSaveDataId(saveDataId)
            , m_Mode(Mode::DiffDuplicateSwap)
        {
            NN_UNUSED(duplicate);
            m_DivisionCount = m_InitialDataTheirs->singleSaveData.divisionCount;
            m_ApplicationId = ncm::ApplicationId{ m_InitialDataTheirs->singleSaveData.extraData.attribute.programId.value };
        }

        Result Initialize() NN_NOEXCEPT
        {
            NN_RESULT_DO(Prohibitee::Initialize());

            if (!IsDiffMode())
            {
                NN_RESULT_SUCCESS;
            }

            // TORIAEZU: 連結サイズのみ比較
            // TODO: 各ファイルのサイズをそれぞれ比較
            std::shared_ptr<fs::IStorage> concatenatedStorage;
            std::shared_ptr<fs::fsa::IFileSystem> fs;
            NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&concatenatedStorage, &fs, m_pTransferCore, m_SpaceId, m_SrcSaveDataId));

            int64_t size;
            NN_RESULT_DO(concatenatedStorage->GetSize(&size));

            if (size == m_InitialDataTheirs->singleSaveData.internalStorageSize)
            {
                NN_RESULT_DO(SaveDataPorterCore::CreateForImport(&m_Porter, std::move(concatenatedStorage), std::move(fs), std::move(m_InitialDataTheirs)));
            }
            else
            {
                // サイズが変化している場合は InitializeImport() 後に m_Porter をセットアップする
                // この時点ではローカル側セーブを開く必要が無い

                if (m_Mode == Mode::DiffDuplicateSwap)
                {
                    // 複製するメリットが現状無いので FullSwap に移行する
                    m_Mode = Mode::FullSwap;
                }
            }

            // TODO: saveDataId 指定でなく userId 指定とする
            fs::SaveDataExtraData extraDataOurs;
            NN_RESULT_DO(m_pTransferCore->ReadSaveDataFileSystemExtraDataCore(&extraDataOurs, m_SpaceId, m_SrcSaveDataId, false));
            m_UserId = extraDataOurs.attribute.userId;

            NN_RESULT_SUCCESS;
        }

        ~SaveDataImporterVersion2() NN_NOEXCEPT {}

        Result InitializeImport(nn::sf::Out<int64_t> outRestSize, int64_t processSize) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(m_InitialDataTheirs != nullptr, fs::ResultPreconditionViolation());

            const auto& extraDataTheirs = m_InitialDataTheirs->singleSaveData.extraData;

            if (IsDiffMode())
            {

                // TODO: 判定条件の見直し
                if (m_Mode == Mode::Diff && m_Porter == nullptr)
                {
                    fs::SaveDataExtraData extraDataOurs;
                    NN_RESULT_DO(m_pTransferCore->ReadSaveDataFileSystemExtraDataCore(&extraDataOurs, m_SpaceId, m_SrcSaveDataId, false));

                    if (extraDataTheirs.availableSize < extraDataOurs.availableSize ||
                        extraDataTheirs.journalSize   < extraDataOurs.journalSize)
                    {
                        // いずれかでも縮小するなら削除再作成
                        // TODO: saveDataId を引き継いで再作成する検討
                        NN_RESULT_DO(m_pTransferCore->DeleteSaveDataFileSystemBySaveDataSpaceId(static_cast<uint8_t>(m_SpaceId), m_SrcSaveDataId));
                        NN_RESULT_DO(CreateDstSaveData(false));

                        // TODO: 空き容量不足時にサイズを返す
                    }
                    else if (extraDataTheirs.availableSize > extraDataOurs.availableSize ||
                             extraDataTheirs.journalSize   > extraDataOurs.journalSize)
                    {
                        // 縮小が無く、拡張がある場合は拡張
                        NN_RESULT_DO(m_pTransferCore->ExtendSaveDataFileSystem(static_cast<uint8_t>(m_SpaceId), m_SrcSaveDataId, extraDataTheirs.availableSize, extraDataTheirs.journalSize));

                        // TODO: 空き容量不足時にサイズを返す
                    }
                }
                else if (m_Mode == Mode::DiffDuplicateSwap)
                {
                    if (m_Duplicator == util::nullopt)
                    {
                        // 複製先作成
                        NN_RESULT_DO(CreateDstSaveData(true));

                        // TODO: 空き容量不足時にサイズを返す

                        std::shared_ptr<fs::IStorage> srcFileStorage;
                        NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&srcFileStorage, m_pTransferCore, m_SpaceId, m_SrcSaveDataId));

                        std::shared_ptr<fs::IStorage> dstFileStorage;
                        NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&dstFileStorage, m_pTransferCore, m_SpaceId, m_DstSaveDataId));

                        m_Duplicator.emplace(std::move(srcFileStorage), std::move(dstFileStorage));
                        NN_RESULT_DO(m_Duplicator->Initialize());
                    }

                    // 複製処理を進める
                    {
                        int64_t restSize;
                        NN_RESULT_DO(m_Duplicator->ProcessDuplication(&restSize, processSize));

                        if (restSize == 0)
                        {
                            // 完了
                            m_Duplicator = util::nullopt;
                            // fall through
                        }
                        else
                        {
                            // 未完了
                            outRestSize.Set(restSize);
                            m_IsImporting = true;
                            NN_RESULT_SUCCESS;
                        }

                    }

                }
            }
            else
            {
                NN_RESULT_DO(CreateDstSaveData(IsSwapMode()));
            }

            NN_RESULT_DO(m_pTransferCore->WriteSaveDataFileSystemExtraDataCore(m_SpaceId, m_DstSaveDataId, extraDataTheirs, extraDataTheirs.attribute.type, false));

            // PorterCore 作成
            std::shared_ptr<fs::IStorage> concatenatedStorage;
            std::shared_ptr<fs::fsa::IFileSystem> fs;
            NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&concatenatedStorage, &fs, m_pTransferCore, m_SpaceId, m_DstSaveDataId));

            // TODO: Mode::Diff の場合は作り直す必要無し
            NN_RESULT_DO(SaveDataPorterCore::CreateForImport(&m_Porter, std::move(concatenatedStorage), std::move(fs), std::move(m_InitialDataTheirs)));

            outRestSize.Set(0);
            m_IsImporting = true;
            NN_RESULT_SUCCESS;
        }

        Result FinalizeImport() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(m_IsImporting, fs::ResultPreconditionViolation());
            NN_RESULT_DO(m_Porter->FinalizeImport());
            NN_RESULT_DO(m_pTransferCore->FinalizeSaveDataCreation(m_DstSaveDataId, m_SpaceId));

            if (IsSwapMode())
            {
                // TODO: MAC 有効化と swap をアトミック化
                NN_RESULT_DO(m_pTransferCore->SwapSaveDataKeyAndState(static_cast<uint8_t>(m_SpaceId), m_SrcSaveDataId, m_DstSaveDataId));
                NN_RESULT_DO(m_pTransferCore->DeleteSaveDataFileSystemBySaveDataSpaceId(static_cast<uint8_t>(m_SpaceId), m_SrcSaveDataId));
            }

            m_Porter.reset();

            m_IsImporting = false;
            m_IsInvalidated = true;
            NN_RESULT_SUCCESS;
        }

        Result CancelImport() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(m_IsImporting, fs::ResultPreconditionViolation());
            Invalidate();

            if (IsSwapMode() || m_Mode == Mode::Full)
            {
                // セーブデータの作成をキャンセルする
                NN_RESULT_DO(m_pTransferCore->CancelSaveDataCreation(m_DstSaveDataId, m_SpaceId));
            }
            else
            {
                NN_RESULT_THROW(nn::fs::ResultNotImplemented());
            }

            m_IsImporting = false;
            m_IsInvalidated = true;
            NN_RESULT_SUCCESS;
        }

        Result OpenSaveDataDiffChunkIterator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkIterator>> outValue) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            if (m_Porter == nullptr) // porter 未作成＝対象セーブデータ未セットアップ
            {
                return SaveDataPorterCore::OpenSaveDataDiffChunkIteratorAll(outValue, false, m_DivisionCount);
            }
            else
            {
                return m_Porter->OpenSaveDataDiffChunkIterator(outValue, false);
            }
        }

        Result OpenSaveDataChunkImporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataChunkImporter>> outValue, uint32_t chunkId) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(m_IsImporting, fs::ResultPreconditionViolation());
            NN_RESULT_THROW_UNLESS(chunkId != fs::SaveDataChunkIdForInitialData, fs::ResultInvalidArgument());
            NN_RESULT_THROW_UNLESS(m_Porter != nullptr, fs::ResultPreconditionViolation());

            std::shared_ptr<fs::IStorage> storage;
            NN_RESULT_DO(m_Porter->OpenSaveDataChunkStorage(&storage, static_cast<uint16_t>(chunkId)));

            auto chunkImporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataChunkImporter, SaveDataChunkImporter>(m_Configuration, m_KeySeedPackage);
            NN_RESULT_THROW_UNLESS(chunkImporter != nullptr, nn::fs::ResultAllocationMemoryFailed());
            NN_RESULT_DO(chunkImporter.GetImpl().Initialize(std::move(storage), m_Porter->GetSize(static_cast<uint16_t>(chunkId))));
            outValue.Set(std::move(chunkImporter));

            NN_RESULT_SUCCESS;
        }

        virtual Result GetImportInitialDataAad(nn::sf::Out<fs::ISaveDataDivisionImporter::InitialDataAad> outInitialDataAad) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(!m_IsInvalidated, fs::ResultPreconditionViolation());
            // 初期化時に与えられた InitialData の中から Aad を返す
            if (m_Porter == nullptr)
            {
                // porter 未作成時は m_InitialDataTheirs の中から返す
                memcpy(outInitialDataAad.GetPointer()->data, m_InitialDataTheirs->singleSaveData.aad.data, fs::detail::InitialDataAad::Size);
            }
            else
            {
                NN_RESULT_DO(m_Porter->GetInitialDataAadTheirs(outInitialDataAad.GetPointer()));
            }
            NN_RESULT_SUCCESS;
        }

        void Invalidate() NN_NOEXCEPT NN_OVERRIDE
        {
            if (m_Porter != nullptr)
            {
                m_Porter->Invalidate();
            }
            m_Duplicator = util::nullopt;
        }

        nn::ncm::ApplicationId GetApplicationId() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_ApplicationId;
        }

    private:
        Result CreateDstSaveData(bool asSecondary)
        {
            NN_SDK_ASSERT(m_InitialDataTheirs != nullptr);

            const auto& extraDataTheirs = m_InitialDataTheirs->singleSaveData.extraData;

            fs::SaveDataAttribute attribute = extraDataTheirs.attribute;
            {
                attribute.userId = m_UserId; // userId 付け替え
                if (asSecondary)
                {
                    attribute.rank = fs::SaveDataRank::Secondary; // Finalize 時に swap する
                }
            }

            fs::SaveDataCreationInfo creationInfo;
            {
                creationInfo.size = extraDataTheirs.availableSize;
                creationInfo.journalSize = extraDataTheirs.journalSize;
                creationInfo.blockSize = fs::detail::DefaultSaveDataBlockSize;
                creationInfo.ownerId = extraDataTheirs.ownerId;
                creationInfo.flags = extraDataTheirs.flags;
                creationInfo.spaceId = m_SpaceId;
                creationInfo.isPseudoSaveFs = false;
            }

            fs::SaveDataMetaInfo metaInfo;
            if (extraDataTheirs.attribute.type == fs::SaveDataType::Account)
            {
                metaInfo.type = fs::SaveDataMetaType::Thumbnail;
                metaInfo.size = fs::detail::ThumbnailFileSize;
            }
            else
            {
                metaInfo.type = fs::SaveDataMetaType::None;
                metaInfo.size = 0;
            }

            nn::fs::SaveDataHashSalt hashSalt;
            NN_RESULT_DO(m_pTransferCore->CreateSaveDataFileSystemCore(attribute, creationInfo, metaInfo, hashSalt, true));


            fs::SaveDataInfo info;
            NN_RESULT_DO(m_pTransferCore->GetSaveDataInfo(&info, m_SpaceId, attribute));
            m_DstSaveDataId = info.saveDataId;

            NN_RESULT_SUCCESS;
        }

    private:
        ISaveDataTransferCoreInterface* const m_pTransferCore;
        std::shared_ptr<SaveDataPorterCore> m_Porter;
        std::unique_ptr<InitialDataVersion2Detail::Content> m_InitialDataTheirs;

        // TODO: 必要なものだけ渡す？
        const SaveDataTransferCryptoConfiguration& m_Configuration;
        const SaveDataTransferManagerVersion2::KeySeedPackage::Content& m_KeySeedPackage;

        const fs::SaveDataSpaceId m_SpaceId;
        fs::UserId m_UserId;
        const fs::SaveDataId m_SrcSaveDataId;
        fs::SaveDataId m_DstSaveDataId;
        int m_DivisionCount = 0; // TODO: porter との管理重複
        nn::ncm::ApplicationId m_ApplicationId;
        bool m_IsImporting = false;
        bool m_IsInvalidated = false;

        Mode m_Mode;

        util::optional<StorageDuplicator> m_Duplicator;
    };

} // namespace

SaveDataTransferManagerVersion2::SaveDataTransferManagerVersion2(const SaveDataTransferCryptoConfiguration& configuration, ISaveDataTransferCoreInterface* pImpl, SaveDataPorterManager* pPorterManager) NN_NOEXCEPT
    : m_pImpl(pImpl, true)
    , m_Configuration(configuration)
    , m_pPorterManager(pPorterManager)
{
    m_Configuration.pGenerateRandom(m_Challenge.data, Challenge::Size);
}


Result SaveDataTransferManagerVersion2::GetChallenge(const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(outBuffer.GetSize() >= sizeof(Challenge), fs::ResultInvalidSize());
    auto pChallenge = reinterpret_cast<Challenge*>(outBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pChallenge != nullptr, fs::ResultNullptrArgument());

    memcpy(pChallenge->data, m_Challenge.data, Challenge::Size);
    NN_RESULT_SUCCESS;
}

Result SaveDataTransferManagerVersion2::SetKeySeedPackage(const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(!m_IsKspSet, fs::ResultPreconditionViolation());
    NN_FSP_REQUIRES(inBuffer.GetSize() >= sizeof(KeySeedPackage), fs::ResultInvalidSize());

    bool isVerificationFailure = false;

    // TORIAEZU: ksp 検証に失敗しても続行可能とする
    // TODO: remove
    {
        m_IsKspSet = true;
        memset(&m_KeySeedPackage, 0, sizeof(m_KeySeedPackage));
    }

    auto pKsp = reinterpret_cast<const KeySeedPackage*>(inBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pKsp != nullptr, fs::ResultNullptrArgument());

    KeySeedPackage ksp;
    memcpy(&ksp, pKsp, sizeof(KeySeedPackage));

    char mac[crypto::Aes128GcmDecryptor::MacSize];
    char key[crypto::Aes128GcmDecryptor::BlockSize];

    m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken, nullptr, 0);

    // AES-GCM 検証
    auto decryptedSize = crypto::DecryptAes128Gcm(
        &ksp.encryptedArea, sizeof(ksp.encryptedArea),
        &mac, sizeof(mac),
        key, sizeof(key),
        ksp.kspIv, sizeof(ksp.kspIv),
        &ksp.encryptedArea, sizeof(ksp.encryptedArea),
        nullptr, 0
    );
    NN_RESULT_THROW_UNLESS(decryptedSize == sizeof(ksp.encryptedArea), fs::ResultUnknown());
    //NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, ksp.kspMac, crypto::Aes128GcmDecryptor::MacSize), fs::ResultSaveDataTransferTokenMacVerificationFailed());
    if (!crypto::IsSameBytes(mac, ksp.kspMac, crypto::Aes128GcmDecryptor::MacSize))
    {
        NN_SDK_LOG("fs::ResultSaveDataTransferTokenMacVerificationFailed() (ignored)\n");
        isVerificationFailure = true;
    }

    // サーバ署名検証
    // TORIAEZU: セーブデータ移行と同じ鍵を流用
    // TODO: クラウドバックアップ用の鍵に差し替え
    const char KspSignKeyPublicExponent[] = { 0x01, 0x00, 0x01 };
    /*
    NN_RESULT_THROW_UNLESS(
        crypto::VerifyRsa2048PssSha256(
            ksp.encryptedArea.sign, sizeof(ksp.encryptedArea.sign),
            m_Configuration.kspSignKeyModulus, sizeof(m_Configuration.tokenSignKeyModulus),
            KspSignKeyPublicExponent, sizeof(KspSignKeyPublicExponent),
            &ksp.encryptedArea.signedArea, sizeof(ksp.encryptedArea.signedArea)
        ),
        fs::ResultSaveDataTransferTokenSignatureVerificationFailed()
    );*/
    if (!crypto::VerifyRsa2048PssSha256(
        ksp.encryptedArea.sign, sizeof(ksp.encryptedArea.sign),
        m_Configuration.kspSignKeyModulus, sizeof(m_Configuration.kspSignKeyModulus),
        KspSignKeyPublicExponent, sizeof(KspSignKeyPublicExponent),
        &ksp.encryptedArea.signedArea, sizeof(ksp.encryptedArea.signedArea)
    ))
    {
        NN_SDK_LOG("fs::ResultSaveDataTransferTokenSignatureVerificationFailed() (ignored)\n");
        isVerificationFailure = true;
    }


    // challenge 検証
    //NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(m_Challenge.data, ksp.encryptedArea.signedArea.challenge, sizeof(m_Challenge.data)), fs::ResultSaveDataTransferTokenChallengeVerificationFailed());
    if (!crypto::IsSameBytes(m_Challenge.data, ksp.encryptedArea.signedArea.challenge, sizeof(m_Challenge.data)))
    {
        NN_SDK_LOG("fs::ResultSaveDataTransferTokenChallengeVerificationFailed() (ignored)\n");
        isVerificationFailure = true;
    }

    m_KeySeedPackage = ksp.encryptedArea.signedArea;
    m_IsKspSet = true;

    if (isVerificationFailure)
    {
        memset(m_KeySeedPackage.keySeed, 0x0, sizeof(m_KeySeedPackage.keySeed));
    }

    NN_RESULT_SUCCESS;
}

Result SaveDataTransferManagerVersion2::OpenSaveDataExporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataDivisionExporter>> outValue, uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT
{
    std::shared_ptr<fs::IStorage> concatenatedStorage;
    NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&concatenatedStorage, m_pImpl.Get(), static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId));

    fs::SaveDataExtraData extraData;
    NN_RESULT_DO(m_pImpl->ReadSaveDataFileSystemExtraDataCore(&extraData, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, false));

    // TODO: keyseed を生成
    // TORIAEZU: keySeed = 0
    memset(&m_KeySeedPackage, 0, sizeof(m_KeySeedPackage));

    auto exporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataDivisionExporter, SaveDataExporterVersion2>(extraData, m_Configuration, m_KeySeedPackage, m_pPorterManager, false);
    NN_RESULT_THROW_UNLESS(exporter != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(exporter.GetImpl().Initialize(std::move(concatenatedStorage), extraData, nullptr));
    outValue.Set(std::move(exporter));

    NN_RESULT_SUCCESS;
}

Result SaveDataTransferManagerVersion2::OpenSaveDataExporterForDiffExport(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataDivisionExporter>> outValue, const nn::sf::InBuffer& initialDataBuffer, uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(initialDataBuffer.GetSize() >= sizeof(InitialDataVersion2Detail), fs::ResultInvalidSize());
    auto pInitialData = reinterpret_cast<const InitialDataVersion2Detail*>(initialDataBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pInitialData != nullptr, fs::ResultNullptrArgument());

    NN_RESULT_THROW_UNLESS(m_IsKspSet, fs::ResultPreconditionViolation());

    std::unique_ptr<InitialDataVersion2Detail::Content> initialData;
    NN_RESULT_DO(DecryptAndVerifyInitialData(&initialData, *pInitialData));

    std::shared_ptr<fs::IStorage> concatenatedStorage;
    NN_RESULT_DO(OpenConcatenatedSaveDataStorage(&concatenatedStorage, m_pImpl.Get(), static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId));

    fs::SaveDataExtraData extraData;
    NN_RESULT_DO(m_pImpl->ReadSaveDataFileSystemExtraDataCore(&extraData, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, false));

    auto exporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataDivisionExporter, SaveDataExporterVersion2>(extraData, m_Configuration, m_KeySeedPackage, m_pPorterManager, true);
    NN_RESULT_THROW_UNLESS(exporter != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(exporter.GetImpl().Initialize(std::move(concatenatedStorage), extraData, std::move(initialData)));
    outValue.Set(std::move(exporter));

    NN_RESULT_SUCCESS;
}

Result SaveDataTransferManagerVersion2::OpenSaveDataImporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataDivisionImporter>> outValue, const nn::sf::InBuffer& initialDataBuffer, const fs::UserId& userId, uint8_t saveDataSpaceId) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(initialDataBuffer.GetSize() >= sizeof(InitialDataVersion2Detail), fs::ResultInvalidSize());
    auto pInitialData = reinterpret_cast<const InitialDataVersion2Detail*>(initialDataBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pInitialData != nullptr, fs::ResultNullptrArgument());

    NN_RESULT_THROW_UNLESS(m_IsKspSet, fs::ResultPreconditionViolation());

    std::unique_ptr<InitialDataVersion2Detail::Content> initialData;
    NN_RESULT_DO(DecryptAndVerifyInitialData(&initialData, *pInitialData));

    auto importer = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataDivisionImporter, SaveDataImporterVersion2>(m_pImpl.Get(), std::move(initialData), m_Configuration, m_KeySeedPackage, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), userId, m_pPorterManager);
    NN_RESULT_THROW_UNLESS(importer != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(importer.GetImpl().Initialize());
    outValue.Set(std::move(importer));

    NN_RESULT_SUCCESS;
}

Result SaveDataTransferManagerVersion2::OpenSaveDataImporterForDiffImport(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataDivisionImporter>> outValue, const nn::sf::InBuffer& initialDataBuffer, uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(initialDataBuffer.GetSize() >= sizeof(InitialDataVersion2Detail), fs::ResultInvalidSize());
    auto pInitialData = reinterpret_cast<const InitialDataVersion2Detail*>(initialDataBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pInitialData != nullptr, fs::ResultNullptrArgument());

    NN_RESULT_THROW_UNLESS(m_IsKspSet, fs::ResultPreconditionViolation());

    std::unique_ptr<InitialDataVersion2Detail::Content> initialData;
    NN_RESULT_DO(DecryptAndVerifyInitialData(&initialData, *pInitialData));

    // TODO: saveDataId が適切な対象かのチェック
    // TODO: 必要なら対象セーブデータに更新が発生していないことのチェック

    auto importer = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataDivisionImporter, SaveDataImporterVersion2>(m_pImpl.Get(), std::move(initialData), m_Configuration, m_KeySeedPackage, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, m_pPorterManager);
    NN_RESULT_THROW_UNLESS(importer != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(importer.GetImpl().Initialize());
    outValue.Set(std::move(importer));

    NN_RESULT_SUCCESS;
}

// 差分複製インポート
Result SaveDataTransferManagerVersion2::OpenSaveDataImporterForDuplicateDiffImport(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataDivisionImporter>> outValue, const nn::sf::InBuffer& initialDataBuffer, uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(initialDataBuffer.GetSize() >= sizeof(InitialDataVersion2Detail), fs::ResultInvalidSize());
    auto pInitialData = reinterpret_cast<const InitialDataVersion2Detail*>(initialDataBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pInitialData != nullptr, fs::ResultNullptrArgument());

    NN_RESULT_THROW_UNLESS(m_IsKspSet, fs::ResultPreconditionViolation());

    std::unique_ptr<InitialDataVersion2Detail::Content> initialData;
    NN_RESULT_DO(DecryptAndVerifyInitialData(&initialData, *pInitialData));

    // TODO: saveDataId が適切な対象かのチェック

    auto importer = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataDivisionImporter, SaveDataImporterVersion2>(m_pImpl.Get(), std::move(initialData), m_Configuration, m_KeySeedPackage, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, m_pPorterManager, true);
    NN_RESULT_THROW_UNLESS(importer != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(importer.GetImpl().Initialize());
    outValue.Set(std::move(importer));

    NN_RESULT_SUCCESS;
}


Result SaveDataTransferManagerVersion2::DecryptAndVerifyInitialData(std::unique_ptr<InitialDataVersion2Detail::Content>* pOutValue, const InitialDataVersion2Detail& untrustedInitialData) NN_NOEXCEPT
{
    // サイズ的に Newable で取るべきか検討
    auto initialData = std::make_unique<InitialDataVersion2Detail::Content>();

    // AES-GCM 復号・検証
    {
        // TODO: 正式な鍵
        // TODO: 鍵世代
        char key[16];
        m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData, m_KeySeedPackage.keySeed, sizeof(m_KeySeedPackage.keySeed));

        char mac[crypto::Aes128GcmDecryptor::MacSize];
        auto decryptedSize = crypto::DecryptAes128Gcm(
            initialData.get(), sizeof(InitialDataVersion2Detail::Content),
            &mac, sizeof(mac),
            key, sizeof(key),
            untrustedInitialData.aesGcmStreamHeader.iv, sizeof(untrustedInitialData.aesGcmStreamHeader.iv),
            &untrustedInitialData.content, sizeof(untrustedInitialData.content),
            nullptr, 0
        );
        NN_UNUSED(decryptedSize);

        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, untrustedInitialData.aesGcmStreamTail.mac, sizeof(mac)), fs::ResultSaveDataTransferInitialDataMacVerificationFailed());
    }

    // ksp の cmac と検証
    {
        // TODO: 正式な鍵
        // TODO: 鍵世代
        char key[16];
        m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken, nullptr, 0);

        char mac[crypto::Aes128CmacGenerator::MacSize];
        crypto::GenerateAes128Cmac(mac, sizeof(mac), initialData.get(), sizeof(InitialDataVersion2Detail::Content), key, sizeof(key));
#if 0
        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, m_KeySeedPackage.initialDataMac, sizeof(mac)), fs::ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed());
#else
        if (!crypto::IsSameBytes(mac, m_KeySeedPackage.initialDataMac, sizeof(mac)))
        {
            NN_SDK_LOG("fs::ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed() (ignored)\n");
        }
#endif
    }

    *pOutValue = std::move(initialData);
    NN_RESULT_SUCCESS;
}


}}}
