﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <mutex>

#include <nn/os.h>

namespace nn { namespace fssrv { namespace detail {

template <typename T>
class RingBuffer
{
    NN_DISALLOW_COPY(RingBuffer);

public:
    RingBuffer() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_BufferCount(0)
        , m_OffsetWrite(0)
        , m_OffsetRead(0)
        , m_Mutex(false)
    {
    }

    ~RingBuffer() NN_NOEXCEPT
    {
    }

    void SetBuffer(T* buffer, size_t bufferCount) NN_NOEXCEPT
    {
        m_Buffer = buffer;
        m_BufferCount = bufferCount;
    }

    size_t Write(const T* buffer, size_t bufferCount) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return WriteImpl(buffer, bufferCount);
    }

    size_t Read(T* outBuffer, size_t outBufferLength) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return ReadImpl(outBuffer, outBufferLength);
    }

    size_t Peek(T* outBuffer, size_t outBufferLength) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return PeekImpl(outBuffer, outBufferLength);
    }

    size_t Ignore(size_t ignoreCount) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return IgnoreImpl(ignoreCount);
    }

    size_t GetReadableCount() const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return GetReadableCountImpl();
    }

    size_t GetWritableCount() const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        return GetWritableCountImpl();
    }

protected:
    size_t WriteImpl(const T* buffer, size_t bufferCount) NN_NOEXCEPT
    {
        const size_t countWrite = std::min(bufferCount, GetWritableCountImpl());
        for( size_t index = 0; index < countWrite; index++ )
        {
            m_Buffer[m_OffsetWrite] = buffer[index];
            m_OffsetWrite = (m_OffsetWrite + 1) % m_BufferCount;
        }
        return countWrite;
    }

    size_t ReadImpl(T* outBuffer, size_t outBufferLength) NN_NOEXCEPT
    {
        const size_t countRead = std::min(outBufferLength, GetReadableCountImpl());
        for( size_t index = 0; index < countRead; index++ )
        {
            outBuffer[index] = m_Buffer[m_OffsetRead];
            m_OffsetRead = (m_OffsetRead + 1) % m_BufferCount;
        }
        return countRead;
    }

    size_t PeekImpl(T* outBuffer, size_t outBufferLength) NN_NOEXCEPT
    {
        const size_t countRead = std::min(outBufferLength, GetReadableCountImpl());
        for( size_t index = 0; index < countRead; index++ )
        {
            outBuffer[index] = m_Buffer[(m_OffsetRead + index) % m_BufferCount];
        }
        return countRead;
    }

    size_t IgnoreImpl(size_t ignoreCount) NN_NOEXCEPT
    {
        const size_t countRead = std::min(ignoreCount, GetReadableCountImpl());
        m_OffsetRead = (m_OffsetRead + countRead) % m_BufferCount;
        return countRead;
    }

protected:
    size_t GetReadableCountImpl() const NN_NOEXCEPT
    {
        if( m_OffsetRead <= m_OffsetWrite )
        {
            return m_OffsetWrite - m_OffsetRead;
        }
        else
        {
            return m_BufferCount - m_OffsetRead + m_OffsetWrite;
        }
    }

    size_t GetWritableCountImpl() const NN_NOEXCEPT
    {
        return m_BufferCount - GetReadableCountImpl() - 1;
    }

protected:
    std::unique_lock<nn::os::Mutex> GetScopedLock() const NN_NOEXCEPT
    {
        return std::unique_lock<nn::os::Mutex>(m_Mutex);
    }

private:
    T* m_Buffer;
    size_t m_BufferCount;
    size_t m_OffsetWrite;
    size_t m_OffsetRead;
    mutable nn::os::Mutex m_Mutex;
};

}}}
