﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Result.private.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/lr/lr_Result.h>
#include <nn/fssystem/fs_Assert.h>

#include "fssrv_ProgramRegistryManager.h"

namespace nn { namespace fssrv { namespace detail {

    ProgramRegistryManager::ProgramRegistryManager() NN_NOEXCEPT
      : m_ProgramInfoListMutex(false)
    {
    }

    ProgramRegistryManager::~ProgramRegistryManager() NN_NOEXCEPT
    {
    }

    Result ProgramRegistryManager::RegisterProgram(uint64_t processId, uint64_t programId, uint8_t storageId, const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT
    {
        std::unique_ptr<ProgramInfoNode> newInfoNode(new ProgramInfoNode());
        NN_RESULT_THROW_UNLESS(newInfoNode, nn::fs::ResultAllocationMemoryFailedInProgramRegistryManagerA());

        auto newInfo = fssystem::AllocateShared<ProgramInfo>(processId, programId, storageId, fsAccessControlData, dataSize, fsAccessControlDesc, descSize);
        NN_RESULT_THROW_UNLESS(newInfo, nn::fs::ResultAllocationMemoryFailedInProgramRegistryManagerA());
        newInfoNode->pProgramInfo = std::move(newInfo);

        {
            std::lock_guard<os::Mutex> scopedLock(m_ProgramInfoListMutex);
            // TORIEAZU: 線形探索
            for( auto& infoNode : m_ProgramInfoList )
            {
                if(infoNode.pProgramInfo->Contains(processId) )
                {
                    return nn::fs::ResultInvalidArgument();
                }
            }
            m_ProgramInfoList.push_back(*newInfoNode.release());
        }
        NN_RESULT_SUCCESS;
    }

    Result ProgramRegistryManager::UnregisterProgram(uint64_t processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ProgramInfoListMutex);
        for( auto& infoNode : m_ProgramInfoList )
        {
            if(infoNode.pProgramInfo->Contains(processId) )
            {
                m_ProgramInfoList.erase(m_ProgramInfoList.iterator_to(infoNode));
                delete &infoNode;
                NN_RESULT_SUCCESS;
            }
        }
        NN_SDK_LOG("The process ID is not registered\n");
        NN_SDK_ASSERT(false);
        return nn::fs::ResultInvalidArgument();
    }

    Result ProgramRegistryManager::GetProgramInfo(std::shared_ptr<ProgramInfo>* pOutValue, uint64_t processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ProgramInfoListMutex);

        if( IsInitialProgram(processId) ) // 初期プロセスまたは Windows
        {
            *pOutValue = ProgramInfo::GetProgramInfoForInitialProcess();
            NN_RESULT_SUCCESS;
        }

        for( auto& infoNode : m_ProgramInfoList )
        {
            if(infoNode.pProgramInfo->Contains(processId) )
            {
                *pOutValue = infoNode.pProgramInfo;
                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultProgramInfoNotFound();
    }

    Result ProgramRegistryManager::GetProgramInfoByProgramId(std::shared_ptr<ProgramInfo>* pOutValue, Bit64 programId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_ProgramInfoListMutex);

        for( auto& info : m_ProgramInfoList )
        {
            if( info.pProgramInfo->GetProgramIdValue() == programId )
            {
                *pOutValue = info.pProgramInfo;
                NN_RESULT_SUCCESS;
            }
        }
        return nn::fs::ResultProgramInfoNotFound();
    }

}}}
