﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "fssrv_AccessControl.h"

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/ncm/ncm_ContentMetaId.h>

namespace nn { namespace fssrv { namespace detail {

    namespace {
#pragma pack(4)
        struct AccessControlDataHeader
        {
            uint8_t version;
            char reserved[3];
            Bit64 flagBits;
            uint32_t offsetOfContentOwnerInfos;
            uint32_t sizeOfContentOwnerInfos;
            uint32_t offsetOfSaveDataOwnerInfos;
            uint32_t sizeOfSaveDataOwnerInfos;
        };
        NN_STATIC_ASSERT(std::is_pod<AccessControlDataHeader>::value);

        struct AccessControlDescriptor
        {
            uint8_t version;
            uint8_t contentOwnerIdCount;
            uint8_t saveDataOwnerIdCount;
            char reserved;
            Bit64 flagBits;
            Bit64 contentOwnerIdBegin;
            Bit64 contentOwnerIdEnd;
            Bit64 saveDataOwnerIdBegin;
            Bit64 saveDataOwnerIdEnd;
            // Bit64 contentOwnerIds[contentOwnerIdCount];
            // Bit64 saveDataOwnerIds[saveDataOwnerIdCount];
        };
        NN_STATIC_ASSERT(std::is_pod<AccessControlDescriptor>::value);
#pragma pack()

        const uint8_t LatestFsAccessControlInfoVersion = 1; // MakeMeta, MakeDesc, ProgramRegistryManager と合わせる
        bool g_IsDebugFlagEnabled = false;
        const Bit64 DebugFlagDisableMask = 0x3FFFFFFFFFFFFFFF; // FullPermission, Debug を落とす
    }

    AccessControl::AccessControl(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT
        : AccessControl(fsAccessControlData, dataSize, fsAccessControlDesc, descSize, g_IsDebugFlagEnabled ? 0xFFFFFFFFFFFFFFFF : DebugFlagDisableMask)
    {
    }

    AccessControl::AccessControl(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize, Bit64 flagDisableMask) NN_NOEXCEPT
    {
        // 以下の条件のうちいずれかを満たすとき、ABORT します
        // - dataSize < sizeof(AccessControlDataHeader)
        // - descSize < sizeof(AccessControlDescriptor)
        // - fsAccessControlData のバージョンが 0
        // - fsAccessControlData の内部データが異常
        // - dataSize が小さすぎる

        if (fsAccessControlData == nullptr || fsAccessControlDesc == nullptr)
        {
            m_FlagBits.emplace(0ULL);
            return;
        }

        NN_ABORT_UNLESS(dataSize >= static_cast<int64_t>(sizeof(AccessControlDataHeader)));
        NN_ABORT_UNLESS(descSize >= static_cast<int64_t>(sizeof(AccessControlDescriptor)));

        AccessControlDataHeader data = { 0 };
        AccessControlDescriptor desc = { 0 };
        std::memcpy(&data, fsAccessControlData, sizeof(AccessControlDataHeader));
        std::memcpy(&desc, fsAccessControlDesc, sizeof(AccessControlDescriptor));

        if (data.version != desc.version || data.version < LatestFsAccessControlInfoVersion)
        {
            NN_SDK_LOG("[fs] Warning: fs ACI is invalid version and not set. fs ACI ver. = %d (Latest = %d), fs ACID ver. = %d.", data.version, LatestFsAccessControlInfoVersion, desc.version);
            m_FlagBits.emplace(0ULL);
            return;
        }

        desc.flagBits &= flagDisableMask;

        m_FlagBits.emplace(data.flagBits & desc.flagBits);

        uintptr_t pHead = reinterpret_cast<uintptr_t>(fsAccessControlData);
        uintptr_t pDescHead = reinterpret_cast<uintptr_t>(fsAccessControlDesc);

        NN_ABORT_UNLESS(dataSize >= data.offsetOfContentOwnerInfos + data.sizeOfContentOwnerInfos);
        NN_ABORT_UNLESS(descSize >= static_cast<int64_t>(sizeof(AccessControlDescriptor) + desc.contentOwnerIdCount * sizeof(Bit64)));
        if (data.sizeOfContentOwnerInfos != 0)
        {
            uintptr_t numOffset = pHead + data.offsetOfContentOwnerInfos;
            uintptr_t idStart = numOffset + sizeof(uint32_t);

            uint32_t numOfContentOwnerInfos = 0;
            std::memcpy(&numOfContentOwnerInfos, reinterpret_cast<void*>(numOffset), sizeof(numOfContentOwnerInfos));

            uintptr_t idEnd = idStart + numOfContentOwnerInfos * sizeof(Bit64);
            NN_ABORT_UNLESS(idEnd == numOffset + data.sizeOfContentOwnerInfos);

            for (uint32_t i = 0; i < numOfContentOwnerInfos; ++i)
            {
                Bit64 id = *reinterpret_cast<Bit64*>(idStart + i * sizeof(Bit64));

                bool checked = false;

                // Desc に Id が含まれているかチェック
                if (desc.contentOwnerIdCount != 0)
                {
                    for (uint8_t j = 0; j < desc.contentOwnerIdCount; j++)
                    {
                        auto descId = *reinterpret_cast<Bit64*>(pDescHead + sizeof(AccessControlDescriptor) + j * sizeof(Bit64));
                        if (id == descId)
                        {
                            checked = true;
                            break;
                        }
                    }
                }
                // TODO 削除: ContentOwnerIdMin, Max がデフォルト値のままの場合ははじかない
                else if ((desc.contentOwnerIdBegin == 0 && desc.contentOwnerIdEnd == 0) ||
                         (id >= desc.contentOwnerIdBegin && id <= desc.contentOwnerIdEnd))
                {
                    checked = true;
                }

                if (checked)
                {
                    ContentOwnerInfo* info = new ContentOwnerInfo(id);
                    if (!info)
                    {
                        NN_SDK_LOG("[fs] Warning: Out of resource for ACI settings.");
                        return;
                    }
                    m_ContentOwnerInfos.push_front(*info);
                }
            }
        }

        NN_ABORT_UNLESS(dataSize >= data.offsetOfSaveDataOwnerInfos + data.sizeOfSaveDataOwnerInfos);
        NN_ABORT_UNLESS(descSize >= static_cast<int64_t>(sizeof(AccessControlDescriptor) + desc.contentOwnerIdCount * sizeof(Bit64) + desc.saveDataOwnerIdCount * sizeof(Bit64)));
        if (data.sizeOfSaveDataOwnerInfos != 0)
        {
            uintptr_t numOffset = pHead + data.offsetOfSaveDataOwnerInfos;
            uintptr_t accessibilityStart = numOffset + sizeof(uint32_t);

            uint32_t numOfSaveDataOwnerInfos = 0;
            std::memcpy(&numOfSaveDataOwnerInfos, reinterpret_cast<void*>(numOffset), sizeof(numOfSaveDataOwnerInfos));

            static const size_t SaveDataAccessibilityAlignment = 4;
            uintptr_t idStart = accessibilityStart + nn::util::align_up(numOfSaveDataOwnerInfos * sizeof(Accessibility), SaveDataAccessibilityAlignment);
            uintptr_t idEnd = idStart + numOfSaveDataOwnerInfos * sizeof(Bit64);
            NN_ABORT_UNLESS(idEnd == numOffset + data.sizeOfSaveDataOwnerInfos);

            for (uint32_t i = 0; i < numOfSaveDataOwnerInfos; ++i)
            {
                Accessibility accessibility = *reinterpret_cast<Accessibility*>(accessibilityStart + i * sizeof(Accessibility));
                Bit64 id = *reinterpret_cast<Bit64*>(idStart + i * sizeof(Bit64));

                bool checked = false;

                // Desc に Id が含まれているかチェック
                if (desc.saveDataOwnerIdCount != 0)
                {
                    for (uint8_t j = 0; j < desc.saveDataOwnerIdCount; j++)
                    {
                        auto descId = *reinterpret_cast<Bit64*>(pDescHead + sizeof(AccessControlDescriptor) + desc.contentOwnerIdCount * sizeof(Bit64) + j * sizeof(Bit64));
                        if (id == descId)
                        {
                            checked = true;
                            break;
                        }
                    }
                }
                // TODO 削除: SaveDataOwnerIdMin, Max がデフォルト値のままの場合ははじかない
                else if ((desc.saveDataOwnerIdBegin == 0 && desc.saveDataOwnerIdEnd == 0) ||
                         (id >= desc.saveDataOwnerIdBegin && id <= desc.saveDataOwnerIdEnd))
                {
                    checked = true;
                }

                if (checked)
                {
                    SaveDataOwnerInfo* info = new SaveDataOwnerInfo(id, accessibility);
                    if (!info)
                    {
                        NN_SDK_LOG("[fs] Warning: Out of resource for ACI settings.");
                        return;
                    }
                    m_SaveDataOwnerInfos.push_front(*info);
                }
            }
        }
    } // NOLINT(impl/function_size)

    AccessControl::~AccessControl() NN_NOEXCEPT
    {
        while (!m_ContentOwnerInfos.empty())
        {
            ContentOwnerInfo* p = &(*m_ContentOwnerInfos.rbegin());
            m_ContentOwnerInfos.erase(m_ContentOwnerInfos.iterator_to(*p));
            delete p;
        }

        while (!m_SaveDataOwnerInfos.empty())
        {
            SaveDataOwnerInfo* p = &(*m_SaveDataOwnerInfos.rbegin());
            m_SaveDataOwnerInfos.erase(m_SaveDataOwnerInfos.iterator_to(*p));
            delete p;
        }
    }

    bool AccessControl::HasContentOwnerId(Bit64 ownerId) const NN_NOEXCEPT
    {
        for (auto& ownerInfo : m_ContentOwnerInfos)
        {
            if (ownerInfo.GetId() == ownerId)
            {
                return true;
            }
        }
        return false;
    }

    Accessibility AccessControl::GetAccessibilitySaveDataOwnedBy(Bit64 ownerId) const NN_NOEXCEPT
    {
        for (auto& ownerInfo : m_SaveDataOwnerInfos)
        {
            if (ownerInfo.GetId() == ownerId)
            {
                return ownerInfo.GetAccessibility();
            }
        }
        return Accessibility::MakeAccessibility(false, false);
    }

    void AccessControl::ListSaveDataOwnedId(int32_t* outValue, ncm::ApplicationId* outOwnerId, int32_t offset, int32_t count) const NN_NOEXCEPT
    {
        int32_t readOffset = 0;
        int32_t readCount = 0;

        if (count == 0)
        {
            *outValue = m_SaveDataOwnerInfos.size();
            return;
        }

        if (outOwnerId != nullptr)
        {
            ncm::ApplicationId* p = outOwnerId;
            for (auto& ownerInfo : m_SaveDataOwnerInfos)
            {
                if (readOffset < offset)
                {
                    readOffset++;
                    continue;
                }

                p->value = ownerInfo.GetId();
                p++;
                readCount++;

                if (readCount == count)
                {
                    *outValue = readCount;
                    return;
                }
            }
        }

        *outValue = readCount;
        return;
    }

    Accessibility AccessControl::GetAccessibilityFor(AccessibilityType type) const NN_NOEXCEPT
    {
        const AccessControlBits* pFlagBits = &m_FlagBits.value();
        switch (type)
        {
        case AccessibilityType::MountLogo:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountLogoRead(), false);
        case AccessibilityType::MountContentMeta:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountContentMetaRead(), false);
        case AccessibilityType::MountContentControl:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountContentControlRead(), false);
        case AccessibilityType::MountContentManual:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountContentManualRead(), false);
        case AccessibilityType::MountContentData:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountContentDataRead(), false);
        case AccessibilityType::MountApplicationPackage:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountApplicationPackageRead(), false);
        case AccessibilityType::MountSaveDataStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountSaveDataStorageRead(), pFlagBits->CanMountSaveDataStorageWrite());
        case AccessibilityType::MountContentStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountContentStorageRead(), pFlagBits->CanMountContentStorageWrite());
        case AccessibilityType::MountImageAndVideoStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountImageAndVideoStorageRead(), pFlagBits->CanMountImageAndVideoStorageWrite());
        case AccessibilityType::MountCloudBackupWorkStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountCloudBackupWorkStorageRead(), pFlagBits->CanMountCloudBackupWorkStorageWrite());
        case AccessibilityType::MountBisCalibrationFile:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisCalibrationFileRead(), pFlagBits->CanMountBisCalibrationFileWrite());
        case AccessibilityType::MountBisSafeMode:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisSafeModeRead(), pFlagBits->CanMountBisSafeModeWrite());
        case AccessibilityType::MountBisUser:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisUserRead(), pFlagBits->CanMountBisUserWrite());
        case AccessibilityType::MountBisSystem:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisSystemRead(), pFlagBits->CanMountBisSystemWrite());
        case AccessibilityType::MountBisSystemProperEncryption:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisSystemProperEncryptionRead(), pFlagBits->CanMountBisSystemProperEncryptionWrite());
        case AccessibilityType::MountBisSystemProperPartition:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountBisSystemProperPartitionRead(), pFlagBits->CanMountBisSystemProperPartitionWrite());
        case AccessibilityType::MountSdCard:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountSdCardRead(), pFlagBits->CanMountSdCardWrite());
        case AccessibilityType::MountGameCard:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountGameCardRead(), false);
        case AccessibilityType::MountDeviceSaveData:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountDeviceSaveDataRead(), pFlagBits->CanMountDeviceSaveDataWrite());
        case AccessibilityType::MountSystemSaveData:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountSystemSaveDataRead(), pFlagBits->CanMountSystemSaveDataWrite());
        case AccessibilityType::MountOthersSaveData:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountOthersSaveDataRead(), pFlagBits->CanMountOthersSaveDataWrite());
        case AccessibilityType::MountOthersSystemSaveData:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountOthersSystemSaveDataRead(), pFlagBits->CanMountOthersSystemSaveDataWrite());
        case AccessibilityType::OpenBisPartitionBootPartition1Root:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootPartition1RootRead(), pFlagBits->CanOpenBisPartitionBootPartition1RootWrite());
        case AccessibilityType::OpenBisPartitionBootPartition2Root:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootPartition2RootRead(), pFlagBits->CanOpenBisPartitionBootPartition2RootWrite());
        case AccessibilityType::OpenBisPartitionUserDataRoot:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionUserDataRootRead(), pFlagBits->CanOpenBisPartitionUserDataRootWrite());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part1:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part1Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part1Write());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part2:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part2Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part2Write());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part3:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part3Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part3Write());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part4:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part4Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part4Write());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part5:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part5Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part5Write());
        case AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part6:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part6Read(), pFlagBits->CanOpenBisPartitionBootConfigAndPackage2Part6Write());
        case AccessibilityType::OpenBisPartitionCalibrationBinary:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionCalibrationBinaryRead(), pFlagBits->CanOpenBisPartitionCalibrationBinaryWrite());
        case AccessibilityType::OpenBisPartitionCalibrationFile:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionCalibrationFileRead(), pFlagBits->CanOpenBisPartitionCalibrationFileWrite());
        case AccessibilityType::OpenBisPartitionSafeMode:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionSafeModeRead(), pFlagBits->CanOpenBisPartitionSafeModeWrite());
        case AccessibilityType::OpenBisPartitionUser:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionUserRead(), pFlagBits->CanOpenBisPartitionUserWrite());
        case AccessibilityType::OpenBisPartitionSystem:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionSystemRead(), pFlagBits->CanOpenBisPartitionSystemWrite());
        case AccessibilityType::OpenBisPartitionSystemProperEncryption:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionSystemProperEncryptionRead(), pFlagBits->CanOpenBisPartitionSystemProperEncryptionWrite());
        case AccessibilityType::OpenBisPartitionSystemProperPartition:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenBisPartitionSystemProperPartitionRead(), pFlagBits->CanOpenBisPartitionSystemProperPartitionWrite());
        case AccessibilityType::OpenSdCardStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenSdCardStorageRead(), pFlagBits->CanOpenSdCardStorageWrite());
        case AccessibilityType::OpenGameCardStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenGameCardStorageRead(), pFlagBits->CanOpenGameCardStorageWrite());
        case AccessibilityType::MountSystemDataPrivate:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountSystemDataPrivateRead(), false);
        case AccessibilityType::MountHost:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountHostRead(), pFlagBits->CanMountHostWrite());
        case AccessibilityType::MountRegisteredUpdatePartition:
            return Accessibility::MakeAccessibility(pFlagBits->CanMountRegisteredUpdatePartitionRead() && g_IsDebugFlagEnabled, false);
        case AccessibilityType::MountSaveDataInternalStorage:
            return Accessibility::MakeAccessibility(pFlagBits->CanOpenSaveDataInternalStorageRead(), pFlagBits->CanOpenSaveDataInternalStorageWrite());
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    } //NOLINT(impl/function_size)

    bool AccessControl::CanCall(OperationType type) const NN_NOEXCEPT
    {
        const AccessControlBits* pFlagBits = &m_FlagBits.value();
        switch (type)
        {
        case OperationType::InvalidateBisCache:
            return pFlagBits->CanInvalidateBisCache();
        case OperationType::EraseMmc:
            return pFlagBits->CanEraseMmc();
        case OperationType::GetGameCardDeviceCertificate:
            return pFlagBits->CanGetGameCardDeviceCertificate();
        case OperationType::GetGameCardIdSet:
            return pFlagBits->CanGetGameCardIdSet();
        case OperationType::FinalizeGameCardDriver:
            return pFlagBits->CanFinalizeGameCardDriver();
        case OperationType::GetGameCardAsicInfo:
            return pFlagBits->CanGetGameCardAsicInfo();
        case OperationType::CreateSaveData:
            return pFlagBits->CanCreateSaveData();
        case OperationType::DeleteSaveData:
            return pFlagBits->CanDeleteSaveData();
        case OperationType::CreateSystemSaveData:
            return pFlagBits->CanCreateSystemSaveData();
        case OperationType::CreateOthersSystemSaveData:
            return pFlagBits->CanCreateOthersSystemSaveData();
        case OperationType::DeleteSystemSaveData:
            return pFlagBits->CanDeleteSystemSaveData();
        case OperationType::OpenSaveDataInfoReader:
            return pFlagBits->CanOpenSaveDataInfoReader();
        case OperationType::OpenSaveDataInfoReaderForSystem:
            return pFlagBits->CanOpenSaveDataInfoReaderForSystem();
        case OperationType::OpenSaveDataInfoReaderForInternal:
            return pFlagBits->CanOpenSaveDataInfoReaderForInternal();
        case OperationType::OpenSaveDataMetaFile:
            return pFlagBits->CanOpenSaveDataMetaFile();
        case OperationType::SetCurrentPosixTime:
            return pFlagBits->CanSetCurrentPosixTime();
        case OperationType::ReadSaveDataFileSystemExtraData:
            return pFlagBits->CanReadSaveDataFileSystemExtraData();
        case OperationType::SetGlobalAccessLogMode:
            return pFlagBits->CanSetGlobalAccessLogMode();
        case OperationType::SetSpeedEmulationMode:
            return pFlagBits->CanSetSpeedEmulationMode();
        case OperationType::FillBis:
            return pFlagBits->CanFillBis();
        case OperationType::CorruptSaveData:
            return pFlagBits->CanCorruptSaveData();
        case OperationType::CorruptSystemSaveData:
            return pFlagBits->CanCorruptSystemSaveData();
        case OperationType::VerifySaveData:
            return pFlagBits->CanVerifySaveData();
        case OperationType::DebugSaveData:
            return pFlagBits->CanDebugSaveData();
        case OperationType::FormatSdCard:
            return pFlagBits->CanFormatSdCard();
        case OperationType::GetRightsId:
            return pFlagBits->CanGetRightsId();
        case OperationType::RegisterExternalKey:
            return pFlagBits->CanRegisterExternalKey();
        case OperationType::SetEncryptionSeed:
            return pFlagBits->CanSetEncryptionSeed();
        case OperationType::WriteSaveDataFileSystemExtraDataTimeStamp:
            return pFlagBits->CanWriteSaveDataFileSystemExtraDataTimeStamp();
        case OperationType::WriteSaveDataFileSystemExtraDataFlags:
            return pFlagBits->CanWriteSaveDataFileSystemExtraDataFlags();
        case OperationType::WriteSaveDataFileSystemExtraDataCommitId:
            return pFlagBits->CanWriteSaveDataFileSystemExtraDataCommitId();
        case OperationType::WriteSaveDataFileSystemExtraDataAll:
            return pFlagBits->CanWriteSaveDataFileSystemExtraDataAll();
        case OperationType::ExtendSaveData:
            return pFlagBits->CanExtendSaveData();
        case OperationType::ExtendSystemSaveData:
            return pFlagBits->CanExtendSystemSaveData();
        case OperationType::RegisterUpdatePartition:
            return pFlagBits->CanRegisterUpdatePartition() && g_IsDebugFlagEnabled;
        case OperationType::OpenSaveDataTransferManager:
            return pFlagBits->CanOpenSaveDataTransferManager();
        case OperationType::OpenSaveDataTransferManagerVersion2:
            return pFlagBits->CanOpenSaveDataTransferManagerVersion2();
        case OperationType::OpenSaveDataTransferProhibiter:
            return pFlagBits->CanOpenSaveDataTransferProhibiter();
        case OperationType::ListAccessibleSaveDataOwnerId:
            return pFlagBits->CanListAccessibleSaveDataOwnerId();
        case OperationType::GetSaveDataCommitId:
            return pFlagBits->CanGetSaveDataCommitId();
        case OperationType::ControlMmcPatrol:
            return pFlagBits->CanControlMmcPatrol();
        case OperationType::OverrideSaveDataTransferTokenSignVerificationKey:
            return pFlagBits->CanOverrideSaveDataTransferTokenSignVerificationKey();
        case OperationType::OpenSdCardDetectionEventNotifier:
            return pFlagBits->CanOpenSdCardDetectionEventNotifier();
        case OperationType::OpenGameCardDetectionEventNotifier:
            return pFlagBits->CanOpenGameCardDetectionEventNotifier();
        case OperationType::OpenSystemDataUpdateEventNotifier:
            return pFlagBits->CanOpenSystemDataUpdateEventNotifier();
        case OperationType::NotifySystemDataUpdateEvent:
            return pFlagBits->CanNotifySystemDataUpdateEvent();
        case OperationType::OpenAccessFailureDetectionEventNotifier:
            return pFlagBits->CanOpenAccessFailureDetectionEventNotifier();
        case OperationType::GetAccessFailureDetectionEvent:
            return pFlagBits->CanGetAccessFailureDetectionEvent();
        case OperationType::IsAccessFailureDetected:
            return pFlagBits->CanIsAccessFailureDetected();
        case OperationType::ResolveAccessFailure:
            return pFlagBits->CanResolveAccessFailure();
        case OperationType::AbandonAccessFailure:
            return pFlagBits->CanAbandonAccessFailure();
        case OperationType::QuerySaveDataInternalStorageTotalSize:
            return pFlagBits->CanQuerySaveDataInternalStorageTotalSize();
        case OperationType::SetSdCardAccessibility:
            return pFlagBits->CanSetSdCardAccessibility();
        case OperationType::SimulateDevice:
            return pFlagBits->CanSimulateDevice();
        case OperationType::SetDataStorageRedirectTarget:
            return pFlagBits->CanSetDataStorageRedirectTarget() && g_IsDebugFlagEnabled;
        case OperationType::CreateSaveDataWithHashSalt:
            return pFlagBits->CanCreateSaveDataWithHashSalt();

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    } //NOLINT(impl/function_size)

    void SetDebugFlagEnabled(bool enabled) NN_NOEXCEPT
    {
        g_IsDebugFlagEnabled = enabled;
    }

}}}
