﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/fs/detail/fs_AccessLog.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

namespace
{
    const int32_t DefaultSaveDataBlockSize           = 16 * 1024;
    const int32_t DefaultSaveDataSizeForDebug        = 32 * 1024 * 1024;
    const int32_t DefaultSaveDataJournalSizeForDebug = 32 * 1024 * 1024;

    Result MountSaveDataImpl(const char* name, SaveDataSpaceId spaceId, const nn::ncm::ProgramId programId, UserId userId, SaveDataType type, bool isReadOnly, uint16_t index) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(programId, type, userId, InvalidSystemSaveDataId, index);
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        if (isReadOnly)
        {
            NN_RESULT_DO(fileSystemProxy->OpenReadOnlySaveDataFileSystem(&fileSystem, static_cast<uint32_t>(spaceId), attribute));
        }
        else
        {
            NN_RESULT_DO(fileSystemProxy->OpenSaveDataFileSystem(&fileSystem, static_cast<uint32_t>(spaceId), attribute));
        }

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract), nullptr, false, true, true));
        NN_RESULT_SUCCESS;
    }

    Result MountSaveDataImpl(const char* name, SaveDataSpaceId spaceId, const nn::ncm::ProgramId programId, UserId userId, SaveDataType type, bool isReadOnly) NN_NOEXCEPT
    {
        return MountSaveDataImpl(name, spaceId, programId, userId, type, isReadOnly, 0);
    }
}

Result MountSaveData(const char* name, UserId userId) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(InvalidProgramId, SaveDataType::Account, userId, InvalidSystemSaveDataId);

        // SaveDataForDebug の作成は shim から行う
        if( userId == InvalidUserId)
        {
            SaveDataCreationInfo creationInfo;
            creationInfo.size        = DefaultSaveDataSizeForDebug;
            creationInfo.journalSize = DefaultSaveDataJournalSizeForDebug;
            creationInfo.blockSize   = DefaultSaveDataBlockSize;
            creationInfo.ownerId     = 0;
            creationInfo.flags       = 0;
            creationInfo.spaceId     = SaveDataSpaceId::User;
            creationInfo.isPseudoSaveFs = false;

            SaveDataMetaInfo metaInfo;
            metaInfo.type = SaveDataMetaType::None;
            metaInfo.size = 0;

            NN_RESULT_TRY(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo))
                NN_RESULT_CATCH(ResultPathAlreadyExists)
                {
                }
            NN_RESULT_END_TRY
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenSaveDataFileSystem(&fileSystem, static_cast<uint32_t>(SaveDataSpaceId::User), attribute));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract), nullptr, false, true, true));
        NN_RESULT_SUCCESS;
    };

    Result result = NN_DETAIL_FS_ACCESS_LOG_MOUNT(mount(), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSAVEDATA(name, userId));

    // SaveDataForDebug の場合は Result をそのまま返す
    if (userId == InvalidUserId)
    {
        NN_RESULT_DO(result);
    }
    else
    {
        NN_FS_RESULT_DO(result);
    }

    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountSaveData(const char* name, const nn::ncm::ApplicationId applicationId, UserId userId) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(MountSaveDataImpl(name, SaveDataSpaceId::User, applicationId, userId, SaveDataType::Account, false), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSAVEDATA_APPLICATIONID(name, applicationId, userId)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountSaveDataReadOnly(const char* name, const nn::ncm::ApplicationId applicationId, UserId userId) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(MountSaveDataImpl(name, SaveDataSpaceId::User, applicationId, userId, SaveDataType::Account, true), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSAVEDATA_APPLICATIONID(name, applicationId, userId)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

bool IsSaveDataExisting(const UserId& userId) NN_NOEXCEPT
{
    nn::ncm::ApplicationId appId = { InvalidProgramId.value };
    return IsSaveDataExisting(appId, userId);
}

bool IsSaveDataExisting(const nn::ncm::ApplicationId applicationId, const UserId& userId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Account, userId, InvalidSystemSaveDataId);
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

    auto result = fileSystemProxy->OpenSaveDataFileSystem(&fileSystem, static_cast<uint32_t>(SaveDataSpaceId::User), attribute);
    if (result.IsSuccess() || ResultTargetLocked::Includes(result) || ResultSaveDataExtending::Includes(result))
    {
        return true;
    }
    else if (ResultTargetNotFound::Includes(result))
    {
        return false;
    }
    else
    {
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_ABORT();
    }
}

Result MountTemporaryStorage(const char* name) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(MountSaveDataImpl(name, SaveDataSpaceId::Temporary, InvalidProgramId, InvalidUserId, SaveDataType::Temporary, false), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountCacheStorage(const char* name) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_MOUNT(
            MountSaveDataImpl(name, SaveDataSpaceId::User, InvalidProgramId, InvalidUserId, SaveDataType::Cache, false, 0),
            name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name
        )
    );
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountCacheStorage(const char* name, int index) NN_NOEXCEPT
{
    uint16_t internalIndex = static_cast<uint16_t>(index);
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_MOUNT(
            MountSaveDataImpl(name, SaveDataSpaceId::User, InvalidProgramId, InvalidUserId, SaveDataType::Cache, false, internalIndex),
            name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_INDEX(name, index)
        )
    );
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountCacheStorage(const char* name, const nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(
            MountSaveDataImpl(name, SaveDataSpaceId::User, applicationId, InvalidUserId, SaveDataType::Cache, false, 0),
            name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_APPLICATIONID(name, applicationId)
        )
    );
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountCacheStorage(const char* name, const nn::ncm::ApplicationId applicationId, int index) NN_NOEXCEPT
{
    uint16_t internalIndex = static_cast<uint16_t>(index);
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(
            MountSaveDataImpl(name, SaveDataSpaceId::User, applicationId, InvalidUserId, SaveDataType::Cache, false, internalIndex),
            name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_APPLICATIONIDINDEX(name, applicationId, index)
        )
    );
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result OpenSaveDataInternalStorageFileSystem(std::unique_ptr<fsa::IFileSystem>* outValue, SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    NN_RESULT_DO(fileSystemProxy->OpenSaveDataInternalStorageFileSystem(&fileSystem, static_cast<uint32_t>(spaceId), saveDataId));

    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedNew());

    *outValue = std::move(fileSystemAbstract);
    NN_RESULT_SUCCESS;
}

Result MountSaveDataInternalStorage(const char* name, SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto mount = [&]() NN_NOEXCEPT -> nn::Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        NN_RESULT_DO(OpenSaveDataInternalStorageFileSystem(&fileSystemAbstract, spaceId, saveDataId));
        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract)));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(mount());
    NN_RESULT_SUCCESS;
}

}}

