﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_FileSystemProxyTypes.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fs_Logo.h>
#include <nn/util/util_FormatString.h>

#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

Result MountLogo(const char* name, const char* path, ncm::ProgramId programId) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));

        NN_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());
        nn::fssrv::sf::FspPath sfPath;
        NN_RESULT_DO(FspPathPrintf(&sfPath, "%s", path));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenFileSystemWithId(&fileSystem, sfPath, programId.value, detail::FileSystemProxyType::FileSystemProxyType_Logo));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInLogoA());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTLOGO(name, path, programId)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

}}
