﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/fssrv/sf/fssrv_IDeviceOperator.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_DeviceSimulation.h>

#include "fs_Library.h"
#include "fs_EventNotifierObjectAdapter.h"
#include "fs_FileSystemProxyServiceObject.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_StorageServiceObjectAdapter.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"


namespace nn { namespace fs {

using namespace nn::fs::detail;

namespace {

const char* GetGameCardMountNameSuffix(GameCardPartition partition)
{
    switch(partition)
    {
    case GameCardPartition::Update:
        return GameCardFileSystemMountNameSuffixUpdate;
    case GameCardPartition::Normal:
        return GameCardFileSystemMountNameSuffixNormal;
    case GameCardPartition::Secure:
        return GameCardFileSystemMountNameSuffixSecure;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

class GameCardCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public Newable
{
public:
    explicit GameCardCommonMountNameGenerator(GameCardHandle handle, GameCardPartition partition)
        : m_Handle(handle)
        , m_Partition(partition)
    {
    }

    virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t RequiredNameBufferSize = strnlen(GameCardFileSystemMountName, MountNameLengthMax) + strnlen(GetGameCardMountNameSuffix(m_Partition), MountNameLengthMax) + sizeof(m_Handle) * 2 + 1 + 1;
        NN_SDK_REQUIRES(nameSize >= static_cast<size_t>(RequiredNameBufferSize));
        NN_UNUSED(RequiredNameBufferSize);
        auto size = nn::util::SNPrintf(name, nameSize, "%s%s%08x:", GameCardFileSystemMountName, GetGameCardMountNameSuffix(m_Partition), m_Handle);
        NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
        NN_UNUSED(size);

        NN_RESULT_SUCCESS;
    }

private:
    const GameCardHandle m_Handle;
    const GameCardPartition m_Partition;
};


Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType eventType, nn::Result respondingFailureResult, bool isPersistent) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->SetDeviceSimulationEvent(static_cast<uint32_t>(fs::SimulatingDeviceType::GameCard), static_cast<uint32_t>(targetOperationFlag),
        static_cast<uint32_t>(eventType), respondingFailureResult.GetInnerValueForDebug(), isPersistent));
    NN_RESULT_SUCCESS;
}

}

Result GetGameCardHandle(GameCardHandle* pOutValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    uint32_t handle;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardHandle(nn::sf::Out<uint32_t>(&handle)));

    *pOutValue = handle;
    NN_RESULT_SUCCESS;
}

Result OpenGameCardDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> eventNotifier;
    NN_FS_RESULT_DO(fileSystemProxy->OpenGameCardDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>>(&eventNotifier)));
    outValue->reset(new detail::EventNotifierObjectAdapter(std::move(eventNotifier)));
    NN_FS_RESULT_THROW_UNLESS(*outValue, ResultAllocationMemoryFailedInGameCardA());
    NN_RESULT_SUCCESS;
}

bool IsGameCardInserted() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    bool isInserted = false;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(deviceOperator->IsGameCardInserted(nn::sf::Out<bool>(&isInserted)));
    return isInserted;
}


Result OpenGameCardPartition(std::unique_ptr<nn::fs::IStorage>* outValue, GameCardHandle handle, GameCardPartitionRaw partition) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IStorage> storage;
    NN_FS_RESULT_DO(fileSystemProxy->OpenGameCardStorage(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>>(&storage), static_cast<uint32_t>(handle), static_cast<uint32_t>(partition)));

    std::unique_ptr<nn::fs::IStorage> storageAdapter(new detail::StorageServiceObjectAdapter(std::move(storage)));
    NN_FS_RESULT_THROW_UNLESS(storageAdapter, ResultAllocationMemoryFailedInGameCardB());

    *outValue = std::move(storageAdapter);
    NN_RESULT_SUCCESS;
}

Result EraseGameCard(GameCardSize size, uint64_t normalAreaSize) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->EraseGameCard(static_cast<uint32_t>(size), normalAreaSize));
    NN_RESULT_SUCCESS;
}

Result MountGameCardPartition(const char* name, GameCardHandle handle, GameCardPartition partition) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

        NN_RESULT_DO(fileSystemProxy->OpenGameCardFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>>(&fileSystem), static_cast<uint32_t>(handle), static_cast<uint32_t>(partition)));
        std::unique_ptr<FileSystemServiceObjectAdapter> fileSystemAdapter(new FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAdapter.get() != nullptr, ResultAllocationMemoryFailedInGameCardC());

        std::unique_ptr<GameCardCommonMountNameGenerator> mountNameGenerator(new GameCardCommonMountNameGenerator(handle, partition));
        NN_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedInGameCardD());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAdapter), std::move(mountNameGenerator)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTGAMECARDPARTITION(name, handle, partition)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result GetGameCardUpdatePartitionInfo(GameCardUpdatePartitionInfo* pOutUpdatePartitionInfo, GameCardHandle handle) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    uint32_t cupVersion;
    uint64_t cupId;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardUpdatePartitionInfo(nn::sf::Out<uint32_t>(&cupVersion), nn::sf::Out<uint64_t>(&cupId), static_cast<uint32_t>(handle)));

    pOutUpdatePartitionInfo->version = cupVersion;
    pOutUpdatePartitionInfo->id      = cupId;
    NN_RESULT_SUCCESS;
}

void FinalizeGameCardDriver() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(deviceOperator->FinalizeGameCardDriver());
}

Result GetGameCardAttribute(GameCardAttribute* pOutValue, GameCardHandle handle) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    GameCardAttribute gameCardAttribute;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardAttribute(nn::sf::Out<uint8_t>(reinterpret_cast<uint8_t*>(&gameCardAttribute)), static_cast<uint32_t>(handle)));

    *pOutValue = gameCardAttribute;
    NN_RESULT_SUCCESS;
}

Result GetGameCardDeviceCertificate(void* buffer, size_t bufferSize, GameCardHandle handle) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardDeviceCertificate(nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), static_cast<int64_t>(bufferSize), static_cast<uint32_t>(handle)));
    NN_RESULT_SUCCESS;
}

Result GetGameCardAsicInfo(nn::gc::RmaInformation* pOutValue, const void* fwBuffer, size_t fwBufferSize) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    nn::gc::RmaInformation rmaInformation;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardAsicInfo(nn::sf::OutBuffer(reinterpret_cast<char*>(&rmaInformation), sizeof(nn::gc::RmaInformation)), sizeof(nn::gc::RmaInformation), nn::sf::InBuffer(reinterpret_cast<const char*>(fwBuffer), fwBufferSize), static_cast<int64_t>(fwBufferSize)));
    *pOutValue = rmaInformation;
    NN_RESULT_SUCCESS;
}

Result GetGameCardIdSet(nn::gc::GameCardIdSet* pOutValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    nn::gc::GameCardIdSet gcIdSet;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardIdSet(nn::sf::OutBuffer(reinterpret_cast<char*>(&gcIdSet), sizeof(nn::gc::GameCardIdSet)), sizeof(nn::gc::GameCardIdSet)));
    *pOutValue = gcIdSet;
    NN_RESULT_SUCCESS;
}

Result GetGameCardCid(void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(size >= sizeof(nn::gc::GameCardIdSet), ResultInvalidSize());
    NN_UNUSED(size);
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    nn::gc::GameCardIdSet gcIdSet;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardIdSet(nn::sf::OutBuffer(reinterpret_cast<char*>(&gcIdSet), sizeof(nn::gc::GameCardIdSet)), sizeof(nn::gc::GameCardIdSet)));
    memcpy(outBuffer, &gcIdSet, sizeof(nn::gc::GameCardIdSet));
    NN_RESULT_SUCCESS;
}

Result WriteToGameCard(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->WriteToGameCardDirectly(offset, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size), static_cast<int64_t>(size)));
    NN_RESULT_SUCCESS;
}

Result SetVerifyWriteEnalbleFlag(bool verifyWriteEnalbleFlag) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->SetVerifyWriteEnalbleFlag(verifyWriteEnalbleFlag));
    NN_RESULT_SUCCESS;
}

Result GetGameCardImageHash(void* buffer, size_t bufferSize, GameCardHandle handle) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardImageHash(nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), static_cast<int64_t>(bufferSize), static_cast<uint32_t>(handle)));
    NN_RESULT_SUCCESS;
}

Result GetGameCardDeviceIdForProdCard(void* buffer, size_t bufferSize, void* cardHeaderForDev, size_t cardHeaderForDevSize) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardDeviceIdForProdCard(nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), static_cast<int64_t>(bufferSize), nn::sf::InBuffer(reinterpret_cast<const char*>(cardHeaderForDev), cardHeaderForDevSize), static_cast<int64_t>(cardHeaderForDevSize)));
    NN_RESULT_SUCCESS;
}

Result EraseAndWriteParamDirectly(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->EraseAndWriteParamDirectly(nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), bufferSize), static_cast<int64_t>(bufferSize)));
    NN_RESULT_SUCCESS;
}

Result ReadParamDirectly(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->ReadParamDirectly(nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), static_cast<int64_t>(bufferSize)));
    NN_RESULT_SUCCESS;
}

Result ForceEraseGameCard() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->ForceEraseGameCard());
    NN_RESULT_SUCCESS;
}

Result GetGameCardErrorInfo(GameCardErrorInfo* pOutGameCardErrorInfo) NN_NOEXCEPT
{
    nn::fs::GameCardErrorInfo gameCardErrorInfo;

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardErrorInfo(nn::sf::Out<nn::fs::GameCardErrorInfo>(&gameCardErrorInfo)));
    *pOutGameCardErrorInfo = gameCardErrorInfo;
    NN_RESULT_SUCCESS;
}

Result GetGameCardErrorReportInfo(GameCardErrorReportInfo* pOutGameCardErrorInfo) NN_NOEXCEPT
{
    nn::fs::GameCardErrorReportInfo gameCardErrorReportInfo;

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardErrorReportInfo(nn::sf::Out<nn::fs::GameCardErrorReportInfo>(&gameCardErrorReportInfo)));
    *pOutGameCardErrorInfo = gameCardErrorReportInfo;
    NN_RESULT_SUCCESS;
}

Result CheckGameCardPartitionAvailability(GameCardHandle handle, GameCardPartition partition) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

    NN_FS_RESULT_DO(fileSystemProxy->OpenGameCardFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>>(&fileSystem), static_cast<uint32_t>(handle), static_cast<uint32_t>(partition)));

    NN_RESULT_SUCCESS;
}

Result GetGameCardDeviceId(void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(size >= sizeof(nn::gc::GcCardDeviceIdSize), ResultInvalidSize());
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    char buffer[nn::gc::GcCardDeviceIdSize];
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetGameCardDeviceId(nn::sf::OutBuffer(buffer, nn::gc::GcCardDeviceIdSize), nn::gc::GcCardDeviceIdSize));
    memcpy(outBuffer, buffer, nn::gc::GcCardDeviceIdSize);
    NN_RESULT_SUCCESS;
}

Result SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode simulatingDetectionMode, bool isWithEvent) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->SimulateDeviceDetectionEvent(static_cast<uint32_t>(fs::SimulatingDeviceType::GameCard), static_cast<uint32_t>(simulatingDetectionMode), isWithEvent));
    NN_RESULT_SUCCESS;
}

Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType deviceEventType) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(SetGameCardSimulationEvent(targetOperationFlag, deviceEventType, ResultSuccess(), false));
    NN_RESULT_SUCCESS;
}

Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, fs::SimulatingDeviceAccessFailureEventType deviceEventType, bool isPersistent) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(SetGameCardSimulationEvent(targetOperationFlag, deviceEventType, ResultSuccess(), isPersistent));
    NN_RESULT_SUCCESS;
}

Result SetGameCardSimulationEvent(fs::SimulatingDeviceTargetOperation targetOperationFlag, nn::Result respondingFailureResult, bool isPersistent) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(SetGameCardSimulationEvent(targetOperationFlag, SimulatingDeviceAccessFailureEventType::AccessFailure, respondingFailureResult, isPersistent));
    NN_RESULT_SUCCESS;
}

Result ClearGameCardSimulationEvent() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;

    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->ClearDeviceSimulationEvent(static_cast<uint32_t>(fs::SimulatingDeviceType::GameCard)));
    NN_RESULT_SUCCESS;
}


}}
