﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fssrv/sf/fssrv_IFile.h>
#include <nn/fssrv/sf/fssrv_IDirectory.h>
#include <nn/fssrv/sf/fssrv_IFileSystem.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fs_QueryRange.h>

#include "../fsa/fs_MountUtility.h"
#include "fs_FileSystemServiceObjectAdapter.h"

namespace nn { namespace fs { namespace detail {

    FileServiceObjectAdapter::FileServiceObjectAdapter(nn::sf::SharedPointer<nn::fssrv::sf::IFile>&& fileServiceObject) NN_NOEXCEPT
      : m_fileServiceObject(fileServiceObject)
    {
    }
    FileServiceObjectAdapter::~FileServiceObjectAdapter() NN_NOEXCEPT
    {
    }
    Result FileServiceObjectAdapter::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT
    {
        nn::fssrv::sf::ReadOption readOption;
        readOption.reserved = option.reserved;
        int64_t readSize = 0;
        NN_RESULT_DO(m_fileServiceObject->Read(nn::sf::Out<int64_t>(&readSize), offset, nn::sf::OutBuffer(static_cast<char*>(buffer), size), static_cast<int64_t>(size), readOption));

        *outValue = static_cast<size_t>(readSize);

        NN_RESULT_SUCCESS;
    }
    Result FileServiceObjectAdapter::DoWrite(int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT
    {
        nn::fssrv::sf::WriteOption writeOption;
        writeOption.flags = option.flags;
        return m_fileServiceObject->Write(offset, nn::sf::InBuffer(static_cast<const char*>(buffer), size), static_cast<int64_t>(size), writeOption);
    }
    Result FileServiceObjectAdapter::DoFlush() NN_NOEXCEPT
    {
        return m_fileServiceObject->Flush();
    }
    Result FileServiceObjectAdapter::DoSetSize(int64_t size) NN_NOEXCEPT
    {
        return m_fileServiceObject->SetSize(size);
    }
    Result FileServiceObjectAdapter::DoGetSize(int64_t* outValue) NN_NOEXCEPT
    {
        return m_fileServiceObject->GetSize(nn::sf::Out<int64_t>(outValue));
    }
    Result FileServiceObjectAdapter::DoOperateRange(
        void* outBuffer, size_t outBufferSize, OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT
    {
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);

        if( operationId == OperationId::QueryRange )
        {
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(FileQueryRangeInfo), ResultInvalidSize());

            return m_fileServiceObject->OperateRange(
                nn::sf::Out<FileQueryRangeInfo>(reinterpret_cast<FileQueryRangeInfo*>(outBuffer)),
                static_cast<int32_t>(operationId),
                offset,
                size);
        }

        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }

    DirectoryServiceObjectAdapter::DirectoryServiceObjectAdapter(nn::sf::SharedPointer<nn::fssrv::sf::IDirectory>&& directoryServiceObject) NN_NOEXCEPT
      : m_directoryServiceObject(directoryServiceObject)
    {
    }
    DirectoryServiceObjectAdapter::~DirectoryServiceObjectAdapter() NN_NOEXCEPT
    {
    }
    Result DirectoryServiceObjectAdapter::DoRead(int64_t* outValue, DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT
    {
        return m_directoryServiceObject->Read(nn::sf::Out<int64_t>(outValue), nn::sf::OutBuffer(reinterpret_cast<char*>(outEntries), sizeof(nn::fs::DirectoryEntry) * static_cast<size_t>(count)));
    }
    Result DirectoryServiceObjectAdapter::DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT
    {
        return m_directoryServiceObject->GetEntryCount(nn::sf::Out<int64_t>(outValue));
    }

namespace {
    Result GetPathForServiceObject(nn::fssrv::sf::Path* outSfPath, const char* path)
    {
        std::strncpy(outSfPath->str, path, sizeof(outSfPath->str) - 1);
        outSfPath->str[sizeof(outSfPath->str) - 1] = '\0';
        Replace(outSfPath->str, sizeof(outSfPath->str) - 1, '\\', '/');

        const auto mountNameLength = detail::IsWindowsDrive(outSfPath->str) ? 2 : 0;
        const auto pBody = outSfPath->str + mountNameLength;
        const auto bodyLengthMax = EntryNameLengthMax - mountNameLength;
        return VerifyPath(pBody, bodyLengthMax, bodyLengthMax);
    }
}

    FileSystemServiceObjectAdapter::FileSystemServiceObjectAdapter(nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>&& fileSystemServiceObject) NN_NOEXCEPT
      : m_fileSystemServiceObject(fileSystemServiceObject)
    {
    }
    FileSystemServiceObjectAdapter::~FileSystemServiceObjectAdapter() NN_NOEXCEPT
    {
    }
    Result FileSystemServiceObjectAdapter::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->CreateFile(sfPath, size, option);
    }
    Result FileSystemServiceObjectAdapter::DoDeleteFile(const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->DeleteFile(sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoCreateDirectory(const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->CreateDirectory(sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoDeleteDirectory(const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->DeleteDirectory(sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->DeleteDirectoryRecursively(sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->CleanDirectoryRecursively(sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path currentSfPath;
        nn::fssrv::sf::Path newSfPath;
        NN_RESULT_DO(GetPathForServiceObject(&currentSfPath, currentPath));
        NN_RESULT_DO(GetPathForServiceObject(&newSfPath, newPath));
        return m_fileSystemServiceObject->RenameFile(currentSfPath, newSfPath);
    }
    Result FileSystemServiceObjectAdapter::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path currentSfPath;
        nn::fssrv::sf::Path newSfPath;
        NN_RESULT_DO(GetPathForServiceObject(&currentSfPath, currentPath));
        NN_RESULT_DO(GetPathForServiceObject(&newSfPath, newPath));
        return m_fileSystemServiceObject->RenameDirectory(currentSfPath, newSfPath);
    }
    Result FileSystemServiceObjectAdapter::DoGetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->GetEntryType(nn::sf::Out<uint32_t>(reinterpret_cast<uint32_t*>(outValue)), sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->GetFreeSpaceSize(nn::sf::Out<int64_t>(reinterpret_cast<int64_t*>(outValue)), sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->GetTotalSpaceSize(nn::sf::Out<int64_t>(reinterpret_cast<int64_t*>(outValue)), sfPath);
    }
    Result FileSystemServiceObjectAdapter::DoOpenFile(std::unique_ptr<fsa::IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFile> fileServiceObject;
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        NN_RESULT_DO(m_fileSystemServiceObject->OpenFile(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFile>>(&fileServiceObject), sfPath, static_cast<uint32_t>(mode)));
        outValue->reset(new FileServiceObjectAdapter(std::move(fileServiceObject)));
        NN_RESULT_THROW_UNLESS(outValue->get() != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_SUCCESS;
    }
    Result FileSystemServiceObjectAdapter::DoOpenDirectory(std::unique_ptr<fsa::IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IDirectory> directoryServiceObject;
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        NN_RESULT_DO(m_fileSystemServiceObject->OpenDirectory(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDirectory>>(&directoryServiceObject), sfPath, static_cast<uint32_t>(mode)));
        outValue->reset(new DirectoryServiceObjectAdapter(std::move(directoryServiceObject)));
        NN_RESULT_THROW_UNLESS(outValue->get() != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemServiceObjectAdapter::DoCommit() NN_NOEXCEPT
    {
        return m_fileSystemServiceObject->Commit();
    }

    Result FileSystemServiceObjectAdapter::DoGetFileTimeStampRaw(FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->GetFileTimeStampRaw(nn::sf::Out<FileTimeStampRaw>(outTimeStamp), sfPath);
    }

    Result FileSystemServiceObjectAdapter::DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT
    {
        nn::fssrv::sf::Path sfPath;
        NN_RESULT_DO(GetPathForServiceObject(&sfPath, path));
        return m_fileSystemServiceObject->QueryEntry(nn::sf::OutBuffer(static_cast<char*>(outBuffer), outBufferSize), nn::sf::InBuffer(static_cast<const char*>(inBuffer), inBufferSize), static_cast<int32_t>(queryId), sfPath);
    }

}}}
