﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "fs_FileSystemProxyServiceObject.h"

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/fssrv/detail/fssrv_FileSystemProxyServiceObject.h>
#else
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/detail/fs_ServerName.h>
#include <nn/fs/detail/fs_DfcFileSystemProxyServiceObject.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>

#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#endif

#include "fs_Library.h"

namespace nn { namespace fs {
    PriorityRaw GetPriorityRawOnCurrentThreadImpl() NN_NOEXCEPT;
}}

namespace nn { namespace fs {

namespace detail {

    namespace {
#if !defined(NN_BUILD_CONFIG_OS_WIN32)
        bool g_UseStaticBufferForFileSystemProxyServiceObject = false;
        std::aligned_storage<128>::type g_StaticBufferForFileSystemProxyServiceObject;

        bool g_UseStaticBufferForFileSystemProxyForLoaderServiceObject = false;
        std::aligned_storage<128>::type g_StaticBufferForFileSystemProxyForLoaderServiceObject;

        bool g_UseStaticBufferForProgramRegistryServiceObject = false;
        std::aligned_storage<128>::type g_StaticBufferForProgramRegistryServiceObject;

        void SetUseStaticBufferForFileSystemProxyServiceObject(bool value)
        {
            g_UseStaticBufferForFileSystemProxyServiceObject = value;
        }

        void SetUseStaticBufferForFileSystemProxyForLoaderServiceObject(bool value)
        {
            g_UseStaticBufferForFileSystemProxyForLoaderServiceObject = value;
        }

        void SetUseStaticBufferForProgramRegistryServiceObject(bool value)
        {
            g_UseStaticBufferForProgramRegistryServiceObject = value;
        }

        class HipcClientAllocator
        {
        public:
            HipcClientAllocator() NN_NOEXCEPT
            {
            }

            typedef nn::sf::impl::StatelessAllocationPolicy<HipcClientAllocator> Policy;

            void* Allocate(size_t size) NN_NOEXCEPT
            {
                if (g_UseStaticBufferForFileSystemProxyServiceObject)
                {
                    NN_FS_ABORT_UNLESS_WITH_RESULT(size <= sizeof(g_StaticBufferForFileSystemProxyServiceObject), ResultAllocationMemoryFailedInFileSystemProxyServiceObjectA());
                    return &g_StaticBufferForFileSystemProxyServiceObject;
                }
                if (g_UseStaticBufferForFileSystemProxyForLoaderServiceObject)
                {
                    NN_FS_ABORT_UNLESS_WITH_RESULT(size <= sizeof(g_StaticBufferForFileSystemProxyForLoaderServiceObject), ResultAllocationMemoryFailedInFileSystemProxyServiceObjectB());
                    return &g_StaticBufferForFileSystemProxyForLoaderServiceObject;
                }
                if (g_UseStaticBufferForProgramRegistryServiceObject)
                {
                    NN_FS_ABORT_UNLESS_WITH_RESULT(size <= sizeof(g_StaticBufferForProgramRegistryServiceObject), ResultAllocationMemoryFailedInFileSystemProxyServiceObjectC());
                    return &g_StaticBufferForProgramRegistryServiceObject;
                }
                return detail::Allocate(size);
            }

            void Deallocate(void* p, size_t size) NN_NOEXCEPT
            {
                if (p == &g_StaticBufferForFileSystemProxyServiceObject)
                {
                    return;
                }
                if (p == &g_StaticBufferForFileSystemProxyForLoaderServiceObject)
                {
                    return;
                }
                if (p == &g_StaticBufferForProgramRegistryServiceObject)
                {
                    return;
                }
                detail::Deallocate(p, size);
            }
        };

        enum class FileSystemProxySessionSetting
        {
            SystemNormal,
            Application,
            SystemMulti
        };

        FileSystemProxySessionSetting g_FileSystemProxySessionSetting = FileSystemProxySessionSetting::SystemNormal;
        bool g_IsFileSystemProxyObjectInitialized = false;

        class FileSystemProxySessionResourceManager
        {
        public:
            static const int RealtimeSessionCount = 1;
            static const int NonRealtimeSessionCount = 2;
            static const int SessionCount = RealtimeSessionCount + NonRealtimeSessionCount;

        public:
            FileSystemProxySessionResourceManager() NN_NOEXCEPT
                : m_IsEnabled(false),
                  m_IsSessionUsedByRealtimeWaiting(false),
                  m_IsSessionUsedByNonRealtimeWaiting(false),
                  m_SessionUsedByRealtimeCount(0),
                  m_SessionUsedByNonRealtimeCount(0)
            {
            }

        public:
            bool OnAcquire() NN_NOEXCEPT
            {
                if( !m_IsEnabled )
                {
                    return m_DefaultSessionResourceManager.OnAcquire();
                }

                if( GetPriorityRawOnCurrentThreadImpl() == nn::fs::PriorityRaw_Realtime )
                {
                    if( m_SessionUsedByRealtimeCount >= RealtimeSessionCount )
                    {
                        m_IsSessionUsedByRealtimeWaiting = true;
                        return false;
                    }
                    ++m_SessionUsedByRealtimeCount;
                }
                else
                {
                    if( m_SessionUsedByNonRealtimeCount >= NonRealtimeSessionCount )
                    {
                        m_IsSessionUsedByNonRealtimeWaiting = true;
                        return false;
                    }
                    ++m_SessionUsedByNonRealtimeCount;
                }

                return true;
            }

            bool OnRelease() NN_NOEXCEPT
            {
                if( !m_IsEnabled )
                {
                    return m_DefaultSessionResourceManager.OnRelease();
                }

                bool isBroadcastRequired = false;

                if( GetPriorityRawOnCurrentThreadImpl() == nn::fs::PriorityRaw_Realtime )
                {
                    NN_SDK_REQUIRES_LESS(0, m_SessionUsedByRealtimeCount);
                    if( m_IsSessionUsedByRealtimeWaiting )
                    {
                        isBroadcastRequired = m_SessionUsedByRealtimeCount == RealtimeSessionCount;
                        m_IsSessionUsedByRealtimeWaiting = false;
                    }
                    --m_SessionUsedByRealtimeCount;
                }
                else
                {
                    NN_SDK_REQUIRES_LESS(0, m_SessionUsedByNonRealtimeCount);
                    if( m_IsSessionUsedByNonRealtimeWaiting )
                    {
                        isBroadcastRequired = m_SessionUsedByNonRealtimeCount == NonRealtimeSessionCount;
                        m_IsSessionUsedByNonRealtimeWaiting = false;
                    }
                    --m_SessionUsedByNonRealtimeCount;
                }

                return isBroadcastRequired;
            }

        public:
            void SetEnabled(bool isEnabled) NN_NOEXCEPT
            {
                m_IsEnabled = isEnabled;
            }

        private:
            bool m_IsEnabled;
            bool m_IsSessionUsedByRealtimeWaiting;
            bool m_IsSessionUsedByNonRealtimeWaiting;
            int m_SessionUsedByRealtimeCount;
            int m_SessionUsedByNonRealtimeCount;
            sf::hipc::client::DefaultSessionResourceManager m_DefaultSessionResourceManager;
        };

        class FileSystemProxyClientSessionManager
            : public sf::HipcCustomClientSessionManager<FileSystemProxySessionResourceManager>
        {
        public:
            FileSystemProxyClientSessionManager() NN_NOEXCEPT
                : sf::HipcCustomClientSessionManager<FileSystemProxySessionResourceManager>(&m_SessionResourceManager)
            {
            }

        public:
            class DebugApi
            {
            public:
                explicit DebugApi(FileSystemProxyClientSessionManager* pSessionManager) NN_NOEXCEPT
                    : m_pSessionManager(pSessionManager)
                {
                    NN_ABORT_UNLESS_NOT_NULL(pSessionManager);
                }

            public:
                void DisableSessionForRealtimeOnlyAndChangeSessionCount(int sessionCount) const NN_NOEXCEPT
                {
                    // m_pSessionManager が nullptr でないことはコンストラクタで保証
                    m_pSessionManager->m_SessionResourceManager.SetEnabled(false);
                    m_pSessionManager->SetSessionCount(sessionCount);
                }

                void EnableSessionForRealtimeOnly() const NN_NOEXCEPT
                {
                    // m_pSessionManager が nullptr でないことはコンストラクタで保証
                    m_pSessionManager->m_SessionResourceManager.SetEnabled(false);
                    m_pSessionManager->SetSessionCount(FileSystemProxySessionResourceManager::SessionCount);
                    m_pSessionManager->m_SessionResourceManager.SetEnabled(true);
                }

            private:
                FileSystemProxyClientSessionManager* const m_pSessionManager;
            };

        public:
            void Initialize(nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy>* outValue, FileSystemProxySessionSetting sessionType) NN_NOEXCEPT
            {
                auto result = InitializeByName<nn::fssrv::sf::IFileSystemProxy, HipcClientAllocator::Policy>(outValue, nn::fs::detail::FileSystemProxyServiceName);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                switch( sessionType )
                {
                case FileSystemProxySessionSetting::Application:
                    {
                        SetSessionCount(FileSystemProxySessionResourceManager::SessionCount);
                        m_SessionResourceManager.SetEnabled(true);
                    }
                    break;

                case FileSystemProxySessionSetting::SystemNormal:
                case FileSystemProxySessionSetting::SystemMulti:
                    {
                        const int sessionCount = (sessionType == FileSystemProxySessionSetting::SystemMulti ? 2 : 1);
                        SetSessionCount(sessionCount);
                        m_SessionResourceManager.SetEnabled(false);
                    }
                    break;

                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }

            const DebugApi GetDebugApi() NN_NOEXCEPT
            {
                return DebugApi(this);
            }

        private:
            FileSystemProxySessionResourceManager m_SessionResourceManager;
        };

        FileSystemProxyClientSessionManager& GetClientManagerForFileSystemProxy() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(FileSystemProxyClientSessionManager, s_ClientManagerForFileSystemProxy);
            return s_ClientManagerForFileSystemProxy;
        }

        nn::sf::HipcSimpleClientSessionManager& GetClientManagerForFileSystemProxyForLoader() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(nn::sf::HipcSimpleClientSessionManager, s_ClientManagerForFileSystemProxyForLoader);
            return s_ClientManagerForFileSystemProxyForLoader;
        }

        nn::sf::HipcSimpleClientSessionManager& GetClientManagerForProgramRegistry() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(nn::sf::HipcSimpleClientSessionManager, s_ClientManagerForProgramRegistry);
            return s_ClientManagerForProgramRegistry;
        }
#endif

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> g_DfcFileSystemProxyServiceObject;

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> GetFileSystemProxyServiceObjectImpl() NN_NOEXCEPT
        {
            InitializeFileSystemLibrary();

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxyServiceObject;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            fileSystemProxyServiceObject = nn::fssrv::detail::GetFileSystemProxyServiceObject();
            NN_ABORT_UNLESS(fileSystemProxyServiceObject != nullptr);
            fileSystemProxyServiceObject->SetCurrentProcess(0);
#else
            if (g_DfcFileSystemProxyServiceObject != nullptr)
            {
                fileSystemProxyServiceObject = g_DfcFileSystemProxyServiceObject;
            }
            else
            {
                SetUseStaticBufferForFileSystemProxyServiceObject(true);

                GetClientManagerForFileSystemProxy().Initialize(&fileSystemProxyServiceObject, g_FileSystemProxySessionSetting);
                g_IsFileSystemProxyObjectInitialized = true;

                SetUseStaticBufferForFileSystemProxyServiceObject(false);
                fileSystemProxyServiceObject->SetCurrentProcess(0);
            }
#endif
            return fileSystemProxyServiceObject;
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxyForLoader> GetFileSystemProxyForLoaderServiceObjectImpl() NN_NOEXCEPT
        {
            InitializeFileSystemLibrary();

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxyForLoader> fileSystemProxyForLoaderServiceObject;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            fileSystemProxyForLoaderServiceObject = nn::fssrv::detail::GetFileSystemProxyForLoaderServiceObject();
            NN_ABORT_UNLESS(fileSystemProxyForLoaderServiceObject != nullptr);
#else
            SetUseStaticBufferForFileSystemProxyForLoaderServiceObject(true);

            auto result = GetClientManagerForFileSystemProxyForLoader().InitializeByName<nn::fssrv::sf::IFileSystemProxyForLoader, HipcClientAllocator::Policy>(&fileSystemProxyForLoaderServiceObject, nn::fs::detail::FileSystemProxyForLoaderServiceName);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            const int SessionCount = 1;
            GetClientManagerForFileSystemProxyForLoader().SetSessionCount(SessionCount);

            SetUseStaticBufferForFileSystemProxyForLoaderServiceObject(false);
#endif
            return fileSystemProxyForLoaderServiceObject;
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IProgramRegistry> GetProgramRegistryServiceObjectImpl() NN_NOEXCEPT
        {
            InitializeFileSystemLibrary();

            nn::sf::SharedPointer<nn::fssrv::sf::IProgramRegistry> programRegistryServiceObject;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            programRegistryServiceObject = nn::fssrv::detail::GetProgramRegistryServiceObject();
            NN_ABORT_UNLESS(programRegistryServiceObject != nullptr);
#else
            SetUseStaticBufferForProgramRegistryServiceObject(true);

            auto result = GetClientManagerForProgramRegistry().InitializeByName<nn::fssrv::sf::IProgramRegistry, HipcClientAllocator::Policy>(&programRegistryServiceObject, nn::fs::detail::ProgramRegistryServiceName);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            const int SessionCount = 1;
            GetClientManagerForProgramRegistry().SetSessionCount(SessionCount);

            SetUseStaticBufferForProgramRegistryServiceObject(false);
#endif
            return programRegistryServiceObject;
        }
    }

    void InitializeDfcFileSystemProxyServiceObject(nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> pFileSystemProxyServiceObject) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_DfcFileSystemProxyServiceObject == nullptr);
        g_DfcFileSystemProxyServiceObject = pFileSystemProxyServiceObject;
    }

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> GetFileSystemProxyServiceObject() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy>, s_FileSystemProxyServiceObject, = GetFileSystemProxyServiceObjectImpl());
        return s_FileSystemProxyServiceObject;
    }

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxyForLoader> GetFileSystemProxyForLoaderServiceObject() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxyForLoader>, s_FileSystemProxyForLoaderServiceObject, = GetFileSystemProxyForLoaderServiceObjectImpl());
        return s_FileSystemProxyForLoaderServiceObject;
    }

    nn::sf::SharedPointer<nn::fssrv::sf::IProgramRegistry> GetProgramRegistryServiceObject() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<nn::fssrv::sf::IProgramRegistry>, s_ProgramRegistryServiceObject, = GetProgramRegistryServiceObjectImpl());
        return s_ProgramRegistryServiceObject;
    }

    bool IsSessionForRealtimeOnlyEnabled() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        return false;
#else
        return g_FileSystemProxySessionSetting == FileSystemProxySessionSetting::Application;
#endif
    }

    void DisableSessionForRealtimeOnlyAndChangeSessionCount(int sessionCount) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(sessionCount);
#else
        NN_ABORT_UNLESS(detail::g_IsFileSystemProxyObjectInitialized);
        GetClientManagerForFileSystemProxy()
            .GetDebugApi()
            .DisableSessionForRealtimeOnlyAndChangeSessionCount(sessionCount);
#endif
    }

    void EnableSessionForRealtimeOnly() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS(detail::g_IsFileSystemProxyObjectInitialized);
        GetClientManagerForFileSystemProxy().GetDebugApi().EnableSessionForRealtimeOnly();
#endif
    }

}

    void InitializeForApplication() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS(!detail::g_IsFileSystemProxyObjectInitialized);
        NN_ABORT_UNLESS_EQUAL(detail::g_FileSystemProxySessionSetting, detail::FileSystemProxySessionSetting::SystemNormal);
        detail::g_FileSystemProxySessionSetting = detail::FileSystemProxySessionSetting::Application;
#endif
    }

    void InitializeWithMultiSessionForSystem() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS(!detail::g_IsFileSystemProxyObjectInitialized);
        NN_ABORT_UNLESS_EQUAL(detail::g_FileSystemProxySessionSetting, detail::FileSystemProxySessionSetting::SystemNormal);
        detail::g_FileSystemProxySessionSetting = detail::FileSystemProxySessionSetting::SystemMulti;
#endif
    }

    void InitializeWithMultiSessionForTargetTool() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        if( detail::g_FileSystemProxySessionSetting == detail::FileSystemProxySessionSetting::Application )
        {
            NN_DETAIL_FS_INFO("nn::fs::InitializeForApplication() was already called.\n");
            return;
        }
        NN_ABORT_UNLESS(!detail::g_IsFileSystemProxyObjectInitialized);
        NN_ABORT_UNLESS_EQUAL(detail::g_FileSystemProxySessionSetting, detail::FileSystemProxySessionSetting::SystemNormal);
        detail::g_FileSystemProxySessionSetting = detail::FileSystemProxySessionSetting::SystemMulti;
#endif
    }

}}
