﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fssrv/sf/fssrv_IFileSystem.h>

#include "fs_FileSystemAccessor.h"
#include "shim/fs_FileDataCacheAccessor.h"
#include "shim/fs_FileSystemServiceObjectAdapter.h"
#include "shim/fs_PathBasedFileDataCache.h"

namespace nn { namespace fs { namespace detail {
    namespace
    {
        template<class ListType, class EntryType>
        void Remove(ListType* list, const EntryType* entry)
        {
            for (auto iter = list->cbegin(); iter != list->cend(); iter++)
            {
                if (iter.operator->() == entry)
                {
                    list->erase(iter);
                    return;
                }
            }

            NN_SDK_ASSERT(false, "Invalid file or directory object.");
        }

        Result CheckPath(const char* mountName, const char* path) NN_NOEXCEPT
        {
            const auto mountNameLength = strnlen(mountName, MountNameLengthMax);
            const auto pathLength = strnlen(path, EntryNameLengthMax);
            NN_RESULT_THROW_UNLESS(
                mountNameLength + 1 + pathLength <= EntryNameLengthMax, // mountName + ":" + path
                ResultTooLongPath());

            NN_RESULT_SUCCESS;
        }
    }

    FileSystemAccessor::FileSystemAccessor(const char* name, std::unique_ptr<fsa::IFileSystem>&& fileSystem, std::unique_ptr<fsa::ICommonMountNameGenerator>&& mountNameGenerator) NN_NOEXCEPT
        : m_Impl(std::move(fileSystem))
        , m_OpenListLock(false)
        , m_pMountNameGenerator(std::move(mountNameGenerator))
        , m_IsEnabledAccessLog(false)
        , m_IsFileDataCacheAttachable(false)
        , m_IsPathBasedFileDataCacheAttachable(false)
        , m_IsAttachedPathBasedFileDataCache(false)
        , m_IsMultiCommitSupported(false)
    {
        NN_FS_ABORT_UNLESS_WITH_RESULT(name[0] != '\0', ResultInvalidMountName(), "Mount name cannot be empty.\n");
        NN_FS_ABORT_UNLESS_WITH_RESULT(strnlen(name, MountName::LENGTH) < MountName::LENGTH, ResultInvalidMountName(), "Mount name \"%s\" is too long.\n", name);
        strncpy(m_Name.string, name, MountName::LENGTH - 1);
        m_Name.string[MountName::LENGTH - 1] = '\0';
    }

    FileSystemAccessor::~FileSystemAccessor() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);

        DumpUnclosedAccessorList(
            nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend,
            nn::fs::OpenDirectoryMode_All
        );

        NN_FS_ABORT_UNLESS_WITH_RESULT(m_OpenFileList.empty(), ResultFileNotClosed(), "Close opened files before Unmounting.\n");
        NN_FS_ABORT_UNLESS_WITH_RESULT(m_OpenDirectoryList.empty(), ResultDirectoryNotClosed(), "Close opened directories before Unmounting.\n");

        if (m_IsAttachedPathBasedFileDataCache)
        {
            InvalidatePathBasedFileDataCacheEntries(this);
        }
    }

    Result FileSystemAccessor::CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        if (m_IsAttachedPathBasedFileDataCache)
        {
            auto lock = LockPathBasedFileDataCacheEntry(this, path);
            NN_RESULT_DO(m_Impl->CreateFile(path, size, option));
            InvalidatePathBasedFileDataCacheEntry(std::move(lock), this, path);
        }
        else
        {
            NN_RESULT_DO(m_Impl->CreateFile(path, size, option));
        }
        NN_RESULT_SUCCESS;
    }

    Result FileSystemAccessor::DeleteFile(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->DeleteFile(path);
    }

    Result FileSystemAccessor::CreateDirectory(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->CreateDirectory(path);
    }

    Result FileSystemAccessor::DeleteDirectory(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->DeleteDirectory(path);
    }

    Result FileSystemAccessor::DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->DeleteDirectoryRecursively(path);
    }

    Result FileSystemAccessor::CleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->CleanDirectoryRecursively(path);
    }

    Result FileSystemAccessor::RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, currentPath));
        NN_RESULT_DO(CheckPath(m_Name.string, newPath));
        if (m_IsAttachedPathBasedFileDataCache)
        {
            auto lock = LockPathBasedFileDataCacheEntry(this, newPath);
            NN_RESULT_DO(m_Impl->RenameFile(currentPath, newPath));
            InvalidatePathBasedFileDataCacheEntry(std::move(lock), this, newPath);
        }
        else
        {
            NN_RESULT_DO(m_Impl->RenameFile(currentPath, newPath));
        }
        NN_RESULT_SUCCESS;
    }

    Result FileSystemAccessor::RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, currentPath));
        NN_RESULT_DO(CheckPath(m_Name.string, newPath));
        if (m_IsAttachedPathBasedFileDataCache)
        {
            auto lock = LockPathBasedFileDataCacheEntries();
            NN_RESULT_DO(m_Impl->RenameDirectory(currentPath, newPath));
            InvalidatePathBasedFileDataCacheEntries(std::move(lock), this);
        }
        else
        {
            NN_RESULT_DO(m_Impl->RenameDirectory(currentPath, newPath));
        }
        NN_RESULT_SUCCESS;
    }

    Result FileSystemAccessor::GetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->GetEntryType(outValue, path);
    }

    Result FileSystemAccessor::GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->GetFreeSpaceSize(outValue, path);
    }

    Result FileSystemAccessor::GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));
        return m_Impl->GetTotalSpaceSize(outValue, path);
    }

    Result FileSystemAccessor::OpenFile(std::unique_ptr<FileAccessor>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));

        std::unique_ptr<fsa::IFile> file;
        NN_RESULT_DO(m_Impl->OpenFile(&file, path, mode));
        auto accessor = new FileAccessor(std::move(file), this, mode);
        NN_RESULT_THROW_UNLESS(accessor, ResultAllocationMemoryFailedInFileSystemAccessorA());

        {
            std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);
            m_OpenFileList.push_back(*accessor);
        }

        if (m_IsAttachedPathBasedFileDataCache)
        {
            if (mode & nn::fs::OpenMode_AllowAppend)
            {
                InvalidatePathBasedFileDataCacheEntry(this, path);
            }
            else
            {
                std::unique_ptr<FilePathHash> pHash(new FilePathHash());
                NN_RESULT_THROW_UNLESS(pHash, ResultAllocationMemoryFailedInFileSystemAccessorA());
                int hashIndex = 0;
                if (FindPathBasedFileDataCacheEntry(pHash.get(), &hashIndex, this, path))
                {
                    accessor->SetFilePathHash(std::move(pHash), hashIndex);
                }
            }
        }

        outValue->reset(accessor);

        NN_RESULT_SUCCESS;
    }

    Result FileSystemAccessor::OpenDirectory(std::unique_ptr<DirectoryAccessor>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
    {
        NN_RESULT_DO(CheckPath(m_Name.string, path));

        std::unique_ptr<fsa::IDirectory> directory;
        NN_RESULT_DO(m_Impl->OpenDirectory(&directory, path, mode));
        auto accessor = new DirectoryAccessor(std::move(directory), *this);
        NN_RESULT_THROW_UNLESS(accessor, ResultAllocationMemoryFailedInFileSystemAccessorB());

        {
            std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);
            m_OpenDirectoryList.push_back(*accessor);
        }

        outValue->reset(accessor);

        NN_RESULT_SUCCESS;
    }

    Result FileSystemAccessor::Commit() NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);

            DumpUnclosedAccessorList(nn::fs::OpenMode_Write, 0);

            NN_FS_ABORT_UNLESS_WITH_RESULT(
                std::none_of(
                    m_OpenFileList.begin(),
                    m_OpenFileList.end(),
                    [](const FileAccessor &file) NN_NOEXCEPT
                    {
                        return (file.GetOpenMode() & nn::fs::OpenMode_Write) != 0;
                    }),
                ResultWriteModeFileNotClosed(),
                "Close files opened in write mode before committing.\n");
        }

        return m_Impl->Commit();
    }

    Result FileSystemAccessor::GetFileTimeStampRaw(FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        return m_Impl->GetFileTimeStampRaw(outTimeStamp, path);
    }

    Result FileSystemAccessor::QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fsa::QueryId queryId, const char* path) NN_NOEXCEPT
    {
        return m_Impl->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path);
    }

    void FileSystemAccessor::PurgeFileDataCache(FileDataCacheAccessor* pFileDataCacheAccessor) NN_NOEXCEPT
    {
        pFileDataCacheAccessor->Purge(m_Impl.get());
    }

    void FileSystemAccessor::NotifyCloseFile(FileAccessor* file) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);
        Remove(&m_OpenFileList, file);
    }

    void FileSystemAccessor::NotifyCloseDirectory(DirectoryAccessor* directory) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_OpenListLock);
        Remove(&m_OpenDirectoryList, directory);
    }

    Result FileSystemAccessor::GetCommonMountName(char* name, size_t nameSize) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pMountNameGenerator != nullptr, ResultPreconditionViolation());
        return m_pMountNameGenerator->GenerateCommonMountName(name, nameSize);
    }

    void FileSystemAccessor::DumpUnclosedAccessorList(int fileOpenMode, int directoryOpenMode) NN_NOEXCEPT
    {
        if (fileOpenMode != 0)
        {
            const auto unclosedFileCount
                = std::count_if(
                      m_OpenFileList.begin(),
                      m_OpenFileList.end(),
                      [fileOpenMode](const FileAccessor &file) NN_NOEXCEPT
                      {
                          return (file.GetOpenMode() & fileOpenMode) != 0;
                      }
                  );
            if( 0 < unclosedFileCount )
            {
                NN_DETAIL_FS_ERROR("Error: File not closed (count: %d)\n", unclosedFileCount);
                for( auto iter = m_OpenFileList.begin(); iter != m_OpenFileList.end(); ++iter )
                {
                    if( (iter->GetOpenMode() & fileOpenMode) == 0 )
                    {
                        continue;
                    }
                    int64_t fileSize = 0;
                    const auto result = iter->GetSize(&fileSize);
                    if( result.IsFailure() )
                    {
                        fileSize = -1;
                    }
                    NN_DETAIL_FS_ERROR(
                        "     {%s} FileHandle: 0x%p, Mode: %d, Size: %lld, WriteState: %d\n",
                        GetName(),
                        &(*iter),
                        iter->GetOpenMode(),
                        fileSize,
                        iter->GetWriteState());
                }
            }
        }

        if (directoryOpenMode != 0 && !m_OpenDirectoryList.empty())
        {
            NN_DETAIL_FS_ERROR(
                "Error: Directory not closed (count: %d)\n", m_OpenDirectoryList.size());
            for (auto iter = m_OpenDirectoryList.begin(); iter != m_OpenDirectoryList.end(); ++iter)
            {
                int64_t entryCount = 0;
                const auto result = iter->GetEntryCount(&entryCount);
                if (result.IsFailure())
                {
                    entryCount = -1;
                }
                NN_DETAIL_FS_ERROR(
                    "     {%s} DirectoryHandle: 0x%p, EntryCount: %lld\n",
                    GetName(),
                    &(*iter),
                    entryCount);
            }
        }
    }

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> FileSystemAccessor::GetMultiCommitTarget() NN_NOEXCEPT
    {
        if( m_IsMultiCommitSupported )
        {
            auto pAdapter = static_cast<detail::FileSystemServiceObjectAdapter *>(m_Impl.get());
            return pAdapter->GetFileSystem();
        }
        return nullptr;
    }

}}}
