﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_QueryRange.h>

namespace nn { namespace fs {

NN_DEFINE_STATIC_CONSTANT(const int64_t FileStorage::SizeNotInitialized);

FileStorage::FileStorage() NN_NOEXCEPT
    : m_pFile(nullptr)
    , m_Size(SizeNotInitialized)
{
}

FileStorage::FileStorage(nn::fs::fsa::IFile* pBaseFile) NN_NOEXCEPT
    : m_pFile(pBaseFile)
    , m_Size(SizeNotInitialized)
{
}


FileStorage::FileStorage(std::shared_ptr<nn::fs::fsa::IFile> pSharedBaseFile) NN_NOEXCEPT
    : m_pFile(pSharedBaseFile.get())
    , m_pSharedFile(std::move(pSharedBaseFile))
    , m_Size(SizeNotInitialized)
{
}


Result FileStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    NN_RESULT_DO(UpdateSize());
    if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
    {
        return nn::fs::ResultOutOfRange();
    }

    size_t readSize;
    return m_pFile->Read(&readSize, offset, buffer, size, nn::fs::ReadOption());
}

Result FileStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    NN_RESULT_DO(UpdateSize());
    if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
    {
        return nn::fs::ResultOutOfRange();
    }

    return m_pFile->Write(offset, buffer, size, nn::fs::WriteOption());
}

Result FileStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    m_Size = SizeNotInitialized;
    return m_pFile->SetSize(size);
}

Result FileStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(UpdateSize());
    *outValue = m_Size;
    NN_RESULT_SUCCESS;
}

Result FileStorage::Flush() NN_NOEXCEPT
{
    return m_pFile->Flush();
}

Result FileStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(offset);
    NN_UNUSED(size);

    switch( operationId )
    {
    case OperationId::Invalidate:
    case OperationId::QueryRange:
        {
            if( size == 0 )
            {
                if( operationId == OperationId::QueryRange )
                {
                    NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
                    NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(QueryRangeInfo), nn::fs::ResultInvalidSize());
                    reinterpret_cast<QueryRangeInfo*>(outBuffer)->Clear();
                }
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_DO(UpdateSize());
            if( !IStorage::CheckOffsetAndSize(offset, size) )
            {
                return nn::fs::ResultOutOfRange();
            }
            return m_pFile->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize);
        }
    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

Result FileStorage::UpdateSize() NN_NOEXCEPT
{
    if (m_Size == SizeNotInitialized)
    {
        int64_t size;
        NN_RESULT_DO(m_pFile->GetSize(&size));
        m_Size = size;
    }
    NN_RESULT_SUCCESS;
}

FileStorageBasedFileSystem::FileStorageBasedFileSystem() NN_NOEXCEPT
{
}

Result FileStorageBasedFileSystem::Initialize(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    std::unique_ptr<fs::fsa::IFile> pBaseFile;
    NN_RESULT_DO(pBaseFileSystem->OpenFile(&pBaseFile, path, mode));

    SetFile(pBaseFile.get());
    m_BaseFileSystem = std::move(pBaseFileSystem);
    m_BaseFile = std::move(pBaseFile);

    NN_RESULT_SUCCESS;
}

FileHandleStorage::FileHandleStorage(nn::fs::FileHandle handle) NN_NOEXCEPT
    : m_Handle(handle)
    , m_Size(0)
    , m_AccessMutex(false)
{
    auto result = GetFileSize(&m_Size, m_Handle);
    NN_FS_ABORT_UNLESS_WITH_RESULT(result.IsSuccess(), ResultInvalidArgument(), "handle must be valid.");
}

Result FileHandleStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_AccessMutex);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
    {
        return nn::fs::ResultOutOfRange();
    }

    return ReadFile(m_Handle, offset, buffer, size, nn::fs::ReadOption());
}

Result FileHandleStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_AccessMutex);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
    {
        return nn::fs::ResultOutOfRange();
    }

    return WriteFile(m_Handle, offset, buffer, size, nn::fs::WriteOption());
}

Result FileHandleStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    *outValue = m_Size;
    NN_RESULT_SUCCESS;
}

Result FileHandleStorage::Flush() NN_NOEXCEPT
{
    return FlushFile(m_Handle);
}

Result FileHandleStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(inBuffer);
    NN_UNUSED(inBufferSize);

    NN_RESULT_THROW_UNLESS(operationId == OperationId::QueryRange, nn::fs::ResultUnsupportedOperation());

    NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(QueryRangeInfo), nn::fs::ResultInvalidSize());

    NN_RESULT_DO(QueryRange(reinterpret_cast<QueryRangeInfo*>(outBuffer), m_Handle, offset, size));
    NN_RESULT_SUCCESS;
}


}}

