﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/eth/sfdl/eth.sfdl.h>
#include <nn/eth/eth_Result.h>
#include <nn/eth/server/eth_EthServer.h>

#include "eth_InterfaceGroupSfObject.h"
#include "eth_InterfaceSfObject.h"

#include "device/eth_DeviceMain.h"

namespace nn     {
namespace eth    {

nn::lmem::HeapHandle g_HeapHandle;

namespace server {

const int SessionThreadStackSize = (16 * 1024);
const int InterfaceGroupSessions = 5;
const int InterfaceSessions      = 5;
const int SessionMaxCount        = InterfaceGroupSessions + InterfaceSessions;
nn::sf::ExpHeapAllocator         g_ServiceFrameworkAllocator;

enum ipcSessionPort
{
    sessionPort_Group,
    sessionPort_Interface,
    sessionPort_Highest
};

class EthService
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionMaxCount, sessionPort_Highest>
{
private:
    nn::os::ThreadType m_SessionThread;
    uint8_t            m_SessionThreadStack[SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);

    static void SessionThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        EthService* p = (EthService*)arg;
        p->LoopAuto();
    }

    nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort)
    NN_NOEXCEPT
    {
        if (portIndex == sessionPort_Group)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::eth::sf::IEthInterfaceGroup,
                              InterfaceGroupSfObject
                             >(&g_ServiceFrameworkAllocator);

            NN_ABORT_UNLESS(interface != nullptr);

            return this->AcceptImpl(pPort, interface);
        }
        else if (portIndex == sessionPort_Interface)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::eth::sf::IEthInterface,
                              InterfaceSfObject
                             >(&g_ServiceFrameworkAllocator);

            NN_ABORT_UNLESS(interface != nullptr);

            return this->AcceptImpl(pPort, interface);
        }

        return nn::eth::ResultNotSupported();
    }

public:
    void Wait()
    NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_SessionThread);
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        result = this->InitializePort(sessionPort_Group, InterfaceGroupSessions, InterfaceGroupPortName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = this->InitializePort(sessionPort_Interface, InterfaceSessions, InterfacePortName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        this->Start();

        result = nn::os::CreateThread(&m_SessionThread, SessionThreadEntry, this, m_SessionThreadStack, SessionThreadStackSize,
                                      NN_SYSTEM_THREAD_PRIORITY(socket, EthIpcServer));
        SetThreadNamePointer(&m_SessionThread, NN_SYSTEM_THREAD_NAME(socket, EthIpcServer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::StartThread(&m_SessionThread);

        return ResultSuccess();
    }

} g_EthServer;

nn::Result Initialize(void* pEthMemoryIn, size_t ethMemorySize)
NN_NOEXCEPT
{
    nn::Result result;

    g_HeapHandle = nn::lmem::CreateExpHeap(
                            reinterpret_cast<uint8_t*>(pEthMemoryIn),
                            ethMemorySize,
                            nn::lmem::CreationOption_ThreadSafe);

    NN_ABORT_UNLESS(g_HeapHandle != nullptr);

    g_ServiceFrameworkAllocator.Attach(g_HeapHandle);

    result = nn::eth::device::Initialize();

    if (result.IsSuccess())
    {
        result = g_EthServer.Initialize();
    }

    return result;
}

void Wait()
NN_NOEXCEPT
{
    g_EthServer.Wait();
}

}}}
