﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <nn/nn_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/eth/eth_EthTypes.h>

#include "../eth_Interface.h"

namespace nn     {
namespace eth    {
namespace device {

const char UsbInterfaceName[] = "usb";
const char PciInterfaceName[] = "pci";
const char Tx2InterfaceName[] = "tx2";

class MiiConfig
{
public:
    MediaType m_MediaType;
    uint16_t  m_Bmcr;
    uint16_t  m_Anar;
    uint16_t  m_Gtcr;

    explicit MiiConfig(MediaType type, uint32_t bmcr, uint32_t anar, uint32_t gtcr) :
        m_MediaType(type),
        m_Bmcr(bmcr),
        m_Anar(anar),
        m_Gtcr(gtcr)
    {
    }
};

class Gmii :
    public server::Interface
{
private:
    uint32_t AdjustForTickRate(uint32_t rateSeconds) NN_NOEXCEPT;

protected:
    enum
    {
        MIIF_HAVE_GTCR,
        MIIF_STOPPED,
    };

    static const MiiConfig  m_MediaConfiguration[];
    static uint8_t          m_DeviceInstanceCounter;
    MediaType               m_MediaRequestedType;
    MediaType               m_MediaCurrentType;
    MediaList               m_MediaList;
    uint32_t                m_PhyCapabilities;
    uint32_t                m_PhyExtraCapabilities;
    uint32_t                m_Flags;
    uint32_t                m_AutoNegotiationTicks;
    uint64_t                m_Ticks;
    uint32_t                m_PhyUniqueIdentifier;
    uint32_t                m_PhyModel;
    uint32_t                m_PhyRevision;
    uint8_t                 m_PhyMacAddress[6];
    char                    m_Name[MaxInterfaceNameLength];
    char                    m_FullName[MaxInterfaceNameLength];
    uint32_t                m_EventCounter;
    uint8_t                 m_Instance;
    uint32_t                m_TickRateMs;

    virtual nn::Result PhyReadReg(uint16_t miiRegister, uint16_t* miiValue)
    NN_NOEXCEPT
    {
        return nn::os::ResultNotImplemented();
    }

    virtual nn::Result PhyWriteReg(uint16_t miiRegister, uint16_t miiValue)
    NN_NOEXCEPT
    {
        return nn::os::ResultNotImplemented();
    }

    virtual nn::Result PhySetMedia(MediaType type)
    NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    virtual nn::Result PhyGetMedia(MediaType* type)
    NN_NOEXCEPT
    {
        return nn::os::ResultNotImplemented();
    }

    virtual nn::Result PhyReset()
    NN_NOEXCEPT
    {
        return nn::os::ResultNotImplemented();
    }

    virtual nn::Result PhyLinkReset()
    NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    // adapters may require adjustment after auto neg completes
    virtual nn::Result PhyAdjustMedia(MediaType type)
    NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    nn::Result CheckSupportedMedia(MediaType type) NN_NOEXCEPT;

public:

    Gmii(const char* name, uint32_t tickRateMs) NN_NOEXCEPT :
        m_MediaRequestedType(MediaType_Unknown),
        m_MediaCurrentType(MediaType_Unknown),
        m_PhyCapabilities(0),
        m_PhyExtraCapabilities(0),
        m_Flags(MIIF_STOPPED),
        m_PhyUniqueIdentifier(0),
        m_PhyModel(0),
        m_EventCounter(0),
        m_Instance(m_DeviceInstanceCounter++),
        m_TickRateMs(tickRateMs)
    {
        nn::util::Strlcpy(m_Name, name, sizeof(m_Name));
        nn::util::SNPrintf(m_FullName, sizeof(m_FullName), "%s%d", m_Name, m_Instance);
    }

    virtual ~Gmii()
    NN_NOEXCEPT
    {

    }

    const char* DeviceName()
    NN_NOEXCEPT
    {
        return m_FullName;
    }

    nn::Result Reset() NN_NOEXCEPT;
    nn::Result LinkReset() NN_NOEXCEPT;
    nn::Result UpdateLinkStatus() NN_NOEXCEPT;
    nn::Result GetInterfaceInfo(InterfaceInfo* pInterfaceInfo) NN_NOEXCEPT;
    nn::Result GetMediaList(MediaList* pMediaList) NN_NOEXCEPT;
    nn::Result GetMediaType(MediaType* pRequestedType, MediaType* pCurrentType, uint32_t* pEventCounter) NN_NOEXCEPT;
    nn::Result SetMediaType(MediaType mediaType) NN_NOEXCEPT;
    nn::Result Tick() NN_NOEXCEPT;
    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result Finalize() NN_NOEXCEPT;
};

}}}
