﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <cstring>
#include <nn/nn_SdkLog.h>

#include "dt_FdtReaderUtil.h"

extern "C"
{
#include "libfdt.h"
}

namespace nn { namespace dt { namespace server {

#if !defined(NN_SDK_BUILD_RELEASE)
namespace {

bool IsStringList(const char* pValue, int size)
{
    // 長さが 1 以下なら stringlist ではない
    if (size <= 1)
    {
        return false;
    }

    // 末尾がヌル文字でなければ stringlist ではない
    if (pValue[size - 1] != '\0')
    {
        return false;
    }

    uint8_t prev = '\0';

    for (int i = 0; i < size - 1; i++)
    {
        uint8_t c = pValue[i];

        // ヌル文字または表示可能文字以外が含まれていたら stringlist ではない
        if (c != '\0' && !isprint(c))
        {
            return false;
        }

        // ヌル文字が連続したら stringlist ではない
        if (prev == '\0' && c == '\0')
        {
            return false;
        }

        prev = c;
    }

    return true;
}

bool IsCellList(const char* pValue, int size)
{
    NN_UNUSED(pValue);
    return size % 4 == 0;
}

void PrintStringList(const char* pValue, int size)
{
    NN_SDK_LOG("\"");
    for (int i = 0; i < size - 1; i++)
    {
        uint8_t c = pValue[i];
        if (c == '\0')
        {
            NN_SDK_LOG("\", \"");
        }
        else
        {
            NN_SDK_LOG("%c", pValue[i]);
        }
    }
    NN_SDK_LOG("\"");
}

void PrintCellList(const char* pValue, int size)
{
    auto p = reinterpret_cast<const int32_t*>(pValue);
    auto count = size / (sizeof(int32_t) / sizeof(char));

    NN_SDK_LOG("<");
    for (int i = 0; i < count; i++)
    {
        NN_SDK_LOG("0x%x", fdt32_to_cpu(p[i]));
        if (i != count - 1)
        {
            NN_SDK_LOG(" ");
        }
    }
    NN_SDK_LOG(">");
}

void PrintBinary(const char* pValue, int size)
{
    NN_SDK_LOG("[");
    for (int i = 0; i < size; i++)
    {
        NN_SDK_LOG("%02x", pValue[i]);
        if (i != size - 1)
        {
            NN_SDK_LOG(" ");
        }
    }
    NN_SDK_LOG("]");
}

void PrintProperty(const char* pValue, int size) NN_NOEXCEPT
{
    if (size == 0)
    {
        return;
    }

    if (IsStringList(pValue, size))
    {
        PrintStringList(pValue, size);
    }
    else if (IsCellList(pValue, size))
    {
        PrintCellList(pValue, size);
    }
    else
    {
        PrintBinary(pValue, size);
    }
}

void PrintNodeRecursive(const void* pFdt, int nodeOffset, int depth, int indent) NN_NOEXCEPT
{
    auto pName = fdt_get_name(pFdt, nodeOffset, nullptr);
    if (pName == nullptr)
    {
        return;
    }

    // Dump start
    if (std::strlen(pName) >= 1)
    {
        NN_SDK_LOG("%*s%s {\n", indent, "", pName);
    }
    else
    {
        NN_SDK_LOG("%*s{\n", indent, "");
    }

    // Dump property
    int propertyOffset;
    fdt_for_each_property_offset(propertyOffset, pFdt, nodeOffset)
    {
        int len;
        auto pProperty = fdt_get_property_by_offset(pFdt, propertyOffset, &len);
        auto propertyName = fdt_string(pFdt, fdt32_to_cpu(pProperty->nameoff));

        NN_SDK_LOG("%*s%s", indent + IndentSize, " ", propertyName);

        if (len > 0)
        {
            NN_SDK_LOG(" = ");
            PrintProperty(pProperty->data, len);
        }

        NN_SDK_LOG(";\n");
    }

    // Dump child
    int childOffset;
    int count = 0;
    fdt_for_each_subnode(childOffset, pFdt, nodeOffset)
    {
        count++;

        if (depth != 0)
        {
            // 子ノードの dump は1行空ける
            if (count != 0)
            {
                NN_SDK_LOG("\n");
            }

            if (depth < 0)
            {
                // depth が負ならリーフまで探索
                PrintNodeRecursive(pFdt, childOffset, -1, indent + IndentSize);
            }
            else
            {
                PrintNodeRecursive(pFdt, childOffset, depth - 1, indent + IndentSize);
            }
        }
    }

    if (depth == 0)
    {
        NN_SDK_LOG("%*s// child count: %d\n", indent + IndentSize, "", count);
    }

    // Dump end
    NN_SDK_LOG("%*s};\n", indent, "");
}

}
#endif

void PrintNode(const void* pFdt, Node* pNode, int depth) NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    PrintNodeRecursive(pFdt, pNode->GetOffset(), depth, 0);
#else
    NN_UNUSED(pFdt);
    NN_UNUSED(pNode);
    NN_UNUSED(depth);
#endif
}

}}}
