﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <cstdio>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/dd.h>

#include <nn/pcie/pcie.h>

#include <nn/drivers/eth/driverEth.h>
#include "eth_Realtek.h"

namespace nn { namespace drivers { namespace eth {

// Log Macros
#define ETH_ABORT_UNLESS_SUCCESS(result) NN_ABORT_UNLESS((result).IsSuccess())
#define ETH_LOG_INFO(format,...)      NN_SDK_LOG("[eth] info: " format, ##__VA_ARGS__)
#define ETH_LOG_WARN(format,...)      NN_SDK_LOG("[eth] warning: " format, ##__VA_ARGS__)
#define ETH_LOG_ERROR(format,...)     NN_SDK_LOG("[eth] error: " format, ##__VA_ARGS__)
#define ETH_LOG_NOPREFIX(format,...)  NN_SDK_LOG(format, ##__VA_ARGS__)

// Error handling Macros
#define ETH_BREAK_UPON_ERROR(attemptedMethodCall)                \
    if (!((result=(attemptedMethodCall)).IsSuccess()))           \
    {                                                            \
        ETH_LOG_WARN("call to %s failed, handled with break.\n", \
                     #attemptedMethodCall);  break;             \
    }


// Endian swap
#define ETH_ENDIAN_SWAP_U32(x)(((((uint32_t)(x)) & 0xff000000) >> 24)  | \
                               ((((uint32_t)(x)) & 0x00ff0000) >> 8)   | \
                               ((((uint32_t)(x)) & 0x0000ff00) << 8)   | \
                               ((((uint32_t)(x)) & 0x000000ff) << 24))

#define ETH_U32_ROUNDUP( value, alignment ) \
    ((((uint32_t)(value)) + ((alignment) - 1)) & ~((alignment) - 1))
#define ETH_DATA_SIZEOF_ROUNDUP(DATA_TYPE, size) \
    ETH_U32_ROUNDUP(sizeof(DATA_TYPE),(size))


// Mask computation
#define DMA_BIT_MASK(value)    ((1ULL << value) - 1)

// Configuration
#define ENABLE_ASPM   FALSE

class EthernetDriver
{
private:
    static const size_t     ETHER_FRAME_BUF_SIZE  = 1518;
    static const size_t     ETHER_CRC_SIZE        = 4;
    static const size_t     ETHER_MTU_SIZE        = ETHER_FRAME_BUF_SIZE - ETHER_CRC_SIZE;
    static const int32_t    NUM_TX_DESC = 32;
    static const int32_t    NUM_RX_DESC = 32;
    static const int32_t    COHERENT_RX_DESC_COUNT = 4;
    static const nn::pcie::IrqType IRQ_TYPE = nn::pcie::IrqType_Msi; //nn::pcie::IrqType_IntX
    static const uint8_t    BAR = 2;

public:
    // Multicast support
    static const int32_t MAX_MULTICAST_ADDRESSES = 32;

    // Link configuration
    enum LinkSpeed
    {
        LinkSpeed_Unknown   = 0,
        LinkSpeed_10        = 10,
        LinkSpeed_100       = 100,
        LinkSpeed_1000      = 1000
    };
    enum LinkDuplex
    {
        LinkDuplex_Unknown  = 0,
        LinkDuplex_Half     = 1,
        LinkDuplex_Full     = 2
    };
    enum LinkAutoNegotiation
    {
        LinkAutoNegotiation_Disable = 0,
        LinkAutoNegotiation_Enable  = 1
    };
    struct LinkState
    {
        bool            linkUp;
        LinkSpeed       speed;
        LinkDuplex      duplex;
        bool            autoNegComplete;
        bool            txFlowControl;
        bool            rxFlowControl;
    };

    EthernetDriver();
    ~EthernetDriver();
    nn::Result Initialize();
    nn::Result Finalize();
    void RegisterFrameReceivedCallBack(FrameReceivedCallBack callBack)
    {
        m_FrameReceivedCallBack = callBack;
    }
    void RegisterSendCompletedCallBack(SendCompletedCallBack callBack)
    {
        m_SendCompletedCallBack = callBack;
    }
    void UnregisterFrameReceivedCallBack()
    {
        m_FrameReceivedCallBack = NULL;
    }
    void UnregisterSendCompletedCallBack()
    {
        m_SendCompletedCallBack = NULL;
    }
    void RegisterInterfaceStateCallBack(InterfaceStateCallBack callback);
    void UnregisterInterfaceStateCallBack();

    nn::Result Send(const void *frameData, int frameBytes);
    nn::Result Send(nnnetOslMbuf *pMbuf);
    nn::Result Send(int32_t numSegments, uint8_t **segmentBuffers, size_t *segmentSizes);
    nn::Result GetMacAddress(uint8_t pMacAddress[NN_NET_MAC_ADDRESS_SIZE]) const;

private:
    bool                  m_IsDriverInit;
    bool                  m_IsDeviceAvailable;
    bool                  m_ReInit;
    LinkState             m_LinkState;
    InterfaceState        m_LastReportedInterfaceState;
    bool                  m_IsPromiscuous;

    // Driver thread
    enum ThreadPorts
    {REEVALUATION_PORT = 0, PCIE_REGISTRATION_PORT, PCIE_PM_PORT, PCIE_IRQ_PORT, MAX_THREAD_PORTS};
    nn::os::SystemEventType m_SystemEvent[MAX_THREAD_PORTS];

    nn::os::ThreadType          m_Thread;
    NN_ALIGNAS(nn::os::StackRegionAlignment) uint8_t m_ThreadStack[1024 * 16];
    nn::os::SemaphoreType       m_AccessSem;
    nn::os::SemaphoreType       m_InitSem;
    nn::os::MultiWaitType       m_MultiWait;
    nn::os::MultiWaitHolderType m_Holder[MAX_THREAD_PORTS];

    // PCI related
    nn::pcie::ClassDriverConfig  m_PcieDrvCfg;
    nn::pcie::ClassDriverHandle  m_PcieDrvHandle;
    nn::pcie::FunctionHandle     m_PcieFuncHandle;
    nn::pcie::FunctionState      m_PcieFuncState;
    nn::pcie::BarProfile         m_Bar2Profile;

    FrameReceivedCallBack    m_FrameReceivedCallBack;
    SendCompletedCallBack    m_SendCompletedCallBack;
    InterfaceStateCallBack   m_InterfaceStateCallBack;
    uint8_t                  m_SerialNum[8];

    // Chip parameter table
    struct ChipParams
    {
        const char *name;
        uint32_t    version;        /* depend on RTL8169 docs */
        uint32_t    rxConfigMask;   /* should clear the bits supported by this chip */
        uint32_t    rxConfig;
    };
    static ChipParams m_ChipInfoList[];
    int32_t m_ChipInfoIndex;

    // Descritor data types
    struct TxDMADescArrayType
    {
        TxDMADesc list[NUM_TX_DESC];
    };
    struct RxDescArrayType
    {
        RxDMADesc list[NUM_RX_DESC];
    };
    struct DmaDescriptorsContainer
    {
        union
        {
            TxDMADescArrayType  data;
            uint8_t             padding[ETH_DATA_SIZEOF_ROUNDUP(TxDMADescArrayType,REALTEK_MAC_DMA_ALIGNMENT_SIZE)];
        }NN_ALIGNAS(REALTEK_MAC_DMA_ALIGNMENT_SIZE) txDesc;
        union
        {
            RxDescArrayType data;
            uint8_t         padding[ETH_DATA_SIZEOF_ROUNDUP(RxDescArrayType,REALTEK_MAC_DMA_ALIGNMENT_SIZE)];
        }NN_ALIGNAS(REALTEK_MAC_DMA_ALIGNMENT_SIZE) rxDesc;
        union
        {
            CountersDMADesc data;
            uint8_t         padding[ETH_DATA_SIZEOF_ROUNDUP(CountersDMADesc,REALTEK_MAC_DMA_ALIGNMENT_SIZE)];
        }NN_ALIGNAS(REALTEK_MAC_DMA_ALIGNMENT_SIZE) counters;
    };

    // Frame data type
    union FrameType
    {
        uint8_t     data[ETHER_FRAME_BUF_SIZE];
        uint8_t     padding[ETH_U32_ROUNDUP(ETHER_FRAME_BUF_SIZE,REALTEK_MAC_DMA_ALIGNMENT_SIZE)];
    };

    // Frame DMA buffers
    union
    {
        FrameType   list[NUM_RX_DESC];
        uint8_t     padding[ETH_U32_ROUNDUP(sizeof(FrameType) * NUM_RX_DESC,
                                            nn::dd::DeviceAddressSpaceMemoryRegionAlignment)];
    }NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY m_DmaRxFrames;

    union
    {
        FrameType   list[NUM_TX_DESC];
        uint8_t     padding[ETH_U32_ROUNDUP(sizeof(FrameType) * NUM_TX_DESC,
                                            nn::dd::DeviceAddressSpaceMemoryRegionAlignment)];
    }NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY m_DmaTxFrames;

    // Descriptor DMA buffers
    union
    {
        DmaDescriptorsContainer  data;
        uint8_t      padding[ETH_DATA_SIZEOF_ROUNDUP(DmaDescriptorsContainer,
                                                     nn::dd::DeviceAddressSpaceMemoryRegionAlignment)];
    }NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY m_DmaDesc;

    // Driver parameters
    uint8_t         m_MacAddress[NN_NET_MAC_ADDRESS_SIZE];

    // Base bus addressess of DMA-able structures
    nn::pcie::BusAddress m_TxDMADescArrayBusAddress;
    nn::pcie::BusAddress m_RxDescArrayBusAddresss;
    nn::pcie::BusAddress m_ShortPacketBufferBusAddress;
    nn::pcie::BusAddress m_CountersBusAddress;
    nn::pcie::BusAddress m_TxFramesBusAddressList[NUM_TX_DESC];
    nn::pcie::BusAddress m_RxFramesBusAddressList[NUM_RX_DESC];

    // Tx transaction data
    struct TxTxn
    {
        const void *pFrame;
        int         size;
    };
    TxTxn      m_TxTxnList[NUM_TX_DESC];

    // Indices
    int32_t    m_TxFreeIndex;
    int32_t    m_TxPendIndex;
    int32_t    m_TxPendCount;
    int32_t    m_RxIndex;

    // Driver statistics
    struct Statistics
    {
        // reported via interrupt status
        uint32_t SysErrInt;
        uint32_t PcsTimeoutInt;
        uint32_t RxFifoOverflowInt;
        uint32_t RxUnderrunInt;
        uint32_t RxDescUnavailInt;
        uint32_t TxErrInt;
        uint32_t RxErrInt;
        uint32_t RxOkInt;
        uint32_t TxOkInt;
        uint32_t RxSizeError;

        // reported via descriptor
        uint32_t RxErrIntDesc;
        uint32_t RxLengthErrDesc;
        uint32_t RxCrcErrDesc;
        uint32_t RxFifoErrDesc;
        uint32_t RxFragmentErrDesc;

        // summary of rx
        uint32_t RxDelivered;
        uint32_t RxDiscarded;

    } m_Statistics;

    // PCI administration callback
    void HandlePcieDriverRegistrationEvent();

    void Lock();
    void Unlock();
    void ReadLinkState(LinkState *pLinkState);

    // Interrupt handling
    void InterruptThread();
    static void  InterruptThreadEntry(void *pThis)
    {
        ((EthernetDriver *)pThis)->InterruptThread();
    }
    void HandleTxCompletions(bool reset);
    void HandleRxCompletions(bool reset);

    // Multicast
    Result SetRxFilter(uint8_t addresses[MAX_MULTICAST_ADDRESSES][NN_NET_MAC_ADDRESS_SIZE],
                       int32_t numMultiCastAddr, bool bPromiscuous);
    uint32_t EtherCRC32BigEndian(uint8_t *buf, size_t len);

    // Phy
    void SetSpeedXmii(uint8_t autoneg, uint16_t speed, uint8_t duplex);
    uint16_t MapPhyOcpAddr(uint16_t PageNum, uint8_t RegNum);
    void MdioWrite(uint16_t PageNum, uint32_t RegAddr, uint32_t value);
    uint32_t MdioRead(uint16_t PageNum, uint32_t RegAddr);
    uint32_t EriRead(int32_t addr, int32_t len, int32_t type);
    int32_t EriWrite(int32_t addr, int32_t len, uint32_t value, int32_t type);

    // Control
    Result InitHardware();
    Result StartHardware(bool isReInit);
    Result StopHardware();
    void MonitorLinkState();
    void ReportInterfaceState();
    void ResetHardware();

    // Statistics
    void SampleCounters();

    // Register access
    void RegWrite8(size_t offset, uint8_t val8);
    void RegWrite16(size_t offset, uint16_t val16);
    void RegWrite32(size_t offset, uint32_t val32);
    uint8_t RegRead8(size_t offset);
    uint16_t RegRead16(size_t offset);
    uint32_t RegRead32(size_t offset);
};

} // namespace eth
} // namespace drivers
} // namespace nn

