﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ro/detail/ro_Elf.h>

namespace nn { namespace diag { namespace detail {

/*!
    @brief  シンボルテーブルの解析器です。
*/
class SymbolTableParser
{
public:
    /*!
        @brief      解析器を初期化します。

        @param[in]  address シンボルテーブルを検索するモジュール内のアドレス。

        @return     初期化に成功したとき true、失敗したとき false。

        @details
                    @a address が不正なとき、初期化に失敗します。
    */
    bool Initialize(uintptr_t address) NN_NOEXCEPT;

    /*!
        @brief      解析器を次に進めます。

        @return     シンボルが見つかったとき true、見つからなかったとき false。

        @details
                    解析器がシンボルテーブルの末尾に達したときに false が返ります。
    */
    bool MoveNext() NN_NOEXCEPT;

    /*!
        @brief      引数に指定したインデックスに、解析器を移動させます。

        @pre
            - 0 < index < m_Count
    */
    void MoveTo(int index) NN_NOEXCEPT;

    /*!
        @brief      解析器が指すシンボルテーブル内のインデックスを取得します。

        @return     インデックス。

        @pre
            - MoveNext() が true を返している。
    */
    int GetIndex() const NN_NOEXCEPT;

    /*!
        @brief      シンボルテーブルに含まれるシンボルの個数を取得します。

        @return     シンボルの個数。

        @pre
            - Initialize() が true を返している。
    */
    int GetCount() const NN_NOEXCEPT;

    /*!
        @brief      解析器が現在指すシンボルの名前を取得します。

        @return     シンボルの名前。

        @pre
            - MoveNext() が true を返している。
    */
    const char* GetName() const NN_NOEXCEPT;

    /*!
        @brief      解析器が現在指すシンボルのサイズを取得します。

        @return     シンボルのサイズ。

        @pre
            - MoveNext() が true を返している。
    */
    size_t GetSize() const NN_NOEXCEPT;

    /*!
        @brief      解析器が現在指すシンボルのアドレスを取得します。

        @return     シンボルのアドレス。

        @pre
            - MoveNext() が true を返している。
    */
    uintptr_t GetAddress() const NN_NOEXCEPT;

    /*!
        @brief      解析器が現在指すシンボルのタイプを取得します。

        @return     シンボルのタイプ。

        @pre
            - MoveNext() が true を返している。
    */
    unsigned char GetType() const NN_NOEXCEPT;

private:
    const ro::detail::Elf::Sym* m_pSymTab;
    const char* m_pStrTab;
    uintptr_t m_ModuleAddress;
    int m_Count;
    int m_Index;
};

}}} // nn::diag::detail
