﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Windows.h>
#include <nn/nn_Common.h>
#include <nn/diag/diag_AbortTypes.h>
#include <nn/diag/diag_AbortObserver.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_CharacterEncodingResult.h>
#include "diag_AbortObserverManager.h"

namespace nn { namespace diag { namespace detail {

namespace {

// ShowAbortDialog で表示するダイアログのメッセージを生成するためのバッファサイズです。
const size_t DialogMsgBufLength = 4096;

const char* ToString(
    AbortReason abortReason) NN_NOEXCEPT
{
    switch( abortReason )
    {
    case AbortReason_SdkAssert:
        return "SDK Assertion Failure";
    case AbortReason_SdkRequires:
        return "Precondition not met";
    case AbortReason_UserAssert:
        return "Assertion Failure";
    case AbortReason_UnexpectedDefault:
        return "Unexpected Default";
    case AbortReason_Abort:
    default:
        return "Abort";
    }
}

// ダイアログ用のメッセージを作成。
bool MakeMessage(
    uint16_t* outMsg16,
    int dstLength,
    const AbortInfo& abortInfo,
    bool useMessage,
    bool useInfo) NN_NOEXCEPT
{
    // メッセージを UTF-8 で作成する。
    char msg8[DialogMsgBufLength] = { 0 };

    int wroteBytes = 0;

    // アボート情報。
    if (useInfo)
    {
        wroteBytes = util::SNPrintf(
            msg8,
            DialogMsgBufLength,
            "%s: \'%s\' at '%s':%d in %s\n---\n",
            ToString(abortInfo.abortReason),
            abortInfo.condition,
            abortInfo.fileName,
            abortInfo.lineNumber,
            abortInfo.functionName);
    }
    else
    {
        // アボート情報を省く。
        wroteBytes = util::SNPrintf(
            msg8,
            DialogMsgBufLength,
            "%s\n---\n",
            ToString(abortInfo.abortReason));
    }

    // メッセージ。
    if (wroteBytes < DialogMsgBufLength)
    {
        if (useMessage)
        {
            wroteBytes += util::VSNPrintf(
                msg8 + wroteBytes,
                DialogMsgBufLength - wroteBytes,
                abortInfo.message->format,
                *abortInfo.message->args);
        }
        else
        {
            // メッセージを差し替える。
            wroteBytes += util::SNPrintf(
                msg8 + wroteBytes,
                DialogMsgBufLength - wroteBytes,
                "(Invalid encoding of message)");
        }
    }

    // デバッグ方法の指示。
    if (wroteBytes < DialogMsgBufLength)
    {
        wroteBytes += util::SNPrintf(
            msg8 + wroteBytes,
            DialogMsgBufLength - wroteBytes,
            "\n\n---\nPress OK to debug the application",
            *abortInfo.message->args);
    }


    // メッセージを UTF-16 に変換する。
    util::CharacterEncodingResult encodingResult =
        util::ConvertStringUtf8ToUtf16Native(
            outMsg16, dstLength - 1, // 終端文字の分だけ空けておく。
            msg8, wroteBytes);

    return encodingResult != util::CharacterEncodingResult_InvalidFormat;
}

// ダイアログでアボートの情報を表示し、ブレークする。
void ShowAbortDialog(const AbortInfo &abortInfo) NN_NOEXCEPT
{
    uint16_t msg16[DialogMsgBufLength] = { 0 };
    const wchar_t* msgW = nullptr;

    // ダイアログのメッセージを作る。
    bool encResult = MakeMessage(msg16, DialogMsgBufLength, abortInfo, true, true);

    if (!encResult)
    {
        // 不正なエンコーディングがあったので、まずメッセージ部分を省いて再試行。
        std::memset(msg16, 0, sizeof(uint16_t) * DialogMsgBufLength);
        encResult = MakeMessage(msg16, DialogMsgBufLength, abortInfo, false, true);
    }

    if (!encResult)
    {
        // まだ不正だったので、ファイル名や関数名を省いて再試行。
        std::memset(msg16, 0, sizeof(uint16_t) * DialogMsgBufLength);
        encResult = MakeMessage(msg16, DialogMsgBufLength, abortInfo, true, false);
    }

    if (!encResult)
    {
        // まだ不正だったので、両方を省いて再試行。
        std::memset(msg16, 0, sizeof(uint16_t) * DialogMsgBufLength);
        encResult = MakeMessage(msg16, DialogMsgBufLength, abortInfo, false, false);
    }

    if (!encResult)
    {
        // 何らかの理由でエンコードに失敗したので、定型文を入れる。
        msgW = L"Abort";
    }
    else
    {
        // メッセージの作成に成功した。
        // Visual C++ において、wchar_t は UTF-16 なので、そのままキャストする。
        msgW = reinterpret_cast<wchar_t*>(msg16);
    }

    // ダイアログを表示。
    int msgboxResult = MessageBoxW(NULL, msgW, L"NintendoSDK", MB_OKCANCEL | MB_ICONERROR | MB_DEFBUTTON1 | MB_SETFOREGROUND);
    if (msgboxResult == IDOK)
    {
        // ブレークする。
        _CrtDbgBreak();
    }
}

}

void DefaultAbortObserver(const AbortInfo &abortInfo) NN_NOEXCEPT;

NN_NORETURN void Abort(const Result* result) NN_NOEXCEPT
{
    NN_UNUSED(result);

    // InvokeAbortObserver で _WRITE_ABORT_MSG フラグが落とされているので、
    // この abort ではダイアログの表示もブレークもされず、単に exit が呼ばれる。
    std::abort();
    while(NN_STATIC_CONDITION(true))
    {
    }
}

extern bool g_EnableDefaultAbortObserver;

void InvokeAbortObserver(
    const AbortInfo& abortInfo) NN_NOEXCEPT
{
    if (g_EnableDefaultAbortObserver)
    {
        DefaultAbortObserver(abortInfo);
    }
    GetAbortObserverManager()->InvokeAllObserver(abortInfo);

    // アボートの情報をダイアログで表示する。
    // これは Windows の abort が出すダイアログの改良版を表示するという役割なので、
    // abort の中でダイアログを出さないようにフラグを変更しつつ、フラグの元の値を取得し、
    // ユーザーが _set_abort_behavior でダイアログを出さないようにしていたら、
    // NintendoSDK でも出さないようにする。
#ifndef NN_SDK_BUILD_RELEASE // Windows の abort に合わせて、Release では無効化。
    if (_set_abort_behavior(0, _WRITE_ABORT_MSG) != 0)
    {
        ShowAbortDialog(abortInfo);
    }
#endif
}

void InvokeSdkAbortObserver(
    const SdkAbortInfo& abortInfo) NN_NOEXCEPT
{
    GetSdkAbortObserverManager()->InvokeAllObserver(abortInfo);
}

}}} // nn::diag::detail
