﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dd/dd_Config.h>
#include <nn/nn_SdkText.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/dd/dd_DeviceAddressSpaceCommon.h>
#include <nn/dd/dd_DeviceAddressSpaceTypes.h>
#include <nn/dd/dd_Types.h>
#include <nn/dd/dd_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Result.h>

#include "dd_DeviceAddressSpaceImpl-os.horizon.h"

namespace nn { namespace dd { namespace detail {

//-----------------------------------------------------------------------------
// 列挙子などの定義が nn::dd と nn::svc で一致していることを静的にチェック
//

NN_DD_STATIC_ASSERT_EQUAL(dd::MemoryPermission_None,      svc::MemoryPermission_None);
NN_DD_STATIC_ASSERT_EQUAL(dd::MemoryPermission_ReadOnly,  svc::MemoryPermission_Read);
NN_DD_STATIC_ASSERT_EQUAL(dd::MemoryPermission_WriteOnly, svc::MemoryPermission_Write);
NN_DD_STATIC_ASSERT_EQUAL(dd::MemoryPermission_ReadWrite, svc::MemoryPermission_ReadWrite);

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
#define NN_DD_SUPPORTS_DEVICE_ADDRESS_SPACE
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Afi,       svc::DeviceName_Afi);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Avpc,      svc::DeviceName_Avpc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dc,        svc::DeviceName_Dc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dcb,       svc::DeviceName_Dcb);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Hc,        svc::DeviceName_Hc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Hda,       svc::DeviceName_Hda);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Isp2,      svc::DeviceName_Isp2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Msenc,     svc::DeviceName_Msenc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nv,        svc::DeviceName_Nv);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nv2,       svc::DeviceName_Nv2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs,      svc::DeviceName_Ppcs);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sata,      svc::DeviceName_Sata);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Vde,       svc::DeviceName_Vde);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Vi,        svc::DeviceName_Vi);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Vic,       svc::DeviceName_Vic);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_XusbHost,  svc::DeviceName_XusbHost);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_XusbDev,   svc::DeviceName_XusbDev);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Tsec,      svc::DeviceName_Tsec);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs1,     svc::DeviceName_Ppcs1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dc1,       svc::DeviceName_Dc1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc1a,   svc::DeviceName_Sdmmc1a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc2a,   svc::DeviceName_Sdmmc2a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc3a,   svc::DeviceName_Sdmmc3a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc4a,   svc::DeviceName_Sdmmc4a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Isp2b,     svc::DeviceName_Isp2b);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Gpu,       svc::DeviceName_Gpu);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Gpub,      svc::DeviceName_Gpub);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs2,     svc::DeviceName_Ppcs2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Num,       svc::DeviceName_Num);

#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
#define NN_DD_SUPPORTS_DEVICE_ADDRESS_SPACE
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Afi,       svc::DeviceName_Afi);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Avpc,      svc::DeviceName_Avpc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dc,        svc::DeviceName_Dc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dcb,       svc::DeviceName_Dcb);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Hc,        svc::DeviceName_Hc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Hda,       svc::DeviceName_Hda);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Isp2,      svc::DeviceName_Isp2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nvenc,     svc::DeviceName_Nvenc);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nv,        svc::DeviceName_Nv);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nv2,       svc::DeviceName_Nv2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs,      svc::DeviceName_Ppcs);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sata,      svc::DeviceName_Sata);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Vi,        svc::DeviceName_Vi);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Vic,       svc::DeviceName_Vic);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_XusbHost,  svc::DeviceName_XusbHost);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_XusbDev,   svc::DeviceName_XusbDev);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Tsec,      svc::DeviceName_Tsec);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs1,     svc::DeviceName_Ppcs1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Dc1,       svc::DeviceName_Dc1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc1a,   svc::DeviceName_Sdmmc1a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc2a,   svc::DeviceName_Sdmmc2a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc3a,   svc::DeviceName_Sdmmc3a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Sdmmc4a,   svc::DeviceName_Sdmmc4a);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Isp2b,     svc::DeviceName_Isp2b);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Gpu,       svc::DeviceName_Gpu);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Gpub,      svc::DeviceName_Gpub);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ppcs2,     svc::DeviceName_Ppcs2);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nvdec,     svc::DeviceName_Nvdec);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Ape,       svc::DeviceName_Ape);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Se,        svc::DeviceName_Se);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nvjpg,     svc::DeviceName_Nvjpg);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Hc1,       svc::DeviceName_Hc1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Se1,       svc::DeviceName_Se1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Axiap,     svc::DeviceName_Axiap);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Etr,       svc::DeviceName_Etr);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Tsecb,     svc::DeviceName_Tsecb);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Tsec1,     svc::DeviceName_Tsec1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Tsecb1,    svc::DeviceName_Tsecb1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Nvdec1,    svc::DeviceName_Nvdec1);
NN_DD_STATIC_ASSERT_EQUAL(dd::DeviceName_Num,       svc::DeviceName_Num);
#endif


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトの作成
//
Result DeviceAddressSpaceImplByHorizon::Create(DeviceAddressSpaceHandle* pHandle, uint64_t address, uint64_t size) NN_NOEXCEPT
{
#if defined( NN_DD_SUPPORTS_DEVICE_ADDRESS_SPACE )

    svc::Handle handle;
    auto result = svc::CreateDeviceAddressSpace(&handle, address, size);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return dd::ResultOutOfMemory();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return dd::ResultOutOfResource();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    *pHandle = static_cast<nnHandle>(handle).value;
    NN_RESULT_SUCCESS;

#else   // DeviceAddressSpace is not supported

    return ResultNotSupported();

#endif
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトの破棄
//
void DeviceAddressSpaceImplByHorizon::Close(DeviceAddressSpaceHandle deviceHandle) NN_NOEXCEPT
{
    auto handle = nn::svc::Handle(deviceHandle);
    if ((handle == svc::PSEUDO_HANDLE_CURRENT_THREAD) ||
        (handle == svc::PSEUDO_HANDLE_CURRENT_PROCESS))
    {
        // 疑似ハンドルの場合は何もしない
        return;
    }

    auto result = svc::CloseHandle(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（Aligned 版）
//
Result DeviceAddressSpaceImplByHorizon::MapAligned(
                        DeviceAddressSpaceHandle deviceHandle,
                        ProcessHandle            processHandle,
                        uint64_t                 processAddress,
                        size_t                   size,
                        DeviceVirtualAddress     deviceAddress,
                        nn::dd::MemoryPermission devicePermission) NN_NOEXCEPT
{
    // svc 呼出し前にチェックして、メッセージを出す。
    NN_ABORT_UNLESS((processAddress & 0x3fffff) == (deviceAddress & 0x3fffff),
        NN_TEXT("processAddress と deviceAddress の下位 22bit が一致していません"));

    auto result = svc::MapDeviceAddressSpaceAligned(
                                             nn::svc::Handle(deviceHandle),
                                             nn::svc::Handle(processHandle),
                                             processAddress,
                                             size,
                                             deviceAddress,
                    static_cast<nn::svc::MemoryPermission>(devicePermission));

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return dd::ResultOutOfMemory();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return dd::ResultOutOfResource();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（NotAligned 版）
//
Result DeviceAddressSpaceImplByHorizon::MapNotAligned(
                        DeviceAddressSpaceHandle deviceHandle,
                        ProcessHandle            processHandle,
                        uint64_t                 processAddress,
                        size_t                   size,
                        DeviceVirtualAddress     deviceAddress,
                        nn::dd::MemoryPermission devicePermission) NN_NOEXCEPT
{
    auto result = svc::MapDeviceAddressSpaceByForce(
                                            nn::svc::Handle(deviceHandle),
                                            nn::svc::Handle(processHandle),
                                            processAddress,
                                            size,
                                            deviceAddress,
                    static_cast<nn::svc::MemoryPermission>(devicePermission));

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return dd::ResultOutOfMemory();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return dd::ResultOutOfResource();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（Partially 版）
//
Result DeviceAddressSpaceImplByHorizon::MapPartially(
                        size_t*                  outMappedSize,
                        DeviceAddressSpaceHandle deviceHandle,
                        ProcessHandle            processHandle,
                        uint64_t                 processAddress,
                        size_t                   size,
                        DeviceVirtualAddress     deviceAddress,
                        nn::dd::MemoryPermission devicePermission) NN_NOEXCEPT
{
    size_t mapSize;
    auto result = svc::MapDeviceAddressSpace(&mapSize,
                                             nn::svc::Handle(deviceHandle),
                                             nn::svc::Handle(processHandle),
                                             processAddress,
                                             size,
                                             deviceAddress,
                    static_cast<nn::svc::MemoryPermission>(devicePermission));

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return dd::ResultOutOfMemory();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return dd::ResultOutOfResource();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    *outMappedSize = mapSize;
    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間からのアンマップ
//
void DeviceAddressSpaceImplByHorizon::Unmap(
                        DeviceAddressSpaceHandle deviceHandle,
                        ProcessHandle            processHandle,
                        uint64_t                 processAddress,
                        size_t                   size,
                        DeviceVirtualAddress     deviceAddress ) NN_NOEXCEPT
{
    Result result = svc::UnmapDeviceAddressSpace(nn::svc::Handle(deviceHandle),
                                                 nn::svc::Handle(processHandle),
                                                 processAddress,
                                                 size,
                                                 deviceAddress);

    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトをデバイスグループにアタッチ
//
Result DeviceAddressSpaceImplByHorizon::Attach(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT
{
    auto result = svc::AttachDeviceAddressSpace(
                                static_cast<nn::svc::DeviceName>(deviceName),
                                nn::svc::Handle(deviceSpace->_deviceHandle));

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return dd::ResultOutOfMemory();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_SDK_LOG("DeviceName= %d\n", deviceName);
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトをデバイスグループからデタッチ
//
void DeviceAddressSpaceImplByHorizon::Detach(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT
{
    Result result = svc::DetachDeviceAddressSpace(
                                static_cast<nn::svc::DeviceName>(deviceName),
                                nn::svc::Handle(deviceSpace->_deviceHandle));

    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}


//-----------------------------------------------------------------------------

}}} // namespace nn::dd::detail

