﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Main DHCP Client Daemon
 *
 * @details Top level
 */

#pragma once

#include "dhcpc_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace dhcpc {

// This is the main top level DHCP client daemon
class DhcpcMain
{
public:
    //  API for main DHCP client daemon (all instances)
    Result Initialize();
    Result Finalize();

    //  API for DHCP client interfaces
    Result InitializeClientInterface(int32_t ifIndex,
                                     const InterfaceConfigType *pIfConfig);
    Result FinalizeClientInterface(int32_t ifIndex, uint32_t options);
    Result GetOptionData(int32_t ifIndex, DhcpProtOption option,
                         void *pRetData, size_t bufferSize, size_t *pRetSize);
    Result GetState(int32_t ifIndex, char *pReturnedStateName, size_t stateNameSizeLimit);

    Result ForceRenew(int32_t ifIndex);
    // Subsystems
    FdMonitor                   m_FdMonitor;
    LocalEventManager           m_LocalEventManager;


    DhcpcMain()
        : m_MainThreadRun(false)
        , m_MessageQueue(m_MessageQueueStorage, MESSAGE_QUEUE_LENGTH)
    {
    }
    ~DhcpcMain()
    {
    }

private:
    enum MainThreadPort
    {
        MainThreadPort_ReEvaluate = 0,
        MainThreadPort_LocalEventTimer,
        MainThreadPort_Message,
        MainThreadPort_MaxPorts
    };
    static const int32_t MESSAGE_QUEUE_LENGTH = 32;


    Result DoRequest(DeferredRequest *pRequest);
    Result DispatchCommand(DeferredRequest *pDr);
    void MainThread();

    // Main thread
    bool                            m_MainThreadRun;
    NetworkThread                   m_MainThread;
    uintptr_t                       m_MessageQueueStorage[MESSAGE_QUEUE_LENGTH];
    nn::os::MessageQueue            m_MessageQueue;
#ifdef NN_SDK_BUILD_DEBUG
    NN_ALIGNAS(nn::os::StackRegionAlignment) uint8_t m_MainThreadStack[1024 * 32];
#else
    NN_ALIGNAS(nn::os::StackRegionAlignment) uint8_t m_MainThreadStack[1024 * 16];
#endif
    static void MainThreadEntry(void *pThis) { ((DhcpcMain *)pThis)->MainThread();}
    nn::os::MultiWaitType           m_MultiWait;
    nn::os::MultiWaitHolderType     m_Holder[MainThreadPort_MaxPorts];
    nn::os::EventType               m_ReEvaluateEvent;

    // Interfaces
    Interface                       m_Interfaces[Config_MaxIf];
};

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn
