﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcpc_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace dhcpc {


//--------------------------------------------------------------------------
//  API for main DHCP client daemon (all instances)
//--------------------------------------------------------------------------
Result DhcpcMain::Initialize()
{
    Result result = ResultSuccess();

    nn::os::InitializeMultiWait(&m_MultiWait);

    // For all ports on which main thread may block
    for (int portIndex = 0; portIndex < MainThreadPort_MaxPorts; portIndex++)
    {
        switch (portIndex)
        {
        case MainThreadPort_ReEvaluate:
            nn::os::InitializeEvent(&m_ReEvaluateEvent, false, nn::os::EventClearMode_AutoClear);
            nn::os::InitializeMultiWaitHolder(&m_Holder[portIndex], &m_ReEvaluateEvent);
            break;
        case MainThreadPort_LocalEventTimer:
            m_LocalEventManager.Initialize(&m_Holder[portIndex]);
            break;
        case MainThreadPort_Message:
            nn::os::InitializeMultiWaitHolder(&m_Holder[portIndex], m_MessageQueue.GetBase(),
                                              nn::os::MessageQueueWaitType_WaitForNotEmpty);
            break;
        default:
            break;
        }
        nn::os::SetMultiWaitHolderUserData(&m_Holder[portIndex], portIndex);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[portIndex]);
    }
    CreateNetworkThread(&m_MainThread, MainThreadEntry,
                        this, m_MainThreadStack,
                        sizeof(m_MainThreadStack),
                        NN_SYSTEM_THREAD_PRIORITY(socket, DhcpClient), -1);
    SetNetworkThreadNamePointer(&m_MainThread,
                                NN_SYSTEM_THREAD_NAME(socket, DhcpClient));
    m_MainThreadRun = true;

    do
    {
        DHCPC_BREAK_UPON_ERROR(m_FdMonitor.Initialize(this));
        StartNetworkThread(&m_MainThread);
    }while (false);

    return result;
}

Result DhcpcMain::Finalize()
{
    Result result = ResultSuccess();

    // Initiate thread exit
    m_MainThreadRun = false;
    nn::os::SignalEvent(&m_ReEvaluateEvent);

    // Synchronize with interrupt thread exit
    WaitNetworkThread(&m_MainThread);

    nn::os::FinalizeMultiWait(&m_MultiWait);

    return result;
}

//--------------------------------------------------------------------------
//  API for DHCP client interfaces
//--------------------------------------------------------------------------
Result DhcpcMain::InitializeClientInterface(int32_t ifIndex,
                                            const InterfaceConfigType *pIfConfig)
{
    DeferredRequest request(DeferredRequest::Command_Initialize, ifIndex);
    request.m_Params.initialize.pIfConfig = pIfConfig;
    return DoRequest(&request);
}

Result DhcpcMain::FinalizeClientInterface(int32_t ifIndex, uint32_t options)
{
    Result result;

    DeferredRequest stopRequest(DeferredRequest::Command_StopIf, ifIndex);
    stopRequest.m_Params.stopIf.options = options;

    if ((result = DoRequest(&stopRequest)).IsSuccess())
    {
        DeferredRequest finalizeRequest(DeferredRequest::Command_FinalizeIf, ifIndex);
        result = DoRequest(&finalizeRequest);
    }

    return result;
}

Result DhcpcMain::GetOptionData(int32_t ifIndex, DhcpProtOption option,
                                void *pRetData, size_t bufferSize, size_t *pRetSize)
{
    DeferredRequest request(DeferredRequest::Command_GetOptionData, ifIndex);
    request.m_Params.getOptionData.option     = option;
    request.m_Params.getOptionData.pRetData   = pRetData;
    request.m_Params.getOptionData.bufferSize = bufferSize;
    request.m_Params.getOptionData.pRetSize   = pRetSize;
    return DoRequest(&request);
}

Result DhcpcMain::GetState(int32_t ifIndex, char *pReturnedStateName, size_t stateNameSizeLimit)
{
    DeferredRequest request(DeferredRequest::Command_GetState, ifIndex);
    request.m_Params.getState.pReturnedStateName = pReturnedStateName;
    request.m_Params.getState.stateNameSizeLimit = stateNameSizeLimit;
    return DoRequest(&request);
}

Result DhcpcMain::ForceRenew(int32_t ifIndex)
{
    DeferredRequest request(DeferredRequest::Command_ForceRenew, ifIndex);
    return DoRequest(&request);
}

//--------------------------------------------------------------------------
//  Internal implementation
//--------------------------------------------------------------------------
Result DhcpcMain::DoRequest(DeferredRequest *pDr)
{
    Result result;

    // Defer incoming external requests to the main thread,
    // greatly simplifying internal DHCP client logic
    if (nn::os::GetCurrentThread() != &m_MainThread.sigloThread)
    {
        m_MessageQueue.Send(reinterpret_cast<uintptr_t>(pDr));
        result = pDr->BlockOnRequest();
    }
    // Else, core DHCP interface is calling back into itself
    // from the main thread, no need to defer
    else
    {
        result = DispatchCommand(pDr);
    }

    return result;
}

Result DhcpcMain::DispatchCommand(DeferredRequest *pDr)
{
    Result result = ResultSuccess();

    if ((pDr->m_IfIndex >= 0) && (pDr->m_IfIndex < static_cast<int32_t>(Config_MaxIf)))
    {
        Interface *pI = m_Interfaces + pDr->m_IfIndex;
        DeferredRequest::Params *pP = &pDr->m_Params;
        switch (pDr->m_Command)
        {
        case DeferredRequest::Command_Initialize:
            new(pI) Interface();
            result = pI->Initialize(this, pP->initialize.pIfConfig);
            break;
        case DeferredRequest::Command_StopIf:
            result = pI->Stop(pDr);
            break;
        case DeferredRequest::Command_FinalizeIf:
            if ((result = pI->Finalize()).IsSuccess())
            {
                pI->~Interface();
            }
            break;
        case DeferredRequest::Command_GetState:
            result = pI->GetState(pP->getState.pReturnedStateName,
                                  pP->getState.stateNameSizeLimit);
            break;
        case DeferredRequest::Command_GetOptionData:
            result = pI->GetOptionData(pP->getOptionData.option,
                                       pP->getOptionData.pRetData,
                                       pP->getOptionData.bufferSize,
                                       pP->getOptionData.pRetSize);
            break;
        case DeferredRequest::Command_ForceRenew:
            result = pI->ForceRenew();
            break;
        default:
            DHCPC_ABORT("m_Command = %d", pDr->m_Command);
            break;
        }
    }
    else
    {
        result = ResultIfInvalid();
    }

    return result;
}

void DhcpcMain::MainThread()
{
    while (m_MainThreadRun)
    {
        Result result = ResultSuccess();
        int32_t portIndex = -1;
        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&m_MultiWait);
        portIndex = GetMultiWaitHolderUserData(holder);
        {
            switch (portIndex)
            {
            case MainThreadPort_ReEvaluate:
                break;
            case MainThreadPort_LocalEventTimer:
                m_LocalEventManager.DispatchTimedEvents();
                break;
            case MainThreadPort_Message:
                uintptr_t msg;

                while (m_MessageQueue.TryReceive(&msg))
                {
                    DeferredRequest *pDr = reinterpret_cast<DeferredRequest *>(msg);
                    result = DispatchCommand(pDr);
                    if (!DHCPC_COMPARE_RESULT(result, ResultDeferred()))
                    {
                        pDr->CompleteRequest(result);
                    }
                }
                break;
            default:
                break;
            }
            m_LocalEventManager.DispatchEvents();
        }
    }
}



} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn




