﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cfg_PrivateIncludes.h"
#include <nn/svc/svc_Base.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <if_types.h>

namespace nn { namespace bsdsocket { namespace cfg {

enum
{
    NN_BSDSOCKET_CFG_CTL_NET = 4,
};

namespace {
    Result RouteMessage(nn::socket::RtMsgHdr **pOutRouteMessageHeader, nn::socket::RtMsgType command, nn::socket::SockAddrIn *pInputSocketAddress, nn::socket::SockAddrDl *pInputLinkLayerSocketAddress, nn::socket::RtFlag flags);
    void PrintEntry(nn::socket::SockAddrDl *pInputLinkLayerSocketAddress, nn::socket::SockAddrIn *addr, nn::socket::RtMsgHdr *pRouteMessageHeader);
    bool IsValidType(int type);
    void PrintEtherAddr(const nn::socket::EtherAddr *addr);
    Result InterfaceIndexToName(uint32_t index, char * interfaceName, size_t bufferSize);
    Result GetArpTable(char * tableBuffer, size_t * tableSize);
    Result DeleteAllEntries(const char * ifName);
}

Result Arp::AddStaticArpEntry(nn::socket::InAddr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize)
{
    nn::socket::SockAddrIn *pArpSocketAddress;
    nn::socket::SockAddrIn inputSocketAddress;
    nn::socket::SockAddrDl *pArpLinkLayerSocketAddress;
    nn::socket::RtMsgHdr *pRouteMessageHeader;
    nn::socket::EtherAddr *pArpEthernetAddress;
    nn::socket::SockAddrDl arpLinkLayerSocketAddress;

    memset(&arpLinkLayerSocketAddress, 0, sizeof(arpLinkLayerSocketAddress));
    arpLinkLayerSocketAddress.sdl_len = sizeof(arpLinkLayerSocketAddress);
    arpLinkLayerSocketAddress.sdl_family = nn::socket::Family::Af_Link;

    memset(&inputSocketAddress, 0, sizeof(inputSocketAddress));
    inputSocketAddress.sin_family = nn::socket::Family::Af_Inet;
    inputSocketAddress.sin_len = sizeof(inputSocketAddress);
    inputSocketAddress.sin_addr = ipAddress;

    pArpEthernetAddress = reinterpret_cast<nn::socket::EtherAddr *>(nn::socket::LlAddr(&arpLinkLayerSocketAddress));

    const nn::socket::EtherAddr *pInputEthernetAddr = reinterpret_cast<const nn::socket::EtherAddr *>(pInputHardwareAddress);

    if (pInputEthernetAddr == NULL)
    {
        CFG_LOG_ERROR("pInputHardwareAddress must be non-null.\n");
        return ResultMacAddressSizeInvalid();
    }

    if(hardwareAddressSize != nn::socket::Ether_Addr_Len)
    {
        CFG_LOG_ERROR("Invalid Ethernet address length %d, expected %d\n", hardwareAddressSize, nn::socket::Ether_Addr_Len);
        return ResultMacAddressSizeInvalid();
    }
    else
    {
        *pArpEthernetAddress = *pInputEthernetAddr;
        arpLinkLayerSocketAddress.sdl_alen = nn::socket::Ether_Addr_Len;
    }

    Result result = RouteMessage(&pRouteMessageHeader, nn::socket::RtMsgType::Rtm_Get, &inputSocketAddress, &arpLinkLayerSocketAddress, nn::socket::RtFlag::Rtf_None);
    if (result.IsFailure())
    {
        return result;
    }
    pArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pRouteMessageHeader + 1);
    pArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(nn::socket::Sa_Size(pArpSocketAddress) + reinterpret_cast<char *>(pArpSocketAddress));

    if (pArpLinkLayerSocketAddress->sdl_family != nn::socket::Family::Af_Link ||
        (pRouteMessageHeader->rtm_flags & nn::socket::RtFlag::Rtf_Gateway) != nn::socket::RtFlag::Rtf_None ||
        !IsValidType(pArpLinkLayerSocketAddress->sdl_type))
    {
        CFG_LOG_ERROR("Ip address %s not on subnet.\n", nn::socket::InetNtoa(inputSocketAddress.sin_addr));
        return ResultInvalidIfAddress();
    }

    arpLinkLayerSocketAddress.sdl_type = pArpLinkLayerSocketAddress->sdl_type;
    arpLinkLayerSocketAddress.sdl_index = pArpLinkLayerSocketAddress->sdl_index;
    if ((result = RouteMessage(NULL, nn::socket::RtMsgType::Rtm_Add, &inputSocketAddress, &arpLinkLayerSocketAddress, nn::socket::RtFlag::Rtf_None)).IsSuccess())
    {
        return ResultSuccess();
    }
    else
    {
        CFG_LOG_ERROR("Failed to Add ARP entry for %s at ", nn::socket::InetNtoa(inputSocketAddress.sin_addr));
        PrintEtherAddr(reinterpret_cast<const nn::socket::EtherAddr *>(pInputHardwareAddress));
        CFG_LOG_NOPREFIX("\n");
        return result;
    }

}

Result Arp::RemoveArpEntry(nn::socket::InAddr ipAddress)
{
    nn::socket::SockAddrIn *pArpSocketAddress = NULL;
    nn::socket::SockAddrIn inputSocketAddress;
    nn::socket::RtMsgHdr *pRouteMessageHeader = NULL;
    nn::socket::SockAddrDl *pArpLinkLayerSocketAddress = NULL;
    nn::socket::SockAddrDl arpLinkLayerSocketAddress;
    nn::socket::RtFlag flags = nn::socket::RtFlag::Rtf_None;
    Result result;

    memset(&inputSocketAddress, 0, sizeof(inputSocketAddress));
    inputSocketAddress.sin_family = nn::socket::Family::Af_Inet;
    inputSocketAddress.sin_len = sizeof(inputSocketAddress);
    inputSocketAddress.sin_addr = ipAddress;

    memset(&arpLinkLayerSocketAddress, 0, sizeof(arpLinkLayerSocketAddress));
    arpLinkLayerSocketAddress.sdl_len = sizeof(arpLinkLayerSocketAddress);
    arpLinkLayerSocketAddress.sdl_family = nn::socket::Family::Af_Link;


    result = RouteMessage(&pRouteMessageHeader, nn::socket::RtMsgType::Rtm_Get, &inputSocketAddress, &arpLinkLayerSocketAddress,flags);
    if (result.IsFailure())
    {
        return result;
    }
    pArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pRouteMessageHeader + 1);
    pArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(nn::socket::Sa_Size(pArpSocketAddress) + reinterpret_cast<char *>(pArpSocketAddress));

    if (pArpLinkLayerSocketAddress->sdl_family == nn::socket::Family::Af_Link &&
        !(pRouteMessageHeader->rtm_flags & nn::socket::RtFlag::Rtf_Gateway) &&
        IsValidType(pArpLinkLayerSocketAddress->sdl_type))
    {
        pArpSocketAddress->sin_addr.S_addr = inputSocketAddress.sin_addr.S_addr;
    }

    pRouteMessageHeader->rtm_flags |= nn::socket::RtFlag::Rtf_LlData;
    if ((result = RouteMessage(NULL,nn::socket::RtMsgType::Rtm_Delete, pArpSocketAddress, NULL, flags)).IsSuccess())
    {
        return ResultSuccess();
    }
    else
    {
        CFG_LOG_WARN("Failed to Delete ARP Entry for %s.\n", nn::socket::InetNtoa(pArpSocketAddress->sin_addr));
        return result;
    }

}


Result Arp::LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, nn::socket::InAddr ipAddress)
{
    size_t arpTableSize = 0;
    char *pEnd, *pArpTable, *pNext;
    nn::socket::RtMsgHdr *pCurrentRouteMessageHeader;
    nn::socket::SockAddrIn *pCurrentArpSocketAddress;
    nn::socket::SockAddrDl *pCurrentArpLinkLayerSocketAddress;
    int foundArpEntry = 0;
    Result result;

    if (pOutputHardwareAddress == NULL)
    {
        CFG_LOG_ERROR("pOutputHardwareAddress must be non-null.\n");
        return ResultMacAddressSizeInvalid();
    }

    if (hardwareAddressSize != nn::socket::Ether_Addr_Len)
    {
        CFG_LOG_ERROR("Hardware Address Size Invalid. Expected: %d Recevied: %d\n", nn::socket::Ether_Addr_Len, hardwareAddressSize);
        return ResultMacAddressSizeInvalid();
    }

    if ((result = GetArpTable(NULL, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Size Retrival Failed.\n");
        return result;
    }
    if (arpTableSize == 0)
    {
        CFG_LOG_WARN("ARP Entry For %s Not Found. Arp Table Empty.\n", nn::socket::InetNtoa(ipAddress));
        return ResultArpEntryNotFound();
    }
    pArpTable = reinterpret_cast<char *>(malloc(arpTableSize));
    if (!pArpTable)
    {
        CFG_LOG_ERROR("Memory Allocation Failed.\n");
        return ResultMemAllocStackError();
    }
    else if ((result = GetArpTable(pArpTable, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Retrival Failed.\n");
        free(pArpTable);
        return result;
    }

    pEnd = pArpTable + arpTableSize;
    for (pNext = pArpTable; pNext < pEnd; pNext += pCurrentRouteMessageHeader->rtm_msglen)
    {
        pCurrentRouteMessageHeader = reinterpret_cast<nn::socket::RtMsgHdr *>(pNext);
        pCurrentArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pCurrentRouteMessageHeader + 1);
        pCurrentArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(reinterpret_cast<char *>(pCurrentArpSocketAddress) + nn::socket::Sa_Size(pCurrentArpSocketAddress));
        if (ipAddress.S_addr == pCurrentArpSocketAddress->sin_addr.S_addr && pCurrentArpLinkLayerSocketAddress->sdl_alen > 0)
        {
            memcpy(pOutputHardwareAddress, nn::socket::LlAddr(pCurrentArpLinkLayerSocketAddress), hardwareAddressSize);
            foundArpEntry = 1;
        }
    }

    free(pArpTable);
    if (foundArpEntry)
    {
        return ResultSuccess();
    }
    else
    {
        CFG_LOG_WARN("ARP Entry For %s Not Found.\n", nn::socket::InetNtoa(ipAddress));
        return ResultArpEntryNotFound();
    }
}

Result Arp::LookupArpEntry(nn::socket::InAddr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize)
{
    size_t arpTableSize = 0;
    char *pEnd, *pArpTable, *pNext;
    nn::socket::RtMsgHdr *pCurrentRouteMessageHeader;
    nn::socket::SockAddrIn *pCurrentArpSocketAddress;
    nn::socket::SockAddrDl *pCurrentArpLinkLayerSocketAddress;
    int foundArpEntry = 0;
    Result result;

    if (pOutputIpAddress == NULL)
    {
        CFG_LOG_ERROR("pOutputIpAddress must be non-null.\n");
        return ResultIpv4AddressSizeInvalid();
    }

    if (pHardwareAddress == NULL)
    {
        CFG_LOG_ERROR("pHardwareAddress must be non-null.\n");
        return ResultMacAddressSizeInvalid();
    }

    if (sizeof(pCurrentArpSocketAddress->sin_addr) != ipAddressSize)
    {
        CFG_LOG_ERROR("IP Address Size Mismatch. Expected: %d Recevied: %d\n", sizeof(pCurrentArpSocketAddress->sin_addr), ipAddressSize);
        return ResultIpv4AddressSizeInvalid();
    }

    if ((result = GetArpTable(NULL, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Size Retrival Failed.\n");
        return result;
    }
    pArpTable = reinterpret_cast<char *>(malloc(arpTableSize));
    if (!pArpTable)
    {
        CFG_LOG_ERROR("Memory Allocation Failed.\n");
        return ResultMemAllocStackError();
    }
    else if((result = GetArpTable(pArpTable, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Retrival Failed.\n");
        free(pArpTable);
        return result;
    }

    pEnd = pArpTable + arpTableSize;
    for (pNext = pArpTable; pNext < pEnd; pNext += pCurrentRouteMessageHeader->rtm_msglen)
    {
        pCurrentRouteMessageHeader = reinterpret_cast<nn::socket::RtMsgHdr *>(pNext);
        pCurrentArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pCurrentRouteMessageHeader + 1);
        pCurrentArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(reinterpret_cast<char *>(pCurrentArpSocketAddress) + nn::socket::Sa_Size(pCurrentArpSocketAddress));
        if (pCurrentArpLinkLayerSocketAddress->sdl_alen != 0 && pCurrentArpLinkLayerSocketAddress->sdl_alen != hardwareAddressSize)
        {
            CFG_LOG_ERROR("Hardware Address Size Mismatch. Expected: %d Recevied: %d\n", pCurrentArpLinkLayerSocketAddress->sdl_alen, hardwareAddressSize);
            free(pArpTable);
            return ResultMacAddressSizeInvalid();
        }
        else if (memcmp(pHardwareAddress, nn::socket::LlAddr(pCurrentArpLinkLayerSocketAddress), hardwareAddressSize) == 0 && pCurrentArpLinkLayerSocketAddress->sdl_alen > 0)
        {
            memcpy(pOutputIpAddress, &pCurrentArpSocketAddress->sin_addr, ipAddressSize);
            foundArpEntry = 1;
        }
    }

    free(pArpTable);
    if (foundArpEntry)
    {
        return ResultSuccess();
    }
    else
    {
        CFG_LOG_WARN("ARP Entry For ");
        PrintEtherAddr(reinterpret_cast<const nn::socket::EtherAddr *>(pHardwareAddress));
        CFG_LOG_NOPREFIX(" Not Found.\n");
        return ResultArpEntryNotFound();
    }
}

Result Arp::ClearArpEntries()
{
    return DeleteAllEntries(NULL);
}

Result Arp::ClearArpEntries(const char * pIfName)
{
    return DeleteAllEntries(pIfName);
}

Result Arp::PrintArpEntries()
{
    size_t arpTableSize = 0;
    char *pEnd, *pArpTable, *pNext;
    nn::socket::RtMsgHdr *pCurrentRouteMessageHeader;
    nn::socket::SockAddrIn *pCurrentArpSocketAddress;
    nn::socket::SockAddrDl *pCurrentArpLinkLayerSocketAddress;
    Result result;
    if ((result = GetArpTable(NULL, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Size Retrival Failed.\n");
        return result;
    }
    if (arpTableSize == 0)
    {
        CFG_LOG_INFO("Arp Table Empty.\n");
        return ResultSuccess();
    }
    pArpTable = reinterpret_cast<char *>(malloc(arpTableSize));
    if (!pArpTable)
    {
        return ResultMemAllocStackError();
    }
    if ((result = GetArpTable(pArpTable, &arpTableSize)).IsFailure())
    {
        CFG_LOG_ERROR("ARP Table Retrival Failed.\n");
        free(pArpTable);
        return result;
    }

    pEnd = pArpTable + arpTableSize;
    for (pNext = pArpTable; pNext < pEnd; pNext += pCurrentRouteMessageHeader->rtm_msglen)
    {
        pCurrentRouteMessageHeader = reinterpret_cast<nn::socket::RtMsgHdr *>(pNext);
        pCurrentArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pCurrentRouteMessageHeader + 1);
        pCurrentArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(reinterpret_cast<char *>(pCurrentArpSocketAddress) + nn::socket::Sa_Size(pCurrentArpSocketAddress));

        PrintEntry(pCurrentArpLinkLayerSocketAddress, pCurrentArpSocketAddress, pCurrentRouteMessageHeader);
    }
    free(pArpTable);
    return ResultSuccess();
}

namespace {

    Result DeleteAllEntries(const char * ifName)
    {
        size_t arpTableSize = 0;
        char *pEnd, *pArpTable, *pNext;
        nn::socket::RtMsgHdr *pCurrentRouteMessageHeader;
        nn::socket::SockAddrIn *pCurrentArpSocketAddress;
        nn::socket::SockAddrDl *pCurrentArpLinkLayerSocketAddress;
        Result result;
        char currentInterfaceName[nn::socket::IfNamSiz];
        if ((result = GetArpTable(NULL, &arpTableSize)).IsFailure())
        {
            CFG_LOG_ERROR("ARP Table Size Retrival Failed.\n");
            return result;
        }
        if (arpTableSize == 0)
        {
            CFG_LOG_INFO("Arp Table Empty.\n");
            return ResultSuccess();
        }
        pArpTable = reinterpret_cast<char *>(malloc(arpTableSize));
        if (!pArpTable)
        {
            CFG_LOG_ERROR("Memory Allocation Failed! Tried to allocate %d bytes.\n", arpTableSize);
            return ResultMemAllocStackError();
        }
        if ((result = GetArpTable(pArpTable, &arpTableSize)).IsFailure())
        {
            CFG_LOG_ERROR("ARP Table Retrival Failed.\n");
            free(pArpTable);
            return result;
        }
        pEnd = pArpTable + arpTableSize;
        for (pNext = pArpTable; pNext < pEnd; pNext += pCurrentRouteMessageHeader->rtm_msglen)
        {
            pCurrentRouteMessageHeader = reinterpret_cast<nn::socket::RtMsgHdr *>(pNext);
            pCurrentArpSocketAddress = reinterpret_cast<nn::socket::SockAddrIn *>(pCurrentRouteMessageHeader + 1);
            pCurrentArpLinkLayerSocketAddress = reinterpret_cast<nn::socket::SockAddrDl *>(reinterpret_cast<char *>(pCurrentArpSocketAddress) + nn::socket::Sa_Size(pCurrentArpSocketAddress));
            if (ifName == NULL || ((result = InterfaceIndexToName(pCurrentRouteMessageHeader->rtm_index, currentInterfaceName, nn::socket::IfNamSiz)).IsSuccess() && strncmp(currentInterfaceName, ifName, nn::socket::IfNamSiz) == 0))
            {
                result = Arp::RemoveArpEntry(pCurrentArpSocketAddress->sin_addr);
            }

            if (result.IsFailure())
            {
                free(pArpTable);
                return result;
            }

        }
        free(pArpTable);
        return ResultSuccess();
    }



    Result RouteMessage(nn::socket::RtMsgHdr **pOutRouteMessageHeader, nn::socket::RtMsgType command, nn::socket::SockAddrIn *pInputSocketAddress, nn::socket::SockAddrDl *pInputLinkLayerSocketAddress, nn::socket::RtFlag flags)
    {
        static int sequenceNumber;
        int returnVal;
        int routeMessageLength;
        nn::socket::SockAddrIn socketMask;
        nn::socket::SockAddrIn *pSocketMask = &socketMask;
        int socket = -1;

        static struct
        {
            nn::socket::RtMsgHdr m_rtm;
            char    m_space[512];
        } routemessage;

        nn::socket::RtMsgHdr *pRouteMessageHeader = &routemessage.m_rtm;
        char *cp = routemessage.m_space;

        socket = nn::socket::SocketExempt(nn::socket::Family::Pf_Route, nn::socket::Type::Sock_Raw, nn::socket::Protocol::IpProto_Ip);
        if (socket < 0)
        {
            nn::socket::Errno errorNum = nn::socket::GetLastError();
            switch (errorNum)
            {
            case nn::socket::Errno::ENoBufs:
                CFG_LOG_ERROR("No internal buffer space available, error num: %d\n", errorNum);
                return ResultMemAllocStackError();

            default:
                CFG_LOG_ERROR("Failed to open socket, error num: %d\n", errorNum);
                break;

            }
            return ResultSocketOpenError();
        }

        memset(&socketMask, 0, sizeof(socketMask));
        socketMask.sin_len = 8;
        socketMask.sin_addr.S_addr = 0xffffffff;
        do
        {
            /*
            * XXX nn::socket::RtMsgType::Rtm_Delete relies on a previous nn::socket::RtMsgType::Rtm_Get to fill the buffer
            * appropriately.
            */
            if (command == nn::socket::RtMsgType::Rtm_Delete)
            {
                break;
            }

            memset(&routemessage, 0, sizeof(routemessage));
            pRouteMessageHeader->rtm_flags = flags;
            pRouteMessageHeader->rtm_version = nn::socket::Rtm_Version;

            switch (command)
            {
            default:
                CFG_LOG_ERROR("ARP internal wrong cmd\n");
                nn::socket::Close(socket);
                return ResultUnknownStackError();

            case nn::socket::RtMsgType::Rtm_Add:
                pRouteMessageHeader->rtm_addrs |= nn::socket::RtaFlag::Rta_Gateway;
                pRouteMessageHeader->rtm_rmx.rmx_expire = 0;
                pRouteMessageHeader->rtm_inits = nn::socket::RtvFlag::Rtv_Expire;
                pRouteMessageHeader->rtm_flags |= (nn::socket::RtFlag::Rtf_Host | nn::socket::RtFlag::Rtf_Static | nn::socket::RtFlag::Rtf_LlData);
                // FALLTHROUGH
            case nn::socket::RtMsgType::Rtm_Get:
                pRouteMessageHeader->rtm_addrs |= nn::socket::RtaFlag::Rta_Dst;
            }
#define NEXTADDR(w, s)\
        do {\
            if ((s) != NULL && (pRouteMessageHeader->rtm_addrs & (w)) != nn::socket::RtaFlag::Rta_None) { \
                bcopy((s), cp, sizeof(*(s)));\
                cp += nn::socket::Sa_Size(s);\
            }\
        } while (0)

            NEXTADDR(nn::socket::RtaFlag::Rta_Dst, pInputSocketAddress);
            NEXTADDR(nn::socket::RtaFlag::Rta_Gateway, pInputLinkLayerSocketAddress);
            NEXTADDR(nn::socket::RtaFlag::Rta_NetMask, pSocketMask);

            pRouteMessageHeader->rtm_msglen = cp - reinterpret_cast<char *>(&routemessage);
        } while (false);

        routeMessageLength = pRouteMessageHeader->rtm_msglen;
        pRouteMessageHeader->rtm_seq = ++sequenceNumber;
        pRouteMessageHeader->rtm_type = command;
        if ((nn::socket::Write(socket, reinterpret_cast<char *>(&routemessage), routeMessageLength)) < 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            Result result = ResultSuccess();
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfInvalid();
                break;
            case nn::socket::Errno::ENoBufs:
                result = ResultMemAllocStackError();
                break;
            case nn::socket::Errno::EInval:
                CFG_LOG_WARN("ARP Entry For %s Not Found.\n", nn::socket::InetNtoa(pInputSocketAddress->sin_addr));
                nn::socket::Close(socket);
                return ResultArpEntryNotFound();
            case nn::socket::Errno::ESrch:
                // This error is OK if we are deleting
                if (command == nn::socket::RtMsgType::Rtm_Delete)
                {
                    break;
                }
                else
                {
                    CFG_LOG_ERROR("Interface does not exist.");
                    result = ResultIfInvalid();
                }
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
            if (result.IsFailure())
            {
                CFG_LOG_WARN("Routing socket Write() failed, errno=%d.\n", lastErrno);
                nn::socket::Close(socket);
                return result;
            }
        }

        do
        {
            returnVal = nn::socket::Read(socket, reinterpret_cast<char *>(&routemessage), sizeof(routemessage));
        } while (returnVal > 0 && (pRouteMessageHeader->rtm_seq != sequenceNumber));

        if (returnVal < 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            Result result;
            nn::socket::Close(socket);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            case nn::socket::Errno::ENoBufs:
                result = ResultMemAllocStackError();
                break;
            case nn::socket::Errno::EInval:
                CFG_LOG_WARN("ARP Entry For %s Not Found.\n", nn::socket::InetNtoa(pInputSocketAddress->sin_addr));
                return ResultArpEntryNotFound();
            default:
                result = ResultUnknownStackError();
                break;
            }
            CFG_LOG_WARN("Routing socket Read() failed, errno=%d.\n", lastErrno);
            return result;

        }
        if (pOutRouteMessageHeader)
        {
            *pOutRouteMessageHeader = pRouteMessageHeader;
        }
        nn::socket::Close(socket);
        return ResultSuccess();
    } //NOLINT(impl/function_size)

    bool IsValidType(int type)
    {

        switch (type) {
        case IFT_ETHER:
        case IFT_FDDI:
        case IFT_ISO88023:
        case IFT_ISO88024:
        case IFT_ISO88025:
        case IFT_L2VLAN:
        case IFT_BRIDGE:
            return (true);
        default:
            return (false);
        }
    }
    // gets arp table. if tableBuffer is null, returns size of arp table.
    Result GetArpTable(char * pOutArpTableBuffer, size_t * tableSize)
    {
        int mib[6];
        size_t needed = 0;
        char *pArpTableBuffer;
        int retVal;

        mib[0] = NN_BSDSOCKET_CFG_CTL_NET;
        mib[1] = static_cast<int>(nn::socket::Family::Pf_Route);
        mib[2] = 0;
        mib[3] = static_cast<int>(nn::socket::Family::Af_Inet);
        mib[4] = nn::socket::Net_Rt_Flags;
        mib[5] = static_cast<int>(nn::socket::RtFlag::Rtf_LlInfo);
        if (nn::socket::Sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            Result result;
            CFG_LOG_ERROR("route-sysctl-estimate\n");
            switch (lastErrno)
            {
            case nn::socket::Errno::ENoMem:
                result = ResultMemAllocStackError();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }

            return result;
        }
        if (needed == 0) /* empty table */
        {
            *tableSize = 0;
            return ResultSuccess();
        }

        if (pOutArpTableBuffer == NULL)
        {
            *tableSize = needed;
            return ResultSuccess();
        }

        pArpTableBuffer = NULL;
        for (;;)
        {
            pArpTableBuffer = reinterpret_cast<char *>(realloc(pArpTableBuffer, needed));
            if (pArpTableBuffer == NULL)
            {
                CFG_LOG_ERROR("could not reallocate memory\n");
                return ResultMemAllocStackError();
            }
            retVal = nn::socket::Sysctl(mib, 6, pArpTableBuffer, &needed, NULL, 0);
            if (retVal == 0 || nn::socket::GetLastError() != nn::socket::Errno::ENoMem)
            {
                break;
            }
            needed += needed / 8;
        }
        if (retVal == -1)
        {
            CFG_LOG_ERROR("Sysctl Failed! Error Num: %d\n", nn::socket::GetLastError());
            free(pArpTableBuffer);
            return ResultUnknownStackError();
        }


        if (*tableSize >= needed)
        {
            memcpy(pOutArpTableBuffer, pArpTableBuffer, needed);
        }
        else
        {
            CFG_LOG_ERROR("Insufficiant Arp Table Buffer Size! tableSize = %d, needed = %d\n", *tableSize, needed);
            free(pArpTableBuffer);
            return ResultUnknownStackError();
        }

        *tableSize = needed;
        free(pArpTableBuffer);
        return ResultSuccess();
    }




    //Get interface name from index
    Result InterfaceIndexToName(uint32_t index, char * interfaceName, size_t bufferSize)
    {
        nn::socket::IfReq ifRequestArray[16];
        nn::socket::IfConf ifConfig;
        nn::socket::IfReq *pCurrentIfRequest, *pIfEnd;

        int sock = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        if (sock < 0)
        {
            nn::socket::Errno errorNum = nn::socket::GetLastError();
            switch (errorNum)
            {
            case nn::socket::Errno::ENoBufs:
                CFG_LOG_ERROR("No internal buffer space available, error num: %d\n", errorNum);
                return ResultMemAllocStackError();

            default:
                CFG_LOG_ERROR("Failed to open socket, error num: %d\n", errorNum);
                break;

            }
            return ResultSocketOpenError();
        }

        ifConfig.ifc_len = sizeof(ifRequestArray);
        ifConfig.ifc_buf = (caddr_t)ifRequestArray;

        if ((nn::socket::Ioctl(sock, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfConf), reinterpret_cast<char *>(&ifConfig), sizeof(ifConfig)) < 0) ||
            (static_cast<size_t>(ifConfig.ifc_len) < sizeof(nn::socket::IfReq)))
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            Result result;

            nn::socket::Close(sock);

            CFG_LOG_ERROR("SiocGIfConf Ioctl Error! Interface name cannot be resolved. Error Num: %d\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENoMem:
                result = ResultMemAllocStackError();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
            *interfaceName = 0;
            return result;
        }

        pCurrentIfRequest = ifRequestArray;
        pIfEnd = reinterpret_cast<nn::socket::IfReq *>(reinterpret_cast<char *>(ifRequestArray) + ifConfig.ifc_len);
        while (pCurrentIfRequest < pIfEnd)
        {
            size_t nextDelta;
            if (nn::socket::Ioctl(sock, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfIndex), pCurrentIfRequest, sizeof(nn::socket::IfReq)) != 0)
            {
                nn::socket::Errno lastErrno = nn::socket::GetLastError();
                if (lastErrno != nn::socket::Errno::ENxIo)
                {
                    CFG_LOG_ERROR("SiocGIfIndex Ioctl Error: %d\n", lastErrno);
                    Result result;

                    nn::socket::Close(sock);

                    switch (lastErrno)
                    {
                    case nn::socket::Errno::ENoMem:
                        result = ResultMemAllocStackError();
                        break;
                    default:
                        result = ResultUnknownStackError();
                        break;
                    }
                    *interfaceName = 0;
                    return result;
                }
            }
            else if (pCurrentIfRequest->ifr_index == static_cast<short>(index))
            {
                nn::socket::Close(sock);
                strncpy(interfaceName, pCurrentIfRequest->ifr_name, bufferSize);
                return ResultSuccess();
            }

            /* Bump interface config pointer */
            nextDelta = pCurrentIfRequest->ifr_addr.sa_len + sizeof(pCurrentIfRequest->ifr_name);
            if (nextDelta < sizeof(*pCurrentIfRequest))
            {
                nextDelta = sizeof(*pCurrentIfRequest);
            }
            pCurrentIfRequest = reinterpret_cast<nn::socket::IfReq *>(reinterpret_cast<char *>(pCurrentIfRequest) + nextDelta);
        }
        nn::socket::Close(sock);
        CFG_LOG_WARN("Interface Index %d Not Found.\n", index);
        *interfaceName = 0;
        return ResultIfInvalid();

    }


    /*
    * Display an arp entry
    */

    void PrintEntry(nn::socket::SockAddrDl *pInputLinkLayerSocketAddress, nn::socket::SockAddrIn *addr, nn::socket::RtMsgHdr *pRouteMessageHeader)
    {
        char ifNameBuf[nn::socket::IfNamSiz];
        NN_UNUSED(addr);

        CFG_LOG_INFO("%s at ", nn::socket::InetNtoa(addr->sin_addr));
        if (pInputLinkLayerSocketAddress->sdl_alen)
        {
            PrintEtherAddr(reinterpret_cast<nn::socket::EtherAddr *>(nn::socket::LlAddr(pInputLinkLayerSocketAddress)));
        }
        else
            CFG_LOG_NOPREFIX("(incomplete)");


        if (InterfaceIndexToName(pInputLinkLayerSocketAddress->sdl_index, ifNameBuf, nn::socket::IfNamSiz).IsSuccess())
        {

            CFG_LOG_NOPREFIX(" on %s (interface# %d)", ifNameBuf, pInputLinkLayerSocketAddress->sdl_index);
        }
        else
        {
            CFG_LOG_NOPREFIX(" on (interface name not found)");
        }

        if (pRouteMessageHeader->rtm_rmx.rmx_expire == 0)
            CFG_LOG_NOPREFIX(" permanent");
        else
        {
            nn::TimeSpan currentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            uint32_t expireTime;
            if ((expireTime = pRouteMessageHeader->rtm_rmx.rmx_expire - currentTime.GetSeconds()) > 0)
                CFG_LOG_NOPREFIX(" expires in %d seconds", (int)expireTime);
            else
                CFG_LOG_NOPREFIX(" expired");
        }

        switch (pInputLinkLayerSocketAddress->sdl_type)
        {
        case IFT_ETHER:
            CFG_LOG_NOPREFIX(" [ethernet]");
            break;
        case IFT_FDDI:
            CFG_LOG_NOPREFIX(" [fddi]");
            break;
        case IFT_ATM:
            CFG_LOG_NOPREFIX(" [atm]");
            break;
        case IFT_L2VLAN:
            CFG_LOG_NOPREFIX(" [vlan]");
            break;
        case IFT_IEEE1394:
            CFG_LOG_NOPREFIX(" [firewire]");
            break;
        case IFT_BRIDGE:
            CFG_LOG_NOPREFIX(" [bridge]");
            break;
        default:
            break;
        }

        CFG_LOG_NOPREFIX("\n");

    }

    void PrintEtherAddr(const nn::socket::EtherAddr *addr)
    {
        NN_UNUSED(addr);
        CFG_LOG_NOPREFIX("%02x:%02x:%02x:%02x:%02x:%02x",
            addr->octet[0], addr->octet[1],
            addr->octet[2], addr->octet[3],
            addr->octet[4], addr->octet[5]);
    }
}
}}}
