﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_Utility.h"
#include "../kern_KScheduler.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include "../kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern { namespace svc {

namespace {
bool CheckCpuNoValue(int32_t cpuNo)
{
    return (0 <= cpuNo && cpuNo < KCPU::NUM_CORE);
}

Result SvcCreateThread(
    nn::svc::Handle*        pOut,
    nn::svc::ThreadFunc     f,
    uintptr_t               param,
    uintptr_t               pStackBottom,
    int32_t                 priority,
    int32_t                 cpuNo )
{
    KProcess& process = GetCurrentProcess();
    if (cpuNo == nn::svc::IdealCoreUseProcessValue)
    {
        cpuNo = process.GetIdealProcessor();
    }

    if (!CheckCpuNoValue(cpuNo))
    {
        return nn::svc::ResultInvalidCoreNumber();
    }
    Bit64 coreMask = process.GetCoreMask();
    if ((coreMask | (1ull << cpuNo)) != coreMask)
    {
        return nn::svc::ResultInvalidCoreNumber();
    }

    if (!(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority))
    {
        return nn::svc::ResultInvalidPriority();
    }
    if (!process.CheckThreadPriority(priority))
    {
        return nn::svc::ResultInvalidPriority();
    }

    // worker thread の処理を待つ
    KScopedResourceLimitTester threadTester(&process, nn::svc::LimitableResource_ThreadCountMax, 1, KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromMilliSeconds(100)));
    if (threadTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KThread* pThread = KThread::Create();
    if (!pThread)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    Result result;
    {
        KScopedLightLock procLocker(&process.GetStateMutex());
        result = InitializeThreadForUser(
                pThread,
                reinterpret_cast<ThreadFunc>(f),
                param,
                pStackBottom,
                priority,
                cpuNo,
                &process );
    }
    if (result.IsFailure())
    {
        return result;
    }
    threadTester.Accepted();

    pThread->GetContext()->CloneFpuStatus();

    result = KThread::Register(pThread);
    if (result.IsFailure())
    {
        return result;
    }

    // カレントプロセスのKHandleTable取得
    // KHandleTableに登録
    KHandleTable& pHandleTable = process.GetHandleTable();
    result = pHandleTable.Add(pOut, pThread);

    return result;
}

Result SvcStartThread(nn::svc::Handle thread)
{
    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), thread);

    if (!pThread)
    {
        NN_WARNING(false, "Failed to get thread object.(handle=%d)", thread.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    Result result = pThread->Run();
    if (result.IsSuccess())
    {
        // ExitThread との対応のため 成功時に Open する
        pThread->Open();
    }
    return result;
}

void SvcSleepThread(int64_t ns)
{
    if (NN_LIKELY(ns > 0))
    {
        int64_t timeout;

        nn::svc::Tick offset(TimeSpan::FromNanoSeconds(ns));
        if (NN_LIKELY(offset > 0))
        {
            timeout = KHardwareTimer::GetTick() + offset + 2;
            if (NN_UNLIKELY(timeout <= 0))
            {
                timeout = INT64_MAX;
            }
        }
        else
        {
            timeout = INT64_MAX;
        }

        GetCurrentThread().Sleep(timeout);
    }
    else if (ns == nn::svc::YieldType_WithoutCoreMigration)
    {
        KScheduler::YieldWithoutCoreMigration();
    }
    else if (ns == nn::svc::YieldType_WithCoreMigration)
    {
        KScheduler::YieldWithCoreMigration();
    }
    else if (ns == nn::svc::YieldType_ToAnyThread)
    {
        KScheduler::SwitchToAnyThread();
    }
    else
    {
        // 他の負値は何もしない
    }
}

void SvcExitThread()
{
    GetCurrentThread().Exit();

    NN_KERNEL_PANIC("exited thread seems to be scheduled?");
}

Result SvcGetThreadId(Bit64* pOut, nn::svc::Handle hThread)
{
    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), hThread);

    if (!pThread)
    {
        NN_WARNING(false, "Failed to get thread object.(handle=%d)", hThread.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    *pOut = pThread->GetId();

    return ResultSuccess();
}

Result SvcGetThreadPriority(int32_t* pOut, nn::svc::Handle hThread)
{
    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), hThread);

    if (!pThread)
    {
        NN_WARNING(false, "Failed to get thread object.(handle=%d)", hThread.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    *pOut = pThread->GetPriority();

    return ResultSuccess();
}


Result SvcSetThreadPriority(nn::svc::Handle hThread, int32_t priority)
{
    KProcess& process = GetCurrentProcess();
    if (!(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority))
    {
        return nn::svc::ResultInvalidPriority();
    }
    if (!process.CheckThreadPriority(priority))
    {
        return nn::svc::ResultInvalidPriority();
    }

    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(process.GetHandleTable(), hThread);
    if (!pThread)
    {
        NN_WARNING(false, "Failed to get thread object.(handle=%d)", hThread.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    pThread->SetPriority(priority);
    return ResultSuccess();
}

Result SvcGetThreadCoreMask(int32_t* pIdealCore, nn::Bit64* pAffinityMask, nn::svc::Handle thread)
{
    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), thread);
    if (!pThread)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    return pThread->GetCoreMask(pIdealCore, pAffinityMask);
}

Result SvcSetThreadCoreMask(nn::svc::Handle thread, int32_t idealCore, Bit64 affinityMask)
{
    if (idealCore == nn::svc::IdealCoreUseProcessValue)
    {
        idealCore = GetCurrentProcess().GetIdealProcessor();
        affinityMask = (1ull << idealCore);
    }
    else
    {
        Bit64 coreMask = GetCurrentProcess().GetCoreMask();
        if ((coreMask | affinityMask) != coreMask)
        {
            return nn::svc::ResultInvalidCoreNumber();
        }
        if (affinityMask == 0)
        {
            return nn::svc::ResultInvalidCombination();
        }

        if (CheckCpuNoValue(idealCore))
        {
            if (!((1ull << idealCore) & affinityMask))
            {
                return nn::svc::ResultInvalidCombination();
            }
        }
        else if (idealCore != nn::svc::IdealCoreNoUpdate && idealCore != nn::svc::IdealCoreDontCare)
        {
            return nn::svc::ResultInvalidCoreNumber();
        }
    }

    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), thread);
    if (!pThread)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    return pThread->SetCoreMask(idealCore, affinityMask);
}

int32_t SvcGetCurrentProcessorNumber()
{
    return GetCurrentCpuNo();
}

Result SvcSetThreadActivity(nn::svc::Handle handle, nn::svc::ThreadActivity activity)
{
    switch (activity)
    {
    case nn::svc::ThreadActivity_Runnable:
    case nn::svc::ThreadActivity_Paused:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    KThread* pThread = GetCurrentProcess().GetHandleTable().GetObject<KThread>(handle);
    if (!pThread)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KThread> autoCloser(pThread);
    if (pThread->GetParentPointer() != &GetCurrentProcess())
    {
        return nn::svc::ResultInvalidHandle();
    }
    if (pThread == &GetCurrentThread())
    {
        return nn::svc::ResultBusy();
    }

    return pThread->SetActivity(activity);
}

Result SvcGetThreadContext3(KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle handle)
{
    KThread* pThread = GetCurrentProcess().GetHandleTable().GetObject<KThread>(handle);
    if (!pThread)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KThread> autoCloser(pThread);
    if (pThread->GetParentPointer() != &GetCurrentProcess())
    {
        return nn::svc::ResultInvalidHandle();
    }
    if (pThread == &GetCurrentThread())
    {
        return nn::svc::ResultBusy();
    }

    nn::svc::ThreadContext context = {};
    Result result = pThread->GetThreadContext3(&context);
    if (result.IsFailure())
    {
        return result;
    }

    result = pContext.CopyFrom(&context);
    if (result.IsFailure())
    {
        return nn::svc::ResultInvalidPointer();
    }

    return ResultSuccess();
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateThread32(nn::svc::Handle* pOut, nn::svc::ilp32::ThreadFunc f, uintptr_t param, uintptr_t pStackBottom, int32_t prio, int32_t cpuNo)
{
    Result result = SvcCreateThread(pOut, f, param, pStackBottom, prio, cpuNo);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartThread32(nn::svc::Handle thread)
{
    Result result = SvcStartThread(thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSleepThread32(int64_t ns)
{
    SvcSleepThread(ns);
    ClearSvcOutRegisters();
}
void SvcExitThread32()
{
    SvcExitThread();
}
Result SvcGetThreadId32(Bit64* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadId(pOut, hThread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadPriority32(int32_t* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadPriority(pOut, hThread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadPriority32(nn::svc::Handle hThread, int32_t prio)
{
    Result result = SvcSetThreadPriority(hThread, prio);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadCoreMask32(int32_t* pIdealCore, nn::Bit64* pAffinityMask, nn::svc::Handle thread)
{
    Result result = SvcGetThreadCoreMask(pIdealCore, pAffinityMask, thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadCoreMask32(nn::svc::Handle thread, int32_t idealCore, Bit64 affinityMask)
{
    Result result = SvcSetThreadCoreMask(thread, idealCore, affinityMask);
    ClearSvcOutRegistersReturnResult();
    return result;
}
int32_t SvcGetCurrentProcessorNumber32()
{
    int32_t coreNo = SvcGetCurrentProcessorNumber();
    ClearSvcOutRegistersReturnInt32();
    return coreNo;
}
Result SvcSetThreadActivity32(nn::svc::Handle handle, nn::svc::ThreadActivity activity)
{
    Result result = SvcSetThreadActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadContext332(KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle handle)
{
    Result result = SvcGetThreadContext3(pContext, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateThread64(nn::svc::Handle* pOut, nn::svc::lp64::ThreadFunc f, uintptr_t param, uintptr_t pStackBottom, int32_t prio, int32_t cpuNo)
{
    Result result = SvcCreateThread(pOut, f, param, pStackBottom, prio, cpuNo);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartThread64(nn::svc::Handle thread)
{
    Result result = SvcStartThread(thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSleepThread64(int64_t ns)
{
    SvcSleepThread(ns);
    ClearSvcOutRegisters();
}
void SvcExitThread64()
{
    SvcExitThread();
}
Result SvcGetThreadId64(Bit64* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadId(pOut, hThread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadPriority64(int32_t* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadPriority(pOut, hThread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadPriority64(nn::svc::Handle hThread, int32_t prio)
{
    Result result = SvcSetThreadPriority(hThread, prio);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadCoreMask64(int32_t* pIdealCore, nn::Bit64* pAffinityMask, nn::svc::Handle thread)
{
    Result result = SvcGetThreadCoreMask(pIdealCore, pAffinityMask, thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadCoreMask64(nn::svc::Handle thread, int32_t idealCore, Bit64 affinityMask)
{
    Result result = SvcSetThreadCoreMask(thread, idealCore, affinityMask);
    ClearSvcOutRegistersReturnResult();
    return result;
}
int32_t SvcGetCurrentProcessorNumber64()
{
    int32_t coreNo = SvcGetCurrentProcessorNumber();
    ClearSvcOutRegistersReturnInt32();
    return coreNo;
}
Result SvcSetThreadActivity64(nn::svc::Handle handle, nn::svc::ThreadActivity activity)
{
    Result result = SvcSetThreadActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadContext364(KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle handle)
{
    Result result = SvcGetThreadContext3(pContext, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateThread64From32(nn::svc::Handle* pOut, nn::svc::ilp32::ThreadFunc f, uintptr_t param, uintptr_t pStackBottom, int32_t prio, int32_t cpuNo)
{
    Result result = SvcCreateThread(pOut, static_cast<nn::svc::ThreadFunc>(f), param, pStackBottom, prio, cpuNo);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartThread64From32(nn::svc::Handle thread)
{
    Result result = SvcStartThread(thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSleepThread64From32(int64_t ns)
{
    SvcSleepThread(ns);
    ClearSvcOutRegisters();
}
void SvcExitThread64From32()
{
    SvcExitThread();
}
Result SvcGetThreadId64From32(Bit64* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadId(pOut, static_cast<nn::svc::Handle>(hThread));
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadPriority64From32(int32_t* pOut, nn::svc::Handle hThread)
{
    Result result = SvcGetThreadPriority(pOut, static_cast<nn::svc::Handle>(hThread));
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadPriority64From32(nn::svc::Handle hThread, int32_t prio)
{
    Result result = SvcSetThreadPriority(static_cast<nn::svc::Handle>(hThread), prio);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadCoreMask64From32(int32_t* pIdealCore, nn::Bit64* pAffinityMask, nn::svc::Handle thread)
{
    Result result = SvcGetThreadCoreMask(pIdealCore, pAffinityMask, thread);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetThreadCoreMask64From32(nn::svc::Handle thread, int32_t idealCore, Bit64 affinityMask)
{
    Result result = SvcSetThreadCoreMask(thread, idealCore, affinityMask);
    ClearSvcOutRegistersReturnResult();
    return result;
}
int32_t SvcGetCurrentProcessorNumber64From32()
{
    int32_t coreNo = SvcGetCurrentProcessorNumber();
    ClearSvcOutRegistersReturnInt32();
    return coreNo;
}
Result SvcSetThreadActivity64From32(nn::svc::Handle handle, nn::svc::ThreadActivity activity)
{
    Result result = SvcSetThreadActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadContext364From32(KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle handle)
{
    Result result = SvcGetThreadContext3(pContext, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
