﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

#ifdef NN_KERN_ENABLE_NX_PROFILER
#include "kern_SystemControl.h"
#include "../kern_InterruptManagerSelect.h"
#include "../kern_KThread.h"
#include "../kern_KProcess.h"
#include "../kern_KScheduler.h"
#include "../ip/ARM64/kern_ExceptionContext.h"
#endif

namespace nn { namespace kern { namespace svc {
namespace {
#ifdef NN_KERN_ENABLE_NX_PROFILER
template <typename T>
    T* GetContextAtSuperVisorStackBottom(void* pBottom)
{
    return reinterpret_cast<T*>(reinterpret_cast<uintptr_t>(pBottom) - sizeof(KThread::ParamsOnStack)) - 1;
}

Result SvcGetLastThreadInfoImpl(nn::svc::LastThreadContext* pContext, uintptr_t* pTlsAddr, nn::Bit32* pFlag)
{
    Result result;
    {
        KDisableInterrupt di;
        KThread* pThread = GetCoreScheduler().GetPrevThread();
        if (pThread)
        {
            if (GetCurrentProcessPointer() == pThread->GetParentPointer())
            {
                *pFlag = 0;
                const nn::kern::ARM64::ExceptionContext* pExcContext = GetContextAtSuperVisorStackBottom<nn::kern::ARM64::ExceptionContext>(pThread->GetKernelStackBottom());
                if ((pExcContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
                {
                    pContext->fp = pExcContext->x[11];
                    pContext->sp = pExcContext->x[13];
                    pContext->lr = pExcContext->x[14];
                    pContext->pc = pExcContext->pc;
                }
                else
                {
                    pContext->fp = pExcContext->x[29];
                    pContext->sp = pExcContext->sp;
                    pContext->lr = pExcContext->x[30];
                    pContext->pc = pExcContext->pc;
                }
                *pTlsAddr = GetAsInteger(pThread->GetThreadLocalRegionAddr());
                if (pThread->IsCallingSvc())
                {
                    *pFlag |= nn::svc::LastThreadInfoFlag_ThreadInSystemCall;
                }
                result = ResultSuccess();
            }
            else
            {
                result = nn::svc::ResultUnknownThread();
            }
        }
        else
        {
            result = nn::svc::ResultNoThread();
        }
    }

    return result;
}
#endif

Result SvcGetLastThreadInfo(nn::svc::LastThreadContext* pContext, uintptr_t* pTlsAddr, nn::Bit32* pFlag)
{
    Result result;
#ifdef NN_KERN_ENABLE_NX_PROFILER
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNoThread();
    }

    result = SvcGetLastThreadInfoImpl(pContext, pTlsAddr, pFlag);
#else
    NN_UNUSED(pContext);
    NN_UNUSED(pTlsAddr);
    NN_UNUSED(pFlag);
    result = nn::svc::ResultNotImplemented();
#endif

    return result;
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcGetLastThreadInfo32(nn::svc::ilp32::LastThreadContext* pContext, uintptr_t* pTlsAddr, nn::Bit32* pFlag)
{
    Result result = SvcGetLastThreadInfo(pContext, pTlsAddr, pFlag);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcGetLastThreadInfo64(nn::svc::lp64::LastThreadContext* pContext, uintptr_t* pTlsAddr, nn::Bit32* pFlag)
{
    Result result = SvcGetLastThreadInfo(pContext, pTlsAddr, pFlag);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcGetLastThreadInfo64From32(nn::svc::ilp32::LastThreadContext* pContext, uintptr_t* pTlsAddr, nn::Bit32* pFlag)
{
    nn::svc::LastThreadContext context = {};
    Result result = SvcGetLastThreadInfo(&context, pTlsAddr, pFlag);
    pContext->fp = context.fp;
    pContext->sp = context.sp;
    pContext->lr = context.lr;
    pContext->pc = context.pc;
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
