﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_KPort.h"
#include "../kern_KClientPort.h"
#include "../kern_KServerPort.h"
#include "../kern_KSession.h"
#include "../kern_KLightSession.h"
#include "../kern_KObjectName.h"
#include "../kern_KHandleTable.h"
#include "../kern_KProcess.h"
#include "../kern_MemoryCopySelect.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include "../kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern { namespace svc {
namespace {
Result SvcCreatePort(nn::svc::Handle* pOutServer, nn::svc::Handle* pOutClient, int32_t maxSessions, bool isLight, uintptr_t name)
{
    Result result;

    if (maxSessions <= 0)
    {
        return nn::svc::ResultOutOfRange();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KPort* pPort = KPort::Create();
    if (pPort)
    {
        pPort->Initialize(maxSessions, isLight, name);

        result = KPort::Register(pPort);

        if (result.IsSuccess())
        {
            result = handleTable.Add(pOutClient, &pPort->GetClientPort());

            if (result.IsSuccess())
            {
                result = handleTable.Add(pOutServer, &pPort->GetServerPort());
                if (result.IsFailure())
                {
                    handleTable.Remove(*pOutClient);
                }
            }
        }

        pPort->GetServerPort().Close();
        pPort->GetClientPort().Close();
    }
    else
    {
        result = nn::svc::ResultOutOfResource();
    }

    return result;
}

Result SvcManageNamedPort(nn::svc::Handle* pOutServer, KUserPointer<const char*> nameInUserSpace, int32_t maxSessions)
{
    Result result;
    char name[KObjectName::NAME_LENGTH] = {0};

    result = nameInUserSpace.CopyStringTo(name, sizeof(name));
    if (result.IsFailure())
    {
        return result;
    }
    if (name[KObjectName::NAME_LENGTH - 1])
    {
        return nn::svc::ResultOutOfRange();
    }
    if (maxSessions < 0)
    {
        return nn::svc::ResultOutOfRange();
    }

    if (maxSessions > 0)
    {
        KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

        KPort* pPort = KPort::Create();
        if (!pPort)
        {
            return nn::svc::ResultOutOfResource();
        }

        result = handleTable.Reserve(pOutServer);
        if (result.IsFailure())
        {
            pPort->Close();
            return result;
        }

        pPort->Initialize(maxSessions, false, 0);

        result = KPort::Register(pPort);

        if (result.IsSuccess())
        {
            handleTable.Register(*pOutServer, &pPort->GetServerPort());
            result = KObjectName::Name(&pPort->GetClientPort(), name);
            if (result.IsFailure())
            {
                handleTable.Remove(*pOutServer);
            }
        }
        else
        {
            handleTable.Unreserve(*pOutServer);
        }

        pPort->GetServerPort().Close();
        pPort->GetClientPort().Close();
    }
    else
    {
        NN_KERN_ASSERT(maxSessions == 0);
        result = KObjectName::Delete<KClientPort>(name);
        *pOutServer = nn::svc::Handle(0);
    }

    return result;
}

Result SvcConnectToNamedPort(nn::svc::Handle* pOut, KUserPointer<const char*> nameInUserSpace)
{
    Result result;
    char name[KObjectName::NAME_LENGTH] = {0};

    result = nameInUserSpace.CopyStringTo(name, sizeof(name));
    if (result.IsFailure())
    {
        return result;
    }
    if (name[KObjectName::NAME_LENGTH - 1])
    {
        return nn::svc::ResultOutOfRange();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KClientPort* pPort = KObjectName::Find<KClientPort>(name);
    if (!pPort)
    {
        return nn::svc::ResultNotFound();
    }
    KScopedAutoObject<KClientPort> autoCloser(pPort);

    result = handleTable.Reserve(pOut);
    if (result.IsFailure())
    {
        return result;
    }

    KClientSession* pSession;
    result = pPort->CreateSession(&pSession);
    if (result.IsFailure())
    {
        handleTable.Unreserve(*pOut);
        return result;
    }

    handleTable.Register(*pOut, pSession);
    pSession->Close();

    return result;
}

Result SvcConnectToPort(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result;

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KClientPort* pPort = handleTable.GetObject<KClientPort>(port);
    if (!pPort)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KClientPort> autoCloser(pPort);

    result = handleTable.Reserve(pOut);
    if (result.IsFailure())
    {
        return result;
    }

    if (pPort->IsLight())
    {
        KLightClientSession* pSession;
        result = pPort->CreateLightSession(&pSession);
        if (result.IsFailure())
        {
            handleTable.Unreserve(*pOut);
            return result;
        }

        handleTable.Register(*pOut, pSession);
        pSession->Close();
    }
    else
    {
        KClientSession* pSession;
        result = pPort->CreateSession(&pSession);
        if (result.IsFailure())
        {
            handleTable.Unreserve(*pOut);
            return result;
        }

        handleTable.Register(*pOut, pSession);
        pSession->Close();
    }

    return result;
}

Result SvcAcceptSession(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result;
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KServerPort* pPort = handleTable.GetObject<KServerPort>(port);
    if (!pPort)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KServerPort> autoCloser(pPort);

    result = handleTable.Reserve(pOut);
    if (result.IsFailure())
    {
        return result;
    }

    if (pPort->IsLight())
    {
        KLightServerSession* pSession = pPort->AcceptLightSession();
        if (!pSession)
        {
            handleTable.Unreserve(*pOut);
            return nn::svc::ResultNotFound();
        }

        handleTable.Register(*pOut, pSession);
        pSession->Close();
    }
    else
    {
        KServerSession* pSession = pPort->AcceptSession();
        if (!pSession)
        {
            handleTable.Unreserve(*pOut);
            return nn::svc::ResultNotFound();
        }

        handleTable.Register(*pOut, pSession);
        pSession->Close();
    }

    return result;
}

template <typename T>
Result SvcCreateSession(nn::svc::Handle* pServerSessionHandle, nn::svc::Handle* pClientSessionHandle, uintptr_t name)
{
    KScopedResourceLimitTester sessionTester(GetCurrentProcessPointer(), nn::svc::LimitableResource_SessionCountMax);
    if (sessionTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    T* pSession = T::Create();
    if (!pSession)
    {
        return nn::svc::ResultOutOfResource();
    }

    pSession->Initialize(nullptr, name);
    sessionTester.Accepted();

    T::Register(pSession);

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    Result result = handleTable.Add(pServerSessionHandle, &pSession->GetServerSession());
    if (result.IsSuccess())
    {
        result = handleTable.Add(pClientSessionHandle, &pSession->GetClientSession());
        if (result.IsFailure())
        {
            handleTable.Remove(*pServerSessionHandle);
        }
    }

    pSession->GetServerSession().Close();
    pSession->GetClientSession().Close();
    return result;
}

Result SvcCreateSession(nn::svc::Handle* pServerSessionHandle, nn::svc::Handle* pClientSessionHandle, bool isLight, uintptr_t name)
{
    if (isLight)
    {
        return SvcCreateSession<KLightSession>(pServerSessionHandle, pClientSessionHandle, name);
    }
    else
    {
        return SvcCreateSession<KSession>(pServerSessionHandle, pClientSessionHandle, name);
    }
}

}
#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreatePort32(nn::svc::Handle* pOutServer, nn::svc::Handle* pOutClient, int32_t maxSessions, bool isLight, uintptr_t name)
{
    Result result = SvcCreatePort(pOutServer, pOutClient, maxSessions, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcManageNamedPort32(nn::svc::Handle* pOutServer, KUserPointer<const char*> nameInUserSpace, int32_t maxSessions)
{
    Result result = SvcManageNamedPort(pOutServer, nameInUserSpace, maxSessions);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToNamedPort32(nn::svc::Handle* pOut, KUserPointer<const char*> nameInUserSpace)
{
    Result result = SvcConnectToNamedPort(pOut, nameInUserSpace);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToPort32(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcConnectToPort(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcAcceptSession32(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcAcceptSession(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateSession32(nn::svc::Handle* pServerSessionHandle, nn::svc::Handle* pClientSessionHandle, bool isLight, uintptr_t name)
{
    Result result = SvcCreateSession(pServerSessionHandle, pClientSessionHandle, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreatePort64(nn::svc::Handle* pOutServer, nn::svc::Handle* pOutClient, int32_t maxSessions, bool isLight, uintptr_t name)
{
    Result result = SvcCreatePort(pOutServer, pOutClient, maxSessions, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcManageNamedPort64(nn::svc::Handle* pOutServer, KUserPointer<const char*> nameInUserSpace, int32_t maxSessions)
{
    Result result = SvcManageNamedPort(pOutServer, nameInUserSpace, maxSessions);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToNamedPort64(nn::svc::Handle* pOut, KUserPointer<const char*> nameInUserSpace)
{
    Result result = SvcConnectToNamedPort(pOut, nameInUserSpace);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToPort64(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcConnectToPort(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcAcceptSession64(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcAcceptSession(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateSession64(nn::svc::Handle* pServerSessionHandle, nn::svc::Handle* pClientSessionHandle, bool isLight, uintptr_t name)
{
    Result result = SvcCreateSession(pServerSessionHandle, pClientSessionHandle, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreatePort64From32(nn::svc::Handle* pOutServer, nn::svc::Handle* pOutClient, int32_t maxSessions, bool isLight, uintptr_t name)
{
    Result result = SvcCreatePort(pOutServer, pOutClient, maxSessions, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcManageNamedPort64From32(nn::svc::Handle* pOutServer, KUserPointer<const char*> nameInUserSpace, int32_t maxSessions)
{
    Result result = SvcManageNamedPort(pOutServer, nameInUserSpace, maxSessions);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToNamedPort64From32(nn::svc::Handle* pOut, KUserPointer<const char*> nameInUserSpace)
{
    Result result = SvcConnectToNamedPort(pOut, nameInUserSpace);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcConnectToPort64From32(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcConnectToPort(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcAcceptSession64From32(nn::svc::Handle* pOut, nn::svc::Handle port)
{
    Result result = SvcAcceptSession(pOut, port);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateSession64From32(nn::svc::Handle* pServerSessionHandle, nn::svc::Handle* pClientSessionHandle, bool isLight, uintptr_t name)
{
    Result result = SvcCreateSession(pServerSessionHandle, pClientSessionHandle, isLight, name);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
