﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_KSynchronization.h"
#include "../kern_Utility.h"
#include "../kern_KEvent.h"
#include "../kern_KReadableEvent.h"
#include "../kern_KWritableEvent.h"
#include "../kern_KProcess.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

namespace nn { namespace kern { namespace svc {

namespace {
NN_FORCEINLINE Result SvcWaitSynchronizationImpl(int32_t* pIndex, int32_t numHandles, int64_t ns)
{
    int64_t timeout;
    if (ns > 0)
    {
        // 途中でオーバーフローして負になった場合、永久待ちになる。
        uint64_t tick = KHardwareTimer::GetTick();
        tick += nn::svc::Tick(TimeSpan::FromNanoSeconds(ns));
        tick += 2;
        timeout = tick;
    }
    else
    {
        timeout = ns;
    }
    return Kernel::GetSynchronization().Wait(pIndex, GetCurrentThread().GetSynchronizationObjectBuffer(), numHandles, timeout);
}

Result SvcWaitSynchronization(int32_t* pIndex, KUserPointer<const nn::svc::Handle*> userHandles, int32_t numHandles, int64_t ns)
{
    if (!(0 <= numHandles && numHandles <= nn::svc::ArgumentHandleCountMax))
    {
        return nn::svc::ResultOutOfRange();
    }

    KSynchronizationObject** objs = GetCurrentThread().GetSynchronizationObjectBuffer();
    nn::svc::Handle* pHandle = GetCurrentThread().GetHandleBuffer();
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    if (numHandles > 0)
    {
        if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(userHandles)), numHandles * sizeof(nn::svc::Handle)))
        {
            return nn::svc::ResultInvalidPointer();
        }
        if (userHandles.CopyArrayTo(pHandle, numHandles).IsFailure())
        {
            return nn::svc::ResultInvalidPointer();
        }

        if (!handleTable.GetObject<KSynchronizationObject>(objs, pHandle, numHandles))
        {
            return nn::svc::ResultInvalidHandle();
        }
    }

    Result result = SvcWaitSynchronizationImpl(pIndex, numHandles, ns);

    if (result <= nn::svc::ResultSessionClosed())
    {
        result = ResultSuccess();
    }

    for (int j = 0; j < numHandles; ++j)
    {
        objs[j]->Close();
    }

    return result;
}

Result SvcCloseHandle(nn::svc::Handle handle)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    if (handleTable.Remove(handle))
    {
        return ResultSuccess();
    }
    else
    {
        NN_WARNING(false, "Invalid handle: %08x", handle.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }
}

Result SvcCancelSynchronization(nn::svc::Handle handle)
{
    KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), handle);

    if (!pThread)
    {
        NN_WARNING(false, "Failed to get thread object.(handle=%d)", handle.GetPrintableBits());
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KThread> autoCloser(pThread);

    pThread->WaitCancel();

    return ResultSuccess();
}

Result SvcResetSignal(nn::svc::Handle handle)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KReadableEvent* pReadableEvent = handleTable.GetObject<KReadableEvent>(handle);
    if (pReadableEvent)
    {
        KScopedAutoObject<KReadableEvent> autoCloser(pReadableEvent);
        return pReadableEvent->Reset();
    }

    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, handle);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        return pProcess->Reset();
    }

    return nn::svc::ResultInvalidHandle();
}


}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcWaitSynchronization32( int32_t* pIndex, KUserPointer<const nn::svc::Handle*> userHandles, int32_t numHandles, int64_t ns )
{
    Result result = SvcWaitSynchronization(pIndex, userHandles, numHandles, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCloseHandle32(nn::svc::Handle handle)
{
    Result result = SvcCloseHandle(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCancelSynchronization32(nn::svc::Handle handle)
{
    Result result = SvcCancelSynchronization(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcResetSignal32(nn::svc::Handle handle)
{
    Result result = SvcResetSignal(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcWaitSynchronization64( int32_t* pIndex, KUserPointer<const nn::svc::Handle*> userHandles, int32_t numHandles, int64_t ns )
{
    Result result = SvcWaitSynchronization(pIndex, userHandles, numHandles, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCloseHandle64(nn::svc::Handle handle)
{
    Result result = SvcCloseHandle(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCancelSynchronization64(nn::svc::Handle handle)
{
    Result result = SvcCancelSynchronization(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcResetSignal64(nn::svc::Handle handle)
{
    Result result = SvcResetSignal(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcWaitSynchronization64From32( int32_t* pIndex, KUserPointer<const nn::svc::Handle*> userHandles, int32_t numHandles, int64_t ns )
{
    Result result = SvcWaitSynchronization(pIndex, userHandles, numHandles, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCloseHandle64From32(nn::svc::Handle handle)
{
    Result result = SvcCloseHandle(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCancelSynchronization64From32(nn::svc::Handle handle)
{
    Result result = SvcCancelSynchronization(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcResetSignal64From32(nn::svc::Handle handle)
{
    Result result = SvcResetSignal(handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
