﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>

#include "kern_Platform.h"
#include "kern_Utility.h"
#include "kern_KThread.h"
#include "kern_PageTableSelect.h"
#include "kern_KProcess.h"
#include "kern_Kernel.h"
#include "kern_KEvent.h"
#include "kern_KInterruptEvent.h"
#include "kern_KPort.h"
#include "kern_KSharedMemory.h"
#include "kern_KTransferMemory.h"
#include "kern_KDeviceAddressSpace.h"
#include "kern_KSession.h"
#include "kern_KLightSession.h"
#include "kern_KLinkedList.h"
#include "kern_KObjectName.h"
#include "kern_KResourceLimit.h"
#include "kern_KScopedResourceLimitTester.h"
#include "kern_KPageBuffer.h"
#include "kern_KMemoryLayout.h"

namespace nn {
    namespace kern {
        namespace {
            Result InitializeThread(
                    KThread*                pThread,
                    ThreadFunc                    f,
                    uintptr_t                 param,
                    KProcessAddress userStackBottom,
                    int32_t                    prio,
                    int32_t                  coreNo,
                    KProcess*               pParent,
                    KThread::Type              type)
            {
                // TORIAEZU:
#if  NN_KERN_HAS_MMU
                KProcessAddress addr;
                const KProcessAddress regionToBeMapped = KMemoryLayout::GetStackRegionBegin();
                const size_t          regionNumPages   = KMemoryLayout::GetStackRegionSize() / NN_KERN_FINEST_PAGE_SIZE;

                KPageBuffer* pPageBuffer = KPageBuffer::Allocate();
                if (!pPageBuffer)
                {
                    return nn::svc::ResultOutOfMemory();
                }

                Result result = Kernel::GetKernelPageTable().MapPages(
                        &addr,
                        NN_KERN_THREAD_SVC_STACK_NUM_PAGE,
                        NN_KERN_THREAD_SVC_STACK_SIZE,
                        pPageBuffer->GetPhysicalAddress(),
                        regionToBeMapped,
                        regionNumPages,
                        KMemoryState_Kernel,
                        KMemoryPermission_KernelReadWrite);

                if(result.IsFailure())
                {
                    KPageBuffer::Free(pPageBuffer);
                    return nn::svc::ResultOutOfMemory();
                }
#else
#error not implemented
#endif
                void* svStackBottom = GetUntypedPointer(addr + NN_KERN_FINEST_PAGE_SIZE * NN_KERN_THREAD_SVC_STACK_NUM_PAGE);

                result = pThread->Initialize(
                        f,
                        param,
                        svStackBottom,
                        userStackBottom,
                        prio,
                        coreNo,
                        pParent,
                        type);
                if (result.IsFailure())
                {
                    CleanupSuperVisorStack(svStackBottom);
                    return result;
                }

                return result;
            }
        }

        static_assert( sizeof(KPageBuffer) >= NN_KERN_THREAD_SVC_STACK_SIZE, "" );

        Result InitializeThreadForUser(
            KThread*            pThread,
            ThreadFunc                f,
            uintptr_t             param,
            KProcessAddress userStackBottom,
            int32_t                prio,
            int32_t              coreNo,
            KProcess*           pParent )
        {
            Result result;

            result = InitializeThread(pThread, f, param, userStackBottom, prio, coreNo, pParent, KThread::TYPE_USER);
#ifdef NN_KERN_STACK_FILL_PATTERN
            if (result.IsSuccess() && pParent->GetPageTable().StackFillPattern(userStackBottom).IsFailure())
            {
                NN_WARNING(false, "faild to fill user stack. TID=%lld", pThread->GetId());
            }
#endif
            return result;
        }

        void CleanupSuperVisorStack(void* svStackBottom)
        {
            uintptr_t addr = reinterpret_cast<uintptr_t>(svStackBottom) - NN_KERN_FINEST_PAGE_SIZE * NN_KERN_THREAD_SVC_STACK_NUM_PAGE;

// TORIAEZU:
#if NN_KERN_HAS_MMU
            KPhysicalAddress physAddr;
            Result result;

            NN_KERN_ABORT_UNLESS(Kernel::GetKernelPageTable().GetPhysicalAddress(&physAddr, addr));
            result = Kernel::GetKernelPageTable().UnmapPages(addr, NN_KERN_THREAD_SVC_STACK_NUM_PAGE, KMemoryState_Kernel);
            NN_KERN_ABORT_IF_FAILED(result);
            KPageBuffer::Free(KPageBuffer::FromPhysicalAddress(physAddr));
#else
#error not implemented
#endif
        }

        /*!
            @brief     きわめて優先度の高いリアルタイム属性のスレッド(優先度:0)を生成します。

            現在はハードウェア割り込みや、スケジューラー用ソフトウェア割り込みの
            後処理スレッド( DPC タスク )にのみ使用されています。

            @param[in]    pThread   KThread::Create()により得られるスレッドオブジェクト
            @param[in]    f         スレッドのエントリーポイント
            @param[in]    param     スレッドに渡す任意のパラメーター

        */
        Result InitializeThreadForRealTime(
                KThread*        pThread,
                ThreadFunc      f,
                uintptr_t       param)
        {
            return InitializeThread(pThread, f, param, 0, 0, GetCurrentCpuNo(), nullptr, KThread::TYPE_REALTIME);
        }

        Result InitializeThreadForKernel(
            KThread*            pThread,
            ThreadFunc          f,
            uintptr_t           param,
            int32_t             prio,
            int32_t             coreNo)
        {
            return InitializeThread(pThread, f, param, 0, prio, coreNo, nullptr, KThread::TYPE_KERNEL);
        }
    }
}
