﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KTransferMemory.h"
#include "kern_KProcess.h"
#include "kern_PageTableSelect.h"
#include "kern_KMemoryManager.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KTransferMemory);

KTransferMemory::KTransferMemory()
: m_pOwner(nullptr)
, m_Addr(0)
, m_OwnerProtection(nn::svc::MemoryPermission_None)
, m_IsInitialized(false)
, m_IsMapped(false)
{
    NN_KERN_THIS_ASSERT();
}

KTransferMemory::~KTransferMemory()
{
    NN_KERN_THIS_ASSERT();
}

Result KTransferMemory::Initialize(KProcessAddress addr, size_t size, nn::svc::MemoryPermission ownerProtection)
{
    NN_KERN_THIS_ASSERT();

    Result result;
    m_pOwner = GetCurrentProcessPointer();
    KProcessPageTable& pageTable = m_pOwner->GetPageTable();

    m_Memory.Initialize(pageTable.GetBlockInfoManager());
    result = pageTable.LockForTransferMemory(&m_Memory, addr, size,
            ConvertToKMemoryPermission(ownerProtection));
    if (result.IsFailure())
    {
        m_Memory.Finalize();
        return result;
    }

    m_pOwner->Open();
    m_OwnerProtection = ownerProtection;
    m_Addr = addr;
    m_IsInitialized = true;
    m_IsMapped = false;

    return ResultSuccess();
}

void KTransferMemory::Finalize()
{
    NN_KERN_THIS_ASSERT();

    size_t size = m_Memory.GetTotalNumPages() * NN_KERN_FINEST_PAGE_SIZE;

    if (!m_IsMapped)
    {
        KProcessPageTable& pageTable = m_pOwner->GetPageTable();
        Result result = pageTable.UnlockForTransferMemory(m_Addr, size, m_Memory);
        NN_KERN_ABORT_IF_FAILED(result);
    }

    m_Memory.Close();
    m_Memory.Finalize();

    KObjectAdaptor<KTransferMemory, KAutoObjectWithList>::Finalize();
}

Result KTransferMemory::Map(
    KProcessAddress     addr,
    size_t              size,
    nn::svc::MemoryPermission ownerProtection)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    if (m_OwnerProtection != ownerProtection)
    {
        return nn::svc::ResultInvalidState();
    }

    KScopedLightLock locker(&m_Lock);

    if (m_IsMapped)
    {
        return nn::svc::ResultInvalidState();
    }

    KMemoryState mapState = ((m_OwnerProtection == nn::svc::MemoryPermission_None)? KMemoryState_Transfered: KMemoryState_SharedTransfered);
    Result result = GetCurrentProcess().GetPageTable().MapPageGroup(addr, m_Memory, mapState, KMemoryPermission_UserReadWrite);
    if (result.IsSuccess())
    {
        m_IsMapped = true;
    }
    return result;
}

Result KTransferMemory::Unmap(KProcessAddress addr, size_t size)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    KScopedLightLock locker(&m_Lock);

    KMemoryState mapState = ((m_OwnerProtection == nn::svc::MemoryPermission_None)? KMemoryState_Transfered: KMemoryState_SharedTransfered);
    Result result = GetCurrentProcess().GetPageTable().UnmapPageGroup(addr, m_Memory, mapState);
    if (result.IsSuccess())
    {
        NN_KERN_ASSERT(m_IsMapped);
        m_IsMapped = false;
    }
    return result;
}

void KTransferMemory::PostFinalize(uintptr_t arg)
{
    KProcess* pOwner = reinterpret_cast<KProcess*>(arg);
    pOwner->ReleaseLimit(nn::svc::LimitableResource_TransferMemoryCountMax, 1);
    pOwner->Close();
}


}}

