﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_KThread.h"

/*! @file
    @brief      ThreadQueue クラスの定義です。

    KThread の扱いに特化したメソッドを持ちます。

    スレッドキューという文字通りのデータ構造と、
    同期オブジェクトのヘルパー関数という二つの面を持っています。

*/

namespace nn { namespace kern {

    class KScheduler;
    class KThreadQueue
    {
        friend class KScheduler;

    private:
        typedef KThread::QueueEntry Entry;

    private:
        Entry   m_Root;

    public:
        //! コンストラクタ
        KThreadQueue()
        {
            m_Root.pPrev = NULL;
            m_Root.pNext = NULL;
        }
        //! デストラクタ
        ~KThreadQueue()
        {
        }

        /*!
            @brief      指定のスレッドをスリープ状態にしつつキューにつみます

            この関数はスケジューラーを停止した状態で呼び出す必要があります。

            @param[in]  pThread     対象のスレッド

            @return     正しくスリープ状態にできたら true を返します。

        */
        bool SleepThread(KThread* p)
        {
            // スレッドを一旦スリープ状態にします。
            // この時点でスレッドは KScheduler から管轄されなくなります。
            //
            // CHECK: State が KThread::STATE_RUNNABLE の確認を推奨します。
            p->m_SleepingQueue = this;
            p->SetState(KThread::STATE_WAIT);

            // 自分のキューに格納します。
            Enqueue(p);

            // スレッドの停止が要求されていた場合はスリープ状態への遷移を取りやめます
            if( p->IsTerminateRequested() )
            {
                // スレッドを起こしなおします
                // STATE_WAIT -> STATE_RUNNABLEと切り替えることで
                // 対象スレッドは実行待機キューの末尾に移動します
                WakupThread(p);
                return false;
            }
            else
            {
                // 正しくスリープキューにつむことができました
                return true;
            }
        }

        /*!
            @brief      指定のスリープ状態であるスレッドを起こしなおし、KScheduler に管理を付け直します。

            この関数はスケジューラーを停止した状態で呼び出す必要があります。

            @param[in]  pThread     対象のスレッド

        */
        void WakupThread(KThread* p)
        {
            NN_KERN_EQUAL_ASSERT( p->GetState(), KThread::STATE_WAIT );

            // 取り除くスレッド "p" は
            //  STATE_WAIT 状態
            //  自分のキューに詰まれている
            // の二つを満たす必要があります。

            // 自分のキューから取り除きます
            Remove(p);

            // 実行可能状態にします
            // このタイミングで KScheduler に管理が引き継がれます。
            p->SetState(KThread::STATE_RUNNABLE);
            p->m_SleepingQueue = NULL;
        }

        /*!
            @brief      キューの先頭にあるスレッドを起こします。

            この関数はスケジューラーを停止した状態で呼び出す必要があります。

            @return     正しく起こせたらその KThread へのポインタを返します

        */
        KThread* WakupFrontThread()
        {
            // キューの先頭要素を取得します
            KThread* p = GetFront();
            if( p != NULL )
            {
                NN_KERN_EQUAL_ASSERT( p->GetState(), KThread::STATE_WAIT );

                // キューの先頭に詰まれた要素を取り除きます
                Dequeue();

                // 実行可能状態にします
                // このタイミングで KScheduler に管理が引き継がれます。
                p->SetState(KThread::STATE_RUNNABLE);
                p->m_SleepingQueue = NULL;
            }
            return p;
        }

        /*!
            @brief      キューの末尾にスレッドを追加します。

            @param[in]  pThread     対象のスレッド

        */

    private:
        void Enqueue(KThread* pAdd)
        {
            // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
            NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );

            // このオブジェクトはまだどのキューにも所属していないことを確認します
            NN_KERN_EQUAL_ASSERT( pAdd->m_QueueEntry.pPrev, NULL );
            NN_KERN_EQUAL_ASSERT( pAdd->m_QueueEntry.pNext, NULL );
            // キューに一つだけの場合は上のチェックをパスしてしまうので
            // 専用にチェックする
            NN_KERN_NOT_EQUAL_ASSERT( m_Root.pPrev, pAdd );
            NN_KERN_NOT_EQUAL_ASSERT( m_Root.pNext, pAdd );

            KThread* pLast = m_Root.pPrev;
            Entry& lastE = (pLast != NULL) ? pLast->m_QueueEntry: m_Root;

            // キューの最後尾に対象のスレッドを足します
            pAdd->m_QueueEntry.pPrev = pLast;
            pAdd->m_QueueEntry.pNext = NULL;

            // 今回追加したスレッドを末尾のオブジェクトとします。
            lastE .pNext    = pAdd;
            m_Root.pPrev    = pAdd;

            // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
            NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
            Entry* pe = &m_Root;
            // リンクを前方に辿ったら pAdd にたどり着くことを確認
            while( (pe->pNext != NULL) && (pe->pNext != pAdd) )
            {
                pe = &pe->pNext->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe->pNext, pAdd );
            // さらにリンクを最後まで辿ったら、最後のエントリにたどり着くことを確認
            while( (pe->pNext != NULL) )
            {
                pe = &pe->pNext->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pPrev->m_QueueEntry );

            // リンクを後方に辿ったら pAdd にたどり着くことを確認
            pe = &m_Root;
            while( (pe->pPrev != NULL) && (pe->pPrev != pAdd) )
            {
                pe = &pe->pPrev->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe->pPrev, pAdd );
            // さらにリンクを最後まで辿ったら、最初のエントリにたどり着くことを確認
            while( (pe->pPrev != NULL) )
            {
                pe = &pe->pPrev->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pNext->m_QueueEntry );
#endif
        }

        /*!
            @brief      キューの先頭からスレッドを一つ取り除きます。

        */
    public:
        void Dequeue()
        {
            // キューの先頭要素を取得します
            KThread* pFirst = GetFront();

            if( pFirst == NULL )
            {
                // キューに一つも詰まれていなかった
                return;
            }

            NN_KERN_EQUAL_ASSERT( pFirst->GetState(), KThread::STATE_WAIT );

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
            Entry* pe = &m_Root;
            // リンクを最後まで辿ったら、最後のエントリにたどり着くことを確認
            while( (pe->pNext != NULL) )
            {
                pe = &pe->pNext->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pPrev->m_QueueEntry );

            // リンクを最後まで辿ったら、最初のエントリにたどり着くことを確認
            pe = &m_Root;
            while( (pe->pPrev != NULL) )
            {
                pe = &pe->pPrev->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pNext->m_QueueEntry );
#endif

            // 先頭要素を付け替えます
            KThread* pSecond = pFirst->m_QueueEntry.pNext;
            Entry& secondE = (pSecond != NULL) ? pSecond->m_QueueEntry: m_Root;

            m_Root .pNext = pSecond;
            secondE.pPrev = NULL;
            pFirst->m_SleepingQueue = NULL;
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
            pFirst->m_QueueEntry.pPrev = NULL;
            pFirst->m_QueueEntry.pNext = NULL;
#endif
        }

        /*!
            @brief      キューから指定したスレッドを一つ取り除きます。

            @param[in]  pThread     対象のスレッド

        */
    private:
        void Remove(KThread* pRemove)
        {
            NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );
            // 安全のためスレッドキューに対象スレッドが入っているかどうかを確認します
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
            Entry* pe = &m_Root;
            // リンクを前方に辿ったら pRemove にたどり着くことを確認
            while( (pe->pNext != NULL) && (pe->pNext != pRemove) )
            {
                pe = &pe->pNext->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe->pNext, pRemove );
            // さらにリンクを最後まで辿ったら、最後のエントリにたどり着くことを確認
            while( (pe->pNext != NULL) )
            {
                pe = &pe->pNext->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pPrev->m_QueueEntry );

            // リンクを後方に辿ったら pRemove にたどり着くことを確認
            pe = &m_Root;
            while( (pe->pPrev != NULL) && (pe->pPrev != pRemove) )
            {
                pe = &pe->pPrev->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe->pPrev, pRemove );
            // さらにリンクを最後まで辿ったら、最初のエントリにたどり着くことを確認
            while( (pe->pPrev != NULL) )
            {
                pe = &pe->pPrev->m_QueueEntry;
            }
            NN_KERN_EQUAL_ASSERT( pe, &m_Root.pNext->m_QueueEntry );
#endif
            // pRemove対象のオブジェクト前後を繋ぎなおします
            KThread* pPrev = pRemove->m_QueueEntry.pPrev;
            KThread* pNext = pRemove->m_QueueEntry.pNext;
            Entry& prevE = (pPrev != NULL) ? pPrev->m_QueueEntry: m_Root;
            Entry& nextE = (pNext != NULL) ? pNext->m_QueueEntry: m_Root;

            prevE.pNext = pNext;
            nextE.pPrev = pPrev;
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
            pRemove->m_QueueEntry.pPrev = NULL;
            pRemove->m_QueueEntry.pNext = NULL;
#endif
            // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
            NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );
        }

        /*!
            @brief      キューに一つ以上スレッドが格納されていることを確認します。

            @return     キューにスレッドが存在する場合 true を返します

        */
    public:
        bool IsNotEmpty() const { return m_Root.pNext != NULL; }
        bool IsEmpty() const { return m_Root.pNext == NULL; }

        /*!
            @brief      キューに一つだけスレッドが格納されていることを確認します。

            @return     キューに一つだけスレッドが存在する場合 true を返します

        */
    private:
        bool HasOnlyOne() const { return IsNotEmpty() && m_Root.pNext == m_Root.pPrev; }

        /*!
            @brief      キューの先頭要素を取得します。

            @return     キューの先頭要素

        */
    public:
        KThread* GetFront() const { return m_Root.pNext; }

        /*!
            @brief      指定したキューの次の要素を取得します。

            @return     次の要素

        */
    public:
        KThread* GetNext(KThread* p) const { return p->m_QueueEntry.pNext; }
    };

}}

