﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_InterlockedSelect.h"
#include "kern_Result.h"
#include "kern_Assert.h"
#include "kern_New.h"
#include "kern_KTaggedAddress.h"
#include <cstring>


namespace nn { namespace kern {

class KSlabAllocatorImpl
{
public:
    struct SlabNode
    {
        SlabNode* next;
    };

public:
    KSlabAllocatorImpl() :
        m_pHead(nullptr),
        m_ObjSize(0)
    {
        NN_KERN_THIS_ASSERT();
    }

    KSlabAllocatorImpl& operator=(const KSlabAllocatorImpl&) = delete;
    KSlabAllocatorImpl(const KSlabAllocatorImpl&) = delete;

    void Initialize(size_t objSize)
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ABORT_UNLESS(m_pHead == nullptr);

        m_ObjSize = objSize;
    }

    size_t GetObjSize() const { return m_ObjSize; }
    void* AllocateFromSlab()
    {
        NN_KERN_THIS_ASSERT();

        AllocateFunc f;
        m_pHead.AtomicUpdateConditional(&f);
        return f.ret;
    }

    void FreeToSlab(void* pObj)
    {
        NN_KERN_THIS_ASSERT();

        FreeFunc f(reinterpret_cast<SlabNode*>(pObj));
        m_pHead.AtomicUpdateConditional(&f);
    }
    SlabNode* GetHead() const { return m_pHead; }

private:
    struct AllocateFunc
    {
        SlabNode* ret;
        bool operator()(SlabNode** head)
        {
            SlabNode* node = *head;
            ret = node;
            if (NN_LIKELY(node != nullptr))
            {
                *head = node->next;
                return true;
            }
            else
            {
                return false;
            }
        }
    };

    struct FreeFunc
    {
        SlabNode* m_p;
        explicit FreeFunc(SlabNode* p) : m_p(p) {}
        bool operator()(SlabNode** head)
        {
            m_p->next = *head;
            *head = m_p;
            return true;
        }
    };

private:
    InterlockedVariable<SlabNode*>      m_pHead;
    size_t                              m_ObjSize;
};


class KSlabAllocatorBase: protected KSlabAllocatorImpl
{
public:
    KSlabAllocatorBase() :
        m_pPeak(0),
        m_pStart(0),
        m_pEnd(0)
    {
        NN_KERN_THIS_ASSERT();
    }

    KSlabAllocatorBase& operator=(const KSlabAllocatorBase&) = delete;
    KSlabAllocatorBase(const KSlabAllocatorBase&) = delete;

    void Initialize(size_t objSize, void* pMem, size_t memSize)
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ABORT_UNLESS(pMem != nullptr);

        KSlabAllocatorImpl::Initialize(objSize);

        const size_t numObj = (memSize / objSize);
        m_pStart    = reinterpret_cast<uintptr_t>(pMem);
        m_pPeak     = m_pStart;
        m_pEnd      = m_pStart + numObj * GetObjSize();

        uint8_t* p = reinterpret_cast<uint8_t*>(m_pEnd);

        for (size_t i = 0; i < numObj; ++i)
        {
            p -= objSize;
            KSlabAllocatorImpl::FreeToSlab(p);
        }
    }

    void Finalize()
    {
        NN_KERN_THIS_ASSERT();
    }

    size_t GetSlabSize() const
    {
        return (m_pEnd - m_pStart) / GetObjSize();
    }

    size_t GetObjSize() const { return KSlabAllocatorImpl::GetObjSize(); }
    void* AllocateFromSlab()
    {
        NN_KERN_THIS_ASSERT();

        void *p = KSlabAllocatorImpl::AllocateFromSlab();

        NN_WARNING(p != nullptr, "AllocateFromSlab failed %p", m_pStart);
#ifdef NN_KERN_FOR_DEVELOPMENT
        if (NN_LIKELY(p != nullptr))
        {
            for (;;)
            {
                uintptr_t pPeak = m_pPeak;
                if (reinterpret_cast<uintptr_t>(p) + GetObjSize() <= pPeak)
                {
                    break;
                }
                if (__sync_bool_compare_and_swap(&m_pPeak, pPeak, reinterpret_cast<uintptr_t>(p) + GetObjSize()))
                {
                    break;
                }
            }
            std::memset(p, 0xbc, GetObjSize());
        }
#endif
        return p;
    }

    void FreeToSlab(void* pObj)
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ABORT_UNLESS(m_pStart <= reinterpret_cast<uintptr_t>(pObj) && reinterpret_cast<uintptr_t>(pObj) < m_pEnd);
#ifdef NN_KERN_FOR_DEVELOPMENT
        std::memset(pObj, 0xde, GetObjSize());
#endif
        KSlabAllocatorImpl::FreeToSlab(pObj);
    }

    size_t GetNumRemain() const
    {
        size_t remain = 0;

#ifdef NN_KERN_FOR_DEVELOPMENT
        // 注意：デバッグ用なので適当
        // ロックを行わずアドレス範囲だけでチェックしている
        for (;;)
        {
            SlabNode* p = GetHead();
            remain = 0;

            while (m_pStart <= reinterpret_cast<uintptr_t>(p) && reinterpret_cast<uintptr_t>(p) < m_pEnd)
            {
                remain++;
                p = p->next;
            }

            if (p == nullptr)
            {
                break;
            }
        }
#endif
        return remain;
    }

    size_t GetPeakNum() const
    {
        return GetIndex(reinterpret_cast<const void*>(m_pPeak));
    }

    size_t GetIndex(const void* pObj) const
    {
        return (reinterpret_cast<uintptr_t>(pObj) - m_pStart) / GetObjSize();
    }

    uintptr_t GetSlabAddr() const
    {
        return m_pStart;
    }

private:
    uintptr_t m_pPeak;
    uintptr_t m_pStart;
    uintptr_t m_pEnd;
};

template <typename T>
class KSlabAllocator : public KSlabAllocatorBase
{
public:
    void Initialize(void* pMem, size_t memSize)
    {
        KSlabAllocatorBase::Initialize(sizeof(T), pMem, memSize);
    }

    T* AllocateFromSlab()
    {
        T* pObj = reinterpret_cast<T*>(KSlabAllocatorBase::AllocateFromSlab());
        if (pObj != nullptr)
        {
            new (static_cast<void*>(pObj)) T();
        }
        return pObj;
    }

    void FreeToSlab(T* pObj)
    {
        KSlabAllocatorBase::FreeToSlab(pObj);
    }

    size_t GetIndex(const T* pObj) const
    {
        return KSlabAllocatorBase::GetIndex(pObj);
    }
};

}}

