﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_Assert.h"
#include "kern_KObjectAdaptor.h"
#include "kern_KWritableEvent.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"
#include "kern_KMemoryBlock.h"

namespace nn { namespace kern {


class KSessionRequest :
    public KUseSlabAllocator<KSessionRequest>,
    public KAutoObject,
    public nn::util::IntrusiveListBaseNode<KSessionRequest>
{
public:
    class SessionMapData
    {
    public:
        explicit SessionMapData(): m_pMapData(nullptr), m_SendNum(0), m_ReceiveNum(0), m_ExchangeNum(0) {}
        void Initialize() {}
        void Finalize();

        size_t GetSendNum()     const { return m_SendNum; }
        size_t GetReceiveNum()  const { return m_ReceiveNum; }
        size_t GetExchangeNum() const { return m_ExchangeNum; }
        Result PushSend(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state);
        Result PushReceive(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state);
        Result PushExchange(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state);
        KProcessAddress GetSendClientAddress(int index) const { return GetSendData(index).GetClientAddr(); }
        KProcessAddress GetReceiveClientAddress(int index) const { return GetReceiveData(index).GetClientAddr(); }
        KProcessAddress GetExchangeClientAddress(int index) const { return GetExchangeData(index).GetClientAddr(); }
        KProcessAddress GetSendServerAddress(int index) const { return GetSendData(index).GetServerAddr(); }
        KProcessAddress GetReceiveServerAddress(int index) const { return GetReceiveData(index).GetServerAddr(); }
        KProcessAddress GetExchangeServerAddress(int index) const { return GetExchangeData(index).GetServerAddr(); }
        size_t GetSendSize(int index) const { return GetSendData(index).GetSize(); }
        size_t GetReceiveSize(int index) const { return GetReceiveData(index).GetSize(); }
        size_t GetExchangeSize(int index) const { return GetExchangeData(index).GetSize(); }
        KMemoryState GetSendMemoryState(int index) const { return GetSendData(index).GetMemoryState(); }
        KMemoryState GetReceiveMemoryState(int index) const { return GetReceiveData(index).GetMemoryState(); }
        KMemoryState GetExchangeMemoryState(int index) const { return GetExchangeData(index).GetMemoryState(); }

    private:
        Result PushMapData(int x, KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state);

        class MapData
        {
        public:
            KProcessAddress GetClientAddr() const { return m_ClientAddr; }
            KProcessAddress GetServerAddr() const { return m_ServerAddr; }
            size_t GetSize()                const { return m_Size; }
            KMemoryState GetMemoryState()   const { return m_State; }

            void Set(KProcessAddress clientAddr, KProcessAddress serverAddr, size_t size, KMemoryState state)
            {
                m_ClientAddr = clientAddr;
                m_ServerAddr = serverAddr;
                m_Size = size;
                m_State = state;
            }
        private:
            KProcessAddress m_ClientAddr;
            KProcessAddress m_ServerAddr;
            size_t m_Size;
            KMemoryState m_State;
        };

        const MapData& GetSendData(int index) const
        {
            NN_KERN_ASSERT(index < m_SendNum);

            int x = index;
            if (x < NumMapDataStatic)
            {
                return m_MapDataStatic[x];
            }
            else
            {
                return m_pMapData[x - NumMapDataStatic];
            }
        }
        const MapData& GetReceiveData(int index) const
        {
            NN_KERN_ASSERT(index < m_ReceiveNum);

            int x = m_SendNum + index;
            if (x < NumMapDataStatic)
            {
                return m_MapDataStatic[x];
            }
            else
            {
                return m_pMapData[x - NumMapDataStatic];
            }
        }
        const MapData& GetExchangeData(int index) const
        {
            NN_KERN_ASSERT(index < m_ExchangeNum);

            int x = m_SendNum + m_ReceiveNum + index;
            if (x < NumMapDataStatic)
            {
                return m_MapDataStatic[x];
            }
            else
            {
                return m_pMapData[x - NumMapDataStatic];
            }
        }
        enum
        {
            NumMapDataStatic = 8
        };
    private:
        MapData m_MapDataStatic[NumMapDataStatic];
        MapData* m_pMapData;
        uint8_t m_SendNum;
        uint8_t m_ReceiveNum;
        uint8_t m_ExchangeNum;
    };


    explicit KSessionRequest() : m_pThread(nullptr), m_pServer(nullptr), m_pEvent(nullptr) {}
    virtual ~KSessionRequest(){}
    static KSessionRequest* Create()
    {
        KSessionRequest* pObj = KSessionRequest::Allocate();
        if (pObj != nullptr)
        {
            KAutoObject::Create(pObj);
        }
        return pObj;
    }

    virtual void Destroy()
    {
        Finalize();
        KSessionRequest::Free(this);
    }

    void Initialize(KWritableEvent* pEvent, uintptr_t addr, size_t size)
    {
        m_MapData.Initialize();

        m_pThread = &GetCurrentThread();
        m_pEvent = pEvent;
        m_Address = addr;
        m_Size = size;

        m_pThread->Open();
        if (m_pEvent)
        {
            m_pEvent->Open();
        }
    }

    virtual void Finalize()
    {
        m_MapData.Finalize();
        if (m_pThread)
        {
            m_pThread->Close();
        }
        if (m_pEvent)
        {
            m_pEvent->Close();
        }
        if (m_pServer)
        {
            m_pServer->Close();
        }
    }
    int32_t GetIndex() const { return KUseSlabAllocator<KSessionRequest>::GetIndex(this); }

    static void PostFinalize(uintptr_t arg) { NN_UNUSED(arg); }

    void ClearThread()
    {
        m_pThread = nullptr;
    }
    void ClearEvent()
    {
        m_pEvent = nullptr;
    }

    KThread* GetThread() const
    {
        return m_pThread;
    }
    KWritableEvent* GetEvent() const
    {
        return m_pEvent;
    }
    uintptr_t GetAddress() const
    {
        return m_Address;
    }
    size_t GetSize() const
    {
        return m_Size;
    }
    KProcess* GetServerProcess() const
    {
        return m_pServer;
    }

    size_t GetSendNum()     const { return m_MapData.GetSendNum(); }
    size_t GetReceiveNum()  const { return m_MapData.GetReceiveNum(); }
    size_t GetExchangeNum() const { return m_MapData.GetExchangeNum(); }
    Result PushSend(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state)
    {
        return m_MapData.PushSend(client, server, size, state);
    }
    Result PushReceive(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state)
    {
        return m_MapData.PushReceive(client, server, size, state);
    }
    Result PushExchange(KProcessAddress client, KProcessAddress server, size_t size, KMemoryState state)
    {
        return m_MapData.PushExchange(client, server, size, state);
    }
    void SetServerProcess(KProcess* pProcess)
    {
        m_pServer = pProcess;
        m_pServer->Open();
    }
    KProcessAddress GetSendClientAddress(int index) const { return m_MapData.GetSendClientAddress(index); }
    KProcessAddress GetReceiveClientAddress(int index) const { return m_MapData.GetReceiveClientAddress(index); }
    KProcessAddress GetExchangeClientAddress(int index) const { return m_MapData.GetExchangeClientAddress(index); }
    KProcessAddress GetSendServerAddress(int index) const { return m_MapData.GetSendServerAddress(index); }
    KProcessAddress GetReceiveServerAddress(int index) const { return m_MapData.GetReceiveServerAddress(index); }
    KProcessAddress GetExchangeServerAddress(int index) const { return m_MapData.GetExchangeServerAddress(index); }
    size_t GetSendSize(int index) const { return m_MapData.GetSendSize(index); }
    size_t GetReceiveSize(int index) const { return m_MapData.GetReceiveSize(index); }
    size_t GetExchangeSize(int index) const { return m_MapData.GetExchangeSize(index); }
    KMemoryState GetSendMemoryState(int index) const { return m_MapData.GetSendMemoryState(index); }
    KMemoryState GetReceiveMemoryState(int index) const { return m_MapData.GetReceiveMemoryState(index); }
    KMemoryState GetExchangeMemoryState(int index) const { return m_MapData.GetExchangeMemoryState(index); }

private:
    SessionMapData m_MapData;
    KThread* m_pThread;
    KProcess* m_pServer;
    KWritableEvent* m_pEvent;
    uintptr_t m_Address;
    size_t m_Size;

private:
    NN_AUTOOBJECT_DERIVED_FUNCSET(KSessionRequest, KAutoObject)
};

}}

