﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/util/util_BitPack.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KSynchronization.h"
#include "kern_KServerSession.h"
#include "kern_KSession.h"
#include "kern_KScheduler.h"
#include "kern_Panic.h"
#include "kern_Utility.h"
#include "kern_KTaggedAddress.h"
#include "kern_KMemoryManager.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KPort.h"
#include "kern_KTrace.h"
#include <cstddef>

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KServerSession);

#if NN_KERN_HAS_MMU
namespace
{
const size_t PointerTransferOneBufferAlign = 16;

// 割り上げ
template <typename T>
inline T DivUp(T x, size_t align)
{
    return static_cast<T>( (x + (align - 1)) / align );
}
// アドレスを含むページの先頭アドレスを求める
inline uintptr_t CalcPageBeginAddr(uintptr_t addr)
{
    return RoundDown(addr, NN_KERN_FINEST_PAGE_SIZE);
}

class ReceiveList
{
private:
    Bit32 m_Data[nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_BufferNumMax * nn::svc::ipc::MessageBuffer::ReceiveListEntry::GetSize() / sizeof(Bit32)];
    int m_ReceiveListNum;
    uintptr_t m_MessageBufferEnd;
    uintptr_t m_MessageBufferSpaceEnd;
public:
    bool IsIndex() const
    {
        return m_ReceiveListNum > nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_BufferNumOffset;
    }
    static int GetListEntryNum(const nn::svc::ipc::MessageBuffer::MessageHeader& header)
    {
        switch (header.GetReceiveListNum())
        {
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_None:
            return 0;
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToMessageBuffer:
            return 0;
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToOneBuffer:
            return 1;
        default:
            return (header.GetReceiveListNum() - nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_BufferNumOffset);
        }
    }

    void GetBuffer(uintptr_t* pBuffer, size_t size, int* pKey) const
    {
        switch (m_ReceiveListNum)
        {
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_None:
            {
                *pBuffer = 0;
            }
            break;
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToMessageBuffer:
            {
                uintptr_t bufferTop = RoundUp(m_MessageBufferEnd + *pKey, PointerTransferOneBufferAlign);

                if ((bufferTop < bufferTop + size) && (bufferTop + size <= m_MessageBufferSpaceEnd))
                {
                    *pBuffer = bufferTop;
                    *pKey = bufferTop + size - m_MessageBufferEnd;
                }
                else
                {
                    *pBuffer = 0;
                }
            }
            break;
        case nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToOneBuffer:
            {
                nn::svc::ipc::MessageBuffer::ReceiveListEntry entry(m_Data[0], m_Data[1]);
                uintptr_t bufferTop = RoundUp(entry.GetDataAddress() + *pKey, PointerTransferOneBufferAlign);

                if ((bufferTop < bufferTop + size) &&
                        (entry.GetDataAddress() < entry.GetDataAddress() + entry.GetDataSize()) &&
                        (bufferTop + size <= entry.GetDataAddress() + entry.GetDataSize()))
                {
                    *pBuffer = bufferTop;
                    *pKey = bufferTop + size - entry.GetDataAddress();
                }
                else
                {
                    *pBuffer = 0;
                }
            }
            break;
        default:
            {
                if (*pKey < m_ReceiveListNum - nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_BufferNumOffset)
                {
                    nn::svc::ipc::MessageBuffer::ReceiveListEntry entry(m_Data[*pKey * 2], m_Data[*pKey * 2 + 1]);
                    if ((entry.GetDataAddress() < entry.GetDataAddress() + entry.GetDataSize()) &&
                            entry.GetDataSize() >= size)
                    {
                        *pBuffer = entry.GetDataAddress();
                    }
                    else
                    {
                        *pBuffer = 0;
                    }
                }
                else
                {
                    *pBuffer = 0;
                }
            }
            break;
        }
    }
    ReceiveList(const Bit32* pToMsg, const nn::svc::ipc::MessageBuffer::MessageHeader& header, const nn::svc::ipc::MessageBuffer::SpecialHeader& specialHeader, size_t msgSize, size_t outOffset)
    {
        m_ReceiveListNum = header.GetReceiveListNum();
        m_MessageBufferEnd = reinterpret_cast<uintptr_t>(pToMsg) + sizeof(Bit32) * outOffset;
        m_MessageBufferSpaceEnd = reinterpret_cast<uintptr_t>(pToMsg) + msgSize;

        uintptr_t toReceiveOffset = nn::svc::ipc::MessageBuffer::GetReceiveListOffset(header, specialHeader);
        const Bit32* pRcvList = pToMsg + toReceiveOffset;
        int numReceiveListEntry = GetListEntryNum(header);
        __builtin_memcpy(m_Data, pRcvList, numReceiveListEntry * nn::svc::ipc::MessageBuffer::ReceiveListEntry::GetSize());
    }
    ReceiveList(const Bit32* pToMsg, uintptr_t processAddress, const KProcessPageTable& pageTable, const nn::svc::ipc::MessageBuffer::MessageHeader& header, const nn::svc::ipc::MessageBuffer::SpecialHeader& specialHeader, size_t msgSize, size_t outOffset)
    {
        m_ReceiveListNum = header.GetReceiveListNum();
        m_MessageBufferEnd = processAddress + sizeof(Bit32) * outOffset;
        m_MessageBufferSpaceEnd = processAddress + msgSize;

        uintptr_t toReceiveOffset = nn::svc::ipc::MessageBuffer::GetReceiveListOffset(header, specialHeader);
        const Bit32* pRcvList = pToMsg + toReceiveOffset;
        uintptr_t pageAddr = CalcPageBeginAddr(processAddress);
        uintptr_t addr = processAddress + toReceiveOffset * sizeof(Bit32);
        int numReceiveListEntry = GetListEntryNum(header);
        for (size_t i = 0; i < numReceiveListEntry * nn::svc::ipc::MessageBuffer::ReceiveListEntry::GetSize() / sizeof(Bit32); i++)
        {
            if (CalcPageBeginAddr(addr) != pageAddr)
            {
                KPhysicalAddress physAddress;
                pageTable.GetPhysicalAddress(&physAddress, KProcessAddress(addr));
                pRcvList = GetTypedPointer<Bit32>(KPageTable::GetHeapVirtualAddress(physAddress));
                pageAddr = CalcPageBeginAddr(addr);
            }
            m_Data[i] = *pRcvList++;
            addr += sizeof(Bit32);
        }
    }
};

NN_FORCEINLINE Result GetMapMemoryState(KMemoryState* pToState, nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute attribute)
{
    switch (attribute)
    {
    case nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc:
        {
            *pToState = KMemoryState_Ipc;
        }
        break;
    case nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_NonSecureIpc:
        {
            *pToState = KMemoryState_NonSecureIpc;
        }
        break;
    case nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_NonDeviceIpc:
        {
            *pToState = KMemoryState_NonDeviceIpc;
        }
        break;
    default:
        return nn::svc::ResultInvalidCombination();
    }
    return ResultSuccess();
}

NN_FORCEINLINE Result GetMapTransferTestMask(Bit32* pTestState, Bit32* pTestAttributeMask, KMemoryState state)
{
    switch (state)
    {
    case KMemoryState_Ipc:
        {
            *pTestState = KMemoryState_FlagsIpc;
            *pTestAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked | KMemoryAttribute_DeviceShared;
        }
        break;
    case KMemoryState_NonSecureIpc:
        {
            *pTestState = KMemoryState_FlagsNonSecureIpc;
            *pTestAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
        }
        break;
    case KMemoryState_NonDeviceIpc:
        {
            *pTestState = KMemoryState_FlagsNonDeviceIpc;
            *pTestAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
        }
        break;
    default:
        return nn::svc::ResultInvalidCombination();
    }
    return ResultSuccess();
}


NN_FORCEINLINE void CleanupSpecialData(
        KProcess* pToProcess,
        Bit32* pToMsg,
        size_t toMsgMaxLength
        )
{
    const nn::svc::ipc::MessageBuffer toMsg(pToMsg, toMsgMaxLength);
    const nn::svc::ipc::MessageBuffer::MessageHeader toHeader(toMsg);
    const nn::svc::ipc::MessageBuffer::SpecialHeader toSpecialHeader(toMsg, toHeader);

    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(toHeader, toSpecialHeader) > toMsgMaxLength)
    {
        return;
    }

    int offset = toMsg.Set(toSpecialHeader);

    if (toSpecialHeader.GetProcessIdFlag())
    {
        offset = toMsg.SetProcessId(offset, 0);
    }

    for (int i = 0; i < (toSpecialHeader.GetCopyHandleNum() + toSpecialHeader.GetMoveHandleNum()); i++)
    {
        nn::svc::Handle toHandle = toMsg.GetHandle(offset);
        KHandleTable& toHandleTable   = pToProcess->GetHandleTable();
        if (toHandle != nn::svc::INVALID_HANDLE_VALUE)
        {
            toHandleTable.Remove(toHandle);
        }

        offset = toMsg.SetHandle(offset, nn::svc::INVALID_HANDLE_VALUE);
    }
}

// To: Client
// From: Server
NN_FORCEINLINE Result CleanupServerMap(KProcessPageTable* pFromPageTable, KSessionRequest* pRequest)
{
    Result result;

    if (!pFromPageTable)
    {
        return ResultSuccess();
    }

    for (size_t i = 0; i < pRequest->GetSendNum(); i++)
    {
        result = pFromPageTable->CleanupForIpcServer(pRequest->GetSendServerAddress(i), pRequest->GetSendSize(i), pRequest->GetSendMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    for (size_t i = 0; i < pRequest->GetReceiveNum(); i++)
    {
        result = pFromPageTable->CleanupForIpcServer(pRequest->GetReceiveServerAddress(i), pRequest->GetReceiveSize(i), pRequest->GetReceiveMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    for (size_t i = 0; i < pRequest->GetExchangeNum(); i++)
    {
        result = pFromPageTable->CleanupForIpcServer(pRequest->GetExchangeServerAddress(i), pRequest->GetExchangeSize(i), pRequest->GetExchangeMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

NN_FORCEINLINE Result CleanupClientMap(KProcessPageTable* pToPageTable, KSessionRequest* pRequest)
{
    Result result;

    if (!pToPageTable)
    {
        return ResultSuccess();
    }
    for (size_t i = 0; i < pRequest->GetSendNum(); i++)
    {
        result = pToPageTable->CleanupForIpcClient(pRequest->GetSendClientAddress(i), pRequest->GetSendSize(i), pRequest->GetSendMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    for (size_t i = 0; i < pRequest->GetReceiveNum(); i++)
    {
        result = pToPageTable->CleanupForIpcClient(pRequest->GetReceiveClientAddress(i), pRequest->GetReceiveSize(i), pRequest->GetReceiveMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    for (size_t i = 0; i < pRequest->GetExchangeNum(); i++)
    {
        result = pToPageTable->CleanupForIpcClient(pRequest->GetExchangeClientAddress(i), pRequest->GetExchangeSize(i), pRequest->GetExchangeMemoryState(i));
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}


NN_FORCEINLINE Result CleanupMap(KProcessPageTable* pClientPageTable, KProcessPageTable* pServerPageTable, KSessionRequest* pRequest)
{
    Result result;

    result = CleanupServerMap(pServerPageTable, pRequest);
    if (result.IsSuccess())
    {
        result = CleanupClientMap(pClientPageTable, pRequest);
    }
    return result;
}

NN_FORCEINLINE Result CleanupServerHandle(uintptr_t pFromMessage, size_t fromMsgMaxLength)
{
    const KThread& from = GetCurrentThread();
    Bit32* pFrom;

    if (pFromMessage)
    {
        pFrom = reinterpret_cast<Bit32*>(pFromMessage);
    }
    else
    {
        pFrom = GetTypedPointer<nn::svc::ThreadLocalRegion>(from.GetThreadLocalRegionAddr())->messageBuffer;
        fromMsgMaxLength = sizeof(nn::svc::ThreadLocalRegion().messageBuffer);
        pFromMessage = reinterpret_cast<uintptr_t>(pFrom);
    }

    const nn::svc::ipc::MessageBuffer fromMsg(pFrom, fromMsgMaxLength);
    const nn::svc::ipc::MessageBuffer::MessageHeader fromHeader(fromMsg);
    const nn::svc::ipc::MessageBuffer::SpecialHeader fromSpecialHeader(fromMsg, fromHeader);

    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(fromHeader, fromSpecialHeader) > fromMsgMaxLength)
    {
        return nn::svc::ResultInvalidCombination();
    }

    // Move Hanle を Closeする
    if (fromHeader.GetSpecialNum())
    {
        size_t offset = fromMsg.GetSpecialDataOffset(fromHeader, fromSpecialHeader);
        if (fromSpecialHeader.GetProcessIdFlag())
        {
            offset += sizeof(Bit64) / sizeof(Bit32);
        }
        if (fromSpecialHeader.GetCopyHandleNum())
        {
            offset += sizeof(nn::svc::Handle) * fromSpecialHeader.GetCopyHandleNum() / sizeof(Bit32);
        }
        for (int i = 0; i < fromSpecialHeader.GetMoveHandleNum(); i++)
        {
            nn::svc::Handle fromHandle = fromMsg.GetHandle(offset);
            KHandleTable& fromHandleTable = from.GetParent().GetHandleTable();
            fromHandleTable.Remove(fromHandle);
            offset += sizeof(nn::svc::Handle) / sizeof(Bit32);
        }
    }

    return ResultSuccess();
}


NN_FORCEINLINE Result ProcessMessageSpecialData(
        int *pOffset,
        KProcess* pToProcess,
        KProcess* pFromProcess,
        KThread* pFromThread,
        const nn::svc::ipc::MessageBuffer& toMsg,
        const nn::svc::ipc::MessageBuffer& fromMsg,
        const nn::svc::ipc::MessageBuffer::SpecialHeader& fromSpecial
        )
{
    Result result = ResultSuccess();
    *pOffset = toMsg.Set(fromSpecial);

    if (fromSpecial.GetProcessIdFlag())
    {
        *pOffset = toMsg.SetProcessId(*pOffset, pFromProcess->GetId());
    }

    KHandleTable& fromHandleTable = pFromProcess->GetHandleTable();
    KHandleTable& toHandleTable   = pToProcess->GetHandleTable();
    for (int i = 0; i < fromSpecial.GetCopyHandleNum(); i++)
    {
        nn::svc::Handle fromHandle = fromMsg.GetHandle(*pOffset);
        nn::svc::Handle toHandle = nn::svc::INVALID_HANDLE_VALUE;

        if (result.IsSuccess() && fromHandle != nn::svc::INVALID_HANDLE_VALUE)
        {
            KAutoObject* pObj = GetObjectFromRealOrPseudoHandleForIpc(fromHandleTable, fromHandle, pFromThread);

            if (pObj)
            {
                Result resultCheck = toHandleTable.Add(&toHandle, pObj);
                pObj->Close();

                if (resultCheck.IsFailure())
                {
                    result = resultCheck;
                    toHandle = nn::svc::INVALID_HANDLE_VALUE;
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }

        *pOffset = toMsg.SetHandle(*pOffset, toHandle);
    }

    for (int i = 0; i < fromSpecial.GetMoveHandleNum(); i++)
    {
        nn::svc::Handle fromHandle = fromMsg.GetHandle(*pOffset);
        nn::svc::Handle toHandle = nn::svc::INVALID_HANDLE_VALUE;

        if (fromHandle != nn::svc::INVALID_HANDLE_VALUE)
        {
            if (result.IsSuccess())
            {
                KAutoObject* pObj = fromHandleTable.GetObjectForIpc(fromHandle);

                if (pObj)
                {
                    Result resultCheck = toHandleTable.Add(&toHandle, pObj);
                    pObj->Close();
                    fromHandleTable.Remove(fromHandle);

                    if (resultCheck.IsFailure())
                    {
                        result = resultCheck;
                        toHandle = nn::svc::INVALID_HANDLE_VALUE;
                    }
                }
                else
                {
                    result = nn::svc::ResultInvalidHandle();
                }
            }
            else
            {
                fromHandleTable.Remove(fromHandle);
            }
        }

        *pOffset = toMsg.SetHandle(*pOffset, toHandle);
    }

    return result;
}

NN_FORCEINLINE Result ProcessSendMessagePointerData(
        int* pOffset,
        int* pReceiveListKey,
        KProcessPageTable* pToPageTable,
        const nn::svc::ipc::MessageBuffer& toMsg,
        const nn::svc::ipc::MessageBuffer& fromMsg,
        const ReceiveList& toReceiveList,
        bool toUserBuffer
        )
{
    int offset = *pOffset;
    nn::svc::ipc::MessageBuffer::PointerData fromData(offset, fromMsg);

    // pOffsetは失敗時でも進めるため先に更新する
    *pOffset = offset + fromData.GetSize() / sizeof(Bit32);

    uintptr_t pointer = fromData.GetPointerAddress();
    uintptr_t rcvPointer = 0;
    size_t rcvSize = fromData.GetPointerSize();

    if (rcvSize)
    {
        if (toReceiveList.IsIndex())
        {
            *pReceiveListKey = fromData.GetPointerIndex();
        }
        toReceiveList.GetBuffer(&rcvPointer, rcvSize, pReceiveListKey);

        if (rcvPointer == 0)
        {
            return nn::svc::ResultOutOfResource();
        }

        Result result;
        result = pToPageTable->CopyMemoryFromUser(
                rcvPointer, rcvSize,
                KMemoryState_FlagsMemory, KMemoryState_FlagsMemory, toUserBuffer? KMemoryPermission_KernelReadWrite: KMemoryPermission_UserReadWrite,
                KMemoryAttribute_Uncached, 0,
                pointer);
        if (result.IsFailure())
        {
            return result;
        }
    }

    toMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(
                reinterpret_cast<void*>(rcvPointer), rcvSize, fromData.GetPointerIndex()));
    return ResultSuccess();
}

NN_FORCEINLINE Result ProcessReceiveMessagePointerData(
        int* pOffset,
        int* pReceiveListKey,
        KProcessPageTable* pFromPageTable,
        const nn::svc::ipc::MessageBuffer& toMsg,
        const nn::svc::ipc::MessageBuffer& fromMsg,
        const ReceiveList& toReceiveList,
        bool toUserBuffer
        )
{
    int offset = *pOffset;
    nn::svc::ipc::MessageBuffer::PointerData fromData(offset, fromMsg);

    // pOffsetは失敗時でも進めるため先に更新する
    *pOffset = offset + fromData.GetSize() / sizeof(Bit32);

    uintptr_t pointer = fromData.GetPointerAddress();
    uintptr_t rcvPointer = 0;
    size_t rcvSize = fromData.GetPointerSize();

    if (rcvSize)
    {
        if (toReceiveList.IsIndex())
        {
            *pReceiveListKey = fromData.GetPointerIndex();
        }
        toReceiveList.GetBuffer(&rcvPointer, rcvSize, pReceiveListKey);

        if (rcvPointer == 0)
        {
            return nn::svc::ResultOutOfResource();
        }

        Result result;
        if (toUserBuffer)
        {
            result = pFromPageTable->CopyMemoryToKernel(
                    rcvPointer, rcvSize,
                    pointer,
                    KMemoryState_FlagsMemory, KMemoryState_FlagsMemory, KMemoryPermission_UserRead,
                    KMemoryAttribute_Uncached, 0);
        }
        else
        {
            result = pFromPageTable->CopyMemoryToUser(
                    rcvPointer, rcvSize,
                    pointer,
                    KMemoryState_FlagsMemory, KMemoryState_FlagsMemory, KMemoryPermission_UserRead,
                    KMemoryAttribute_Uncached, 0);
        }
        if (result.IsFailure())
        {
            return result;
        }
    }

    toMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(
                reinterpret_cast<void*>(rcvPointer), rcvSize, fromData.GetPointerIndex()));
    return ResultSuccess();
}


NN_FORCEINLINE Result ProcessSendMessageDataReceive(
        KProcessPageTable* pToPageTable,
        KProcessAddress clientAddr,
        KProcessAddress serverAddr,
        size_t size,
        KMemoryState fromState)
{
    // 書き戻し
    if (size > 0)
    {
        Result result;
#ifdef NN_KERN_CHECK_IPC_BOUNDARY
        if (size < NN_KERN_FINEST_PAGE_SIZE)
        {
            Bit32 testState = 0;
            Bit32 testAttributeMask = 0;
            result = GetMapTransferTestMask(&testState, &testAttributeMask, fromState);
            if (result.IsFailure())
            {
                return result;
            }

            result = pToPageTable->CopyMemoryFromUser(
                    clientAddr, size,
                    testState, testState, KMemoryPermission_UserReadWrite,
                    testAttributeMask, 0,
                    serverAddr);
            if (result.IsFailure())
            {
                return result;
            }
        }
        else
#endif
        {
            KProcessAddress fromAddrMappingEnd = RoundDown(serverAddr + size, NN_KERN_FINEST_PAGE_SIZE);
            KProcessAddress toAddrBeginAligned = RoundDown(clientAddr, NN_KERN_FINEST_PAGE_SIZE);
            KProcessAddress toAddrEndAligned   = RoundUp(clientAddr + size, NN_KERN_FINEST_PAGE_SIZE);
            KProcessAddress toAddrMappingBegin = RoundUp(clientAddr, NN_KERN_FINEST_PAGE_SIZE);
            KProcessAddress toAddrMappingEnd   = RoundDown(clientAddr + size, NN_KERN_FINEST_PAGE_SIZE);

            Bit32 testState = 0;
            Bit32 testAttributeMask = 0;
            result = GetMapTransferTestMask(&testState, &testAttributeMask, fromState);
            if (result.IsFailure())
            {
                return result;
            }

            if (toAddrBeginAligned != toAddrMappingBegin)
            {
                NN_KERN_ASSERT(clientAddr < toAddrMappingBegin);
                size_t copySize = ((size < static_cast<size_t>(toAddrMappingBegin - clientAddr)) ? size: static_cast<size_t>(toAddrMappingBegin - clientAddr));
                result = pToPageTable->CopyMemoryFromUser(
                        clientAddr, copySize,
                        testState, testState, KMemoryPermission_UserReadWrite,
                        testAttributeMask, 0,
                        serverAddr);
                if (result.IsFailure())
                {
                    return result;
                }
            }
            if (toAddrMappingEnd < toAddrEndAligned && (toAddrBeginAligned < toAddrMappingEnd || toAddrBeginAligned == toAddrMappingBegin))
            {
                size_t copySize = clientAddr + size - toAddrMappingEnd;
                result = pToPageTable->CopyMemoryFromUser(
                        toAddrMappingEnd, copySize,
                        testState, testState, KMemoryPermission_UserReadWrite,
                        testAttributeMask, 0,
                        fromAddrMappingEnd);
                if (result.IsFailure())
                {
                    return result;
                }
            }
        }
    }
    return ResultSuccess();
}

NN_FORCEINLINE void ReplyAsyncError(KProcess* pToProcess, uintptr_t pToMessage, size_t toMsgMaxLength, Result result)
{
    KProcessPageTable& toPageTable = pToProcess->GetPageTable();

    KPhysicalAddress physAddress;
    toPageTable.GetPhysicalAddress(&physAddress, KProcessAddress(pToMessage));

    Bit32* pToMsg = GetTypedPointer<Bit32>(KPageTable::GetHeapVirtualAddress(physAddress));

    nn::svc::ipc::MessageBuffer toMsg(pToMsg, toMsgMaxLength);

    toMsg.SetAsyncResult(result);
}

//! server -> client 方向のメッセージを処理します
NN_FORCEINLINE Result SendMessage(const KThread& to, uintptr_t pFromMessage, size_t fromMsgMaxLength, uintptr_t pToMessage, size_t toMsgMaxLength, KSessionRequest* pClient, KServerSession* pSession)
{
    NN_UNUSED(pSession);
    Result result = ResultSuccess();

    // from(サーバー側)が現在のスレッドであり、アドレス空間です。
    // to側(クライアント側)の仮想アドレス空間とはことなります。
    // アドレス変換しつつアクセスします。
    const KThread&     from          = GetCurrentThread();
    KProcess&          toProcess     = to.GetParent();
    KProcess&          fromProcess   = from.GetParent();
    KProcessPageTable& toPageTable   = toProcess  .GetPageTable();
    KProcessPageTable& fromPageTable = fromProcess.GetPageTable();

    Bit32* pToMsg;
    Bit32* pFrom;
    bool toUserBuffer;
    bool fromUserBuffer;
    bool processSpecialData = false;
    int offset = 0;
    int pointerDataKey = 0;

    if (pToMessage)
    {
        KPhysicalAddress physAddress;
        toPageTable.GetPhysicalAddress(&physAddress, KProcessAddress(pToMessage));
        pToMsg = GetTypedPointer<Bit32>(KPageTable::GetHeapVirtualAddress(physAddress));
        toUserBuffer = true;
    }
    else
    {
        pToMsg = reinterpret_cast<nn::svc::ThreadLocalRegion*>(to.GetThreadLocalRegionHeapAddr())->messageBuffer;
        toMsgMaxLength = sizeof(nn::svc::ThreadLocalRegion().messageBuffer);
        pToMessage = reinterpret_cast<uintptr_t>(GetTypedPointer<nn::svc::ThreadLocalRegion>(to.GetThreadLocalRegionAddr())->messageBuffer);
        toUserBuffer = false;
    }

    if (pFromMessage)
    {
        pFrom = reinterpret_cast<Bit32*>(pFromMessage);
        fromUserBuffer = true;
    }
    else
    {
        pFrom = GetTypedPointer<nn::svc::ThreadLocalRegion>(from.GetThreadLocalRegionAddr())->messageBuffer;
        fromMsgMaxLength = sizeof(nn::svc::ThreadLocalRegion().messageBuffer);
        pFromMessage = reinterpret_cast<uintptr_t>(pFrom);
        fromUserBuffer = false;
    }

    const nn::svc::ipc::MessageBuffer toMsg(pToMsg, toMsgMaxLength);
    const nn::svc::ipc::MessageBuffer fromMsg(pFrom, fromMsgMaxLength);
    const nn::svc::ipc::MessageBuffer::MessageHeader toHeader(toMsg);
    const nn::svc::ipc::MessageBuffer::MessageHeader fromHeader(fromMsg);
    const nn::svc::ipc::MessageBuffer::SpecialHeader toSpecialHeader(toMsg, toHeader);
    const nn::svc::ipc::MessageBuffer::SpecialHeader fromSpecialHeader(fromMsg, fromHeader);
    size_t fromMsgDataEndOffset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(fromHeader, fromSpecialHeader) + fromHeader.GetRawNum();

    // check
    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(fromHeader, fromSpecialHeader) > fromMsgMaxLength)
    {
        result = nn::svc::ResultInvalidCombination();
        goto error_exit;
    }

    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(toHeader, toSpecialHeader) > toMsgMaxLength)
    {
        result = nn::svc::ResultInvalidCombination();
        goto error_exit;
    }

    if (toHeader.GetReceiveListOffset() && toHeader.GetReceiveListOffset() < nn::svc::ipc::MessageBuffer::GetRawDataOffset(toHeader, toSpecialHeader) + toHeader.GetRawNum())
    {
        result = nn::svc::ResultInvalidCombination();
        goto error_exit;
    }

    if (fromMsgDataEndOffset * sizeof(Bit32) > toMsgMaxLength)
    {
        result = nn::svc::ResultTooLargeMessage();
        goto error_exit;
    }

    // replyにマップ転送があればエラー
    if (fromHeader.GetSendNum() != 0 || fromHeader.GetReceiveNum() != 0 || fromHeader.GetExchangeNum() != 0)
    {
        result = nn::svc::ResultInvalidCombination();
        goto error_exit;
    }

    {
        ReceiveList toReceiveList(pToMsg, pToMessage, toPageTable, toHeader, toSpecialHeader, toMsgMaxLength, fromMsgDataEndOffset);

        for (size_t i = 0; i < pClient->GetReceiveNum(); i++)
        {
            result = ProcessSendMessageDataReceive(
                    &toPageTable,
                    pClient->GetReceiveClientAddress(i),
                    pClient->GetReceiveServerAddress(i),
                    pClient->GetReceiveSize(i),
                    pClient->GetReceiveMemoryState(i));
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }
        for (size_t i = 0; i < pClient->GetExchangeNum(); i++)
        {
            result = ProcessSendMessageDataReceive(
                    &toPageTable,
                    pClient->GetExchangeClientAddress(i),
                    pClient->GetExchangeServerAddress(i),
                    pClient->GetExchangeSize(i),
                    pClient->GetExchangeMemoryState(i));
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }

        offset = toMsg.Set(fromHeader);

        NN_KERN_ASSERT(&from == &GetCurrentThread());
        processSpecialData = true;
        if (fromHeader.GetSpecialNum())
        {
            result = ProcessMessageSpecialData(&offset, &toProcess, &fromProcess, &GetCurrentThread(), toMsg, fromMsg, fromSpecialHeader);
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }

        for (int i = 0; i < fromHeader.GetPointerNum(); i++)
        {
            result = ProcessSendMessagePointerData(&offset, &pointerDataKey, &toPageTable, toMsg, fromMsg, toReceiveList,
                    (toUserBuffer && toHeader.GetReceiveListNum() == nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToMessageBuffer));
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }

        for (int i = 0; i < fromHeader.GetSendNum() + fromHeader.GetReceiveNum() + fromHeader.GetExchangeNum(); i++)
        {
            offset = toMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData());
        }

        if (fromHeader.GetRawNum())
        {
            if (!toUserBuffer && !fromUserBuffer)
            {
                std::memcpy(pToMsg + offset, pFrom + offset, fromHeader.GetRawNum() * sizeof(Bit32));
            }
            else
            {
                result = toPageTable.CopyMemoryFromKernel(
                        pToMessage + offset * sizeof(Bit32), fromHeader.GetRawNum() * sizeof(Bit32),
                        KMemoryState_FlagsMemory, KMemoryState_FlagsMemory, toUserBuffer? KMemoryPermission_KernelReadWrite: KMemoryPermission_UserReadWrite,
                        KMemoryAttribute_Uncached, 0,
                        pFromMessage + offset * sizeof(Bit32));
                if (result.IsFailure())
                {
                    goto error_exit;
                }
            }
        }

        NN_KERN_KTRACE_IPC_REPLY(pSession->GetParent()->GetIndex(), pClient->GetIndex(), (fromHeader.GetData())->storage, (fromHeader.GetData() + 1)->storage);

        return CleanupMap(&toPageTable, &fromPageTable, pClient);
    }

error_exit:
    if (processSpecialData)
    {
        if (fromHeader.GetSpecialNum())
        {
            CleanupSpecialData(&toProcess, pToMsg, toMsgMaxLength);
        }
    }
    else
    {
        CleanupServerHandle(pFromMessage, fromMsgMaxLength);
    }
    CleanupMap(&toPageTable, &fromPageTable, pClient);
    return result;
}

// client(from) -> server(to)
NN_FORCEINLINE Result ProcessReceiveMessageData(
        int* pOffset,
        KProcessPageTable* pToPageTable,
        KProcessPageTable* pFromPageTable,
        const nn::svc::ipc::MessageBuffer& toMsg,
        const nn::svc::ipc::MessageBuffer& fromMsg,
        KSessionRequest *pRequest,
        KMemoryPermission permission,
        bool isSend
        )
{
    Result result = ResultSuccess();
    int offset = *pOffset;

    nn::svc::ipc::MessageBuffer::MapData fromData(offset, fromMsg);

    // pOffsetは失敗時でも進めるため先に更新する
    *pOffset = offset + fromData.GetSize() / sizeof(Bit32);

    KProcessAddress pointer = fromData.GetDataAddress();
    size_t length = fromData.GetDataSize();
    KProcessAddress toBegin = 0;
    KMemoryState toState;
    result = GetMapMemoryState(&toState, fromData.GetAttribute());
    if (result.IsFailure())
    {
        return result;
    }

    if (length > 0)
    {
        result = pToPageTable->SetupForIpc(&toBegin, length,
                pointer, pFromPageTable,
                permission,
                toState,
                isSend);
        if (result.IsFailure())
        {
            return result;
        }

        if (permission == KMemoryPermission_UserRead)
        {
            result = pRequest->PushSend(pointer, toBegin, length, toState);
        }
        else if (isSend)
        {
            result = pRequest->PushExchange(pointer, toBegin, length, toState);
        }
        else
        {
            result = pRequest->PushReceive(pointer, toBegin, length, toState);
        }
        if (result.IsFailure())
        {
            pToPageTable->CleanupForIpcServer(toBegin, length, toState);
            return result;
        }
    }

    toMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(GetUntypedPointer(toBegin), length, fromData.GetAttribute()));
    return result;
}

//! client -> server 方向のメッセージを処理します
NN_FORCEINLINE Result ReceiveMessage(bool* pReceiveListBroken, uintptr_t pToMessage, size_t toMsgMaxLength, KThread* pFromThread, uintptr_t pFromMessage, size_t fromMsgMaxLength, KSessionRequest* pRequest, KServerSession* pSession)
{
    Result result = ResultSuccess();
    NN_UNUSED(pSession);

    // to側(サーバー側)が現在のスレッドであり、アドレス空間です。
    // from側(クライアント側)の仮想アドレス空間とはことなります。
    // アドレス変換しつつアクセスします。
   const KThread&      to            = GetCurrentThread();
    KProcess&          toProcess     = to.GetParent();
    KProcess&          fromProcess   = pFromThread->GetParent();
    KProcessPageTable& toPageTable   = toProcess  .GetPageTable();
    KProcessPageTable& fromPageTable = fromProcess.GetPageTable();
    *pReceiveListBroken = false;

    pRequest->SetServerProcess(&toProcess);

    Bit32* pToMsg;
    Bit32* pFrom;
    bool toUserBuffer;
    bool fromUserBuffer;

    if (pToMessage)
    {
        pToMsg = reinterpret_cast<Bit32*>(pToMessage);
        toUserBuffer = true;
    }
    else
    {
        pToMsg = GetTypedPointer<nn::svc::ThreadLocalRegion>(to.GetThreadLocalRegionAddr())->messageBuffer;
        toMsgMaxLength = sizeof(nn::svc::ThreadLocalRegion().messageBuffer);
        pToMessage = reinterpret_cast<uintptr_t>(pToMsg);
        toUserBuffer = false;
    }

    if (pFromMessage)
    {
        KPhysicalAddress physAddress;
        fromPageTable.GetPhysicalAddress(&physAddress, KProcessAddress(pFromMessage));
        pFrom = GetTypedPointer<Bit32>(KPageTable::GetHeapVirtualAddress(physAddress));
        fromUserBuffer = true;
    }
    else
    {
        pFrom = reinterpret_cast<nn::svc::ThreadLocalRegion*>(pFromThread->GetThreadLocalRegionHeapAddr())->messageBuffer;
        fromMsgMaxLength = sizeof(nn::svc::ThreadLocalRegion().messageBuffer);
        pFromMessage = reinterpret_cast<uintptr_t>(GetTypedPointer<nn::svc::ThreadLocalRegion>(pFromThread->GetThreadLocalRegionAddr())->messageBuffer);
        fromUserBuffer = false;
    }

    const nn::svc::ipc::MessageBuffer toMsg(pToMsg, toMsgMaxLength);
    const nn::svc::ipc::MessageBuffer fromMsg(pFrom, fromMsgMaxLength);
    const nn::svc::ipc::MessageBuffer::MessageHeader toHeader(toMsg);
    const nn::svc::ipc::MessageBuffer::MessageHeader fromHeader(fromMsg);
    const nn::svc::ipc::MessageBuffer::SpecialHeader toSpecialHeader(toMsg, toHeader);
    const nn::svc::ipc::MessageBuffer::SpecialHeader fromSpecialHeader(fromMsg, fromHeader);
    size_t fromMsgDataEndOffset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(fromHeader, fromSpecialHeader) + fromHeader.GetRawNum();

    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(fromHeader, fromSpecialHeader) > fromMsgMaxLength)
    {
        return nn::svc::ResultInvalidCombination();
    }

    if (nn::svc::ipc::MessageBuffer::GetMessageBufferSize(toHeader, toSpecialHeader) > toMsgMaxLength)
    {
        return nn::svc::ResultInvalidCombination();
    }

    if (toHeader.GetReceiveListOffset() && toHeader.GetReceiveListOffset() < nn::svc::ipc::MessageBuffer::GetRawDataOffset(toHeader, toSpecialHeader) + toHeader.GetRawNum())
    {
        return nn::svc::ResultInvalidCombination();
    }

    if (fromMsgDataEndOffset * sizeof(Bit32) > toMsgMaxLength)
    {
        return nn::svc::ResultTooLargeMessage();
    }

    int receiveListOffset = nn::svc::ipc::MessageBuffer::GetReceiveListOffset(toHeader, toSpecialHeader);

    ReceiveList toReceiveList(pToMsg, toHeader, toSpecialHeader, toMsgMaxLength, fromMsgDataEndOffset);

    int offset = 0;
    int pointerDataKey = 0;

    nn::svc::ipc::MessageBuffer::MessageHeader newHeader = fromHeader;
    offset = toMsg.Set(newHeader);

    if (fromHeader.GetSpecialNum())
    {
        if (fromSpecialHeader.GetMoveHandleNum())
        {
            return nn::svc::ResultInvalidCombination();
        }
        result = ProcessMessageSpecialData(&offset, &toProcess, &fromProcess, pFromThread, toMsg, fromMsg, fromSpecialHeader);
        if (offset > receiveListOffset)
        {
            *pReceiveListBroken = true;
        }
        if (result.IsFailure())
        {
            goto error_exit;
        }
    }

    for (int i = 0; i < fromHeader.GetPointerNum(); i++)
    {
        result = ProcessReceiveMessagePointerData(&offset, &pointerDataKey, &fromPageTable, toMsg, fromMsg, toReceiveList,
                (toUserBuffer && toHeader.GetReceiveListNum() == nn::svc::ipc::MessageBuffer::MessageHeader::ReceiveList_ToMessageBuffer));
        if (offset > receiveListOffset)
        {
            *pReceiveListBroken = true;
        }
        if (result.IsFailure())
        {
            goto error_exit;
        }
    }

    for (int i = 0; i < fromHeader.GetSendNum() + fromHeader.GetReceiveNum() + fromHeader.GetExchangeNum(); i++)
    {
        KMemoryPermission permission = (i >= fromHeader.GetSendNum())? KMemoryPermission_UserReadWrite: KMemoryPermission_UserRead;
        bool isSend = ((i < fromHeader.GetSendNum()) || (fromHeader.GetSendNum() + fromHeader.GetReceiveNum() <= i));
        result = ProcessReceiveMessageData(&offset, &toPageTable, &fromPageTable, toMsg, fromMsg, pRequest, permission, isSend);
        if (offset > receiveListOffset)
        {
            *pReceiveListBroken = true;
        }
        if (result.IsFailure())
        {
            goto error_exit;
        }
    }

    if (fromHeader.GetRawNum())
    {
        if (!toUserBuffer && !fromUserBuffer)
        {
            std::memcpy(pToMsg + offset, pFrom + offset, fromHeader.GetRawNum() * sizeof(Bit32));
            if (offset + fromHeader.GetRawNum() > receiveListOffset)
            {
                *pReceiveListBroken = true;
            }
        }
        else
        {
            result = fromPageTable.CopyMemoryToKernel(
                    pToMessage + offset * sizeof(Bit32), fromHeader.GetRawNum() * sizeof(Bit32),
                    pFromMessage + offset * sizeof(Bit32),
                    KMemoryState_FlagsMemory, KMemoryState_FlagsMemory, fromUserBuffer? KMemoryPermission_KernelRead: KMemoryPermission_UserRead,
                    KMemoryAttribute_Uncached, 0);
            if (offset + fromHeader.GetRawNum() > receiveListOffset)
            {
                *pReceiveListBroken = true;
            }
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }
    }

    NN_KERN_KTRACE_IPC_RECEIVE(pSession->GetParent()->GetIndex(), pRequest->GetIndex(), (fromHeader.GetData())->storage, (fromHeader.GetData() + 1)->storage);
    return result;

error_exit:
    CleanupMap(&fromPageTable, &toPageTable, pRequest);

    if (fromHeader.GetSpecialNum())
    {
        CleanupSpecialData(&toProcess, pToMsg, toMsgMaxLength);
    }
    if (!*pReceiveListBroken)
    {
        toMsg.Set(toHeader);
        if (toHeader.GetSpecialNum())
        {
            toMsg.Set(toSpecialHeader);
        }
    }
    return result;
}

}
#endif

/* ------------------------------------------------------------------------
        クラスメンバ
   ------------------------------------------------------------------------ */

bool KServerSession::IsSignaledImpl() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    if( m_pParent->IsClientClosed())
    {
        return true;
    }

    // リクエスト待ち行列にリクエストが存在し、現在のリクエストが未処理状態であれば
    // シグナル状態とします
    return !m_RequestQueue.empty() && (m_pCurrentRequest == nullptr);
}

/*!
    @brief      サーバーセッションがシグナル状態になったか。

    クライアントセッションからリクエストを受け取ることでシグナル状態になります。

    @param[in]  thread   未使用

    @return     リクエストをデキューできるのであれば true を返します。
*/
bool KServerSession::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    // リクエスト待ち行列にリクエストが存在し、現在のリクエストが未処理状態であれば
    // シグナル状態とします
    return IsSignaledImpl();
}

void KServerSession::Destroy()
{
    NN_KERN_THIS_ASSERT();

    // 同期待ちで待機しているリクエストを処理します
    m_pParent->OnServerClosed();

    CleanupRequest();

    m_pParent->Close();
    // このタイミングで既に this が破棄されている可能性があるので
    // 以降、this にアクセスしてはいけない。
}

//! 同期待ちで待機しているリクエストをキャンセル処理します
void KServerSession::CleanupRequest()
{
    NN_KERN_THIS_ASSERT();
#if NN_KERN_HAS_MMU
    KScopedLightLock locker(&m_Lock);

    // サーバセッションがすべて Close されたときに呼ばれる
    for (;;)
    {
        KSessionRequest* pRequest = nullptr;

        {
            KScopedSchedulingLock schedLocker;
            if(m_pCurrentRequest)
            {
                pRequest = m_pCurrentRequest;
                m_pCurrentRequest = nullptr;
            }
            else if(!m_RequestQueue.empty())
            {
                pRequest = &m_RequestQueue.front();
                m_RequestQueue.pop_front();
            }
        }
        if (!pRequest)
        {
            break;
        }

        KScopedAutoObject<KSessionRequest> autoCloser(pRequest);

        KWritableEvent* pEvent = pRequest->GetEvent();
        KThread* pClientThread = pRequest->GetThread();
        uintptr_t pClientMessage = pRequest->GetAddress();
        size_t clientBufferSize = pRequest->GetSize();

        KProcess* pServer = pRequest->GetServerProcess();
        KProcess* pClient = pClientThread? &pClientThread->GetParent(): nullptr;
        KProcessPageTable* pToPageTable   = (pClient? &pClient->GetPageTable(): nullptr);
        KProcessPageTable* pFromPageTable = (pServer? &pServer->GetPageTable(): nullptr);

        Result result = CleanupMap(pToPageTable, pFromPageTable, pRequest);

        if (pClientThread)
        {
            if (pEvent)
            {
                ReplyAsyncError(pClient, pClientMessage, clientBufferSize, (result.IsSuccess()? nn::svc::ResultSessionClosed(): result));

                Result resultCheck = pToPageTable->UnlockForIpcUserBuffer(pClientMessage, clientBufferSize);
                if (resultCheck.IsFailure())
                {
                    NN_WARNING(false, "failed to unlock IPC user buffer");
                }

                pEvent->Signal();
            }
            else
            {
                KScopedSchedulingLock schedLocker;
                if (pClientThread->GetState() == KThread::STATE_WAIT)
                {
                    pClientThread->SetSyncedObject(nullptr, (result.IsSuccess()? nn::svc::ResultSessionClosed(): result));
                    pClientThread->SetState(KThread::STATE_RUNNABLE);
                }
            }
        }
    }
#endif  // if NN_KERN_HAS_MMU
}

//! クライアントセッションが閉じたことの通知
void KServerSession::OnClientClosed()
{
    NN_KERN_THIS_ASSERT();

    KScopedLightLock locker(&m_Lock);

    // クライアントセッションがすべて Close されたときに呼ばれる
    // 要求があるのによばれるのは以下のケース
    // ・プロセス終了時のハンドルテーブル Close
    // 　→ Client Thread の 参照カウンタを減らし、プロセスが終了できるようにする
    // 　→ アンマップ処理は必要なし (= Threadをクリアする)
    // ・Async Req を出してEvent待ちをすることなくClose
    // 　→処理中の Async Req は Replyまで保留(マップを解除してはいけない)
    // 　→キュー内の Async Req は Closed 通知
#if NN_KERN_HAS_MMU
    KSessionRequest* pPrevCurrentRequest = nullptr;
    for (;;)
    {
        KSessionRequest* pRequest = nullptr;
        KThread* pThread = nullptr;
        KWritableEvent* pEvent = nullptr;
        bool isCurrentRequest = false;
        bool isTerminateRequest = false;
        {
            KScopedSchedulingLock schedLocker;
            if (m_pCurrentRequest && m_pCurrentRequest != pPrevCurrentRequest)
            {
                pRequest = m_pCurrentRequest;
                // 処理中に解放されないように Open しておく
                pRequest->Open();
                isCurrentRequest = true;
                pThread = pRequest->GetThread();
                pEvent = pRequest->GetEvent();

                if (pThread->IsTerminateRequested())
                {
                    pRequest->ClearThread();
                    pRequest->ClearEvent();
                    isTerminateRequest = true;
                }
                pPrevCurrentRequest = pRequest;
            }
            else if (!m_RequestQueue.empty())
            {
                // 処理後に解放される
                pRequest = &m_RequestQueue.front();
                m_RequestQueue.pop_front();
                pThread = pRequest->GetThread();
                pEvent = pRequest->GetEvent();
            }
        }
        if (!pRequest)
        {
            break;
        }

        NN_KERN_ASSERT(pThread);
        KScopedAutoObject<KSessionRequest> autoCloser(pRequest);

        if (isTerminateRequest)
        {
            pThread->Close();
            if (pEvent)
            {
                pEvent->Close();
            }
        }

        if (pEvent && !isCurrentRequest)
        {
            KProcess* pClient = &pThread->GetParent();
            KProcessPageTable* pPageTable = &pClient->GetPageTable();

            NN_KERN_ASSERT(pRequest->GetSendNum() == 0);
            NN_KERN_ASSERT(pRequest->GetReceiveNum() == 0);
            NN_KERN_ASSERT(pRequest->GetExchangeNum() == 0);


            ReplyAsyncError(pClient, pRequest->GetAddress(), pRequest->GetSize(), nn::svc::ResultSessionClosed());

            // result は無視
            Result resultCheck = pPageTable->UnlockForIpcUserBuffer(pRequest->GetAddress(), pRequest->GetSize());
            if (resultCheck.IsFailure())
            {
                NN_WARNING(false, "failed to unlock IPC user buffer");
            }
            pEvent->Signal();
        }
    }
#endif  // if NN_KERN_HAS_MMU

    NotifyAbort(nn::svc::ResultSessionClosed());
}

//! リクエストを受け取ったときの処理
Result KServerSession::ReceiveRequest(uintptr_t pServerMessage, size_t serverBufferSize)
{
    NN_KERN_THIS_ASSERT();

#if NN_KERN_HAS_MMU
    KScopedLightLock locker(&m_Lock);

    KSessionRequest *pRequest;
    KThread* pClientThread;

    {
        KScopedSchedulingLock schedLocker;

        if (m_pParent->IsClientClosed())
        {
            return nn::svc::ResultSessionClosed();
        }
        if (m_pCurrentRequest)
        {
            return nn::svc::ResultNotFound();
        }
        if (m_RequestQueue.empty())
        {
            return nn::svc::ResultNotFound();
        }

        pRequest = &m_RequestQueue.front();
        m_RequestQueue.pop_front();

        pClientThread = pRequest->GetThread();
        if (!pClientThread)
        {
            return nn::svc::ResultSessionClosed();
        }

        pClientThread->Open();
    }
    m_pCurrentRequest = pRequest;

    KScopedAutoObject<KThread> autoThreadCloser(pClientThread);

    // クライアント → サーバースレッド間でメッセージ交換します
    uintptr_t pClientMessage = pRequest->GetAddress();
    size_t clientBufferSize = pRequest->GetSize();
    bool receiveListBroken = false;

    Result result = ReceiveMessage(&receiveListBroken, pServerMessage, serverBufferSize, pClientThread, pClientMessage, clientBufferSize, pRequest, this);

    if (result.IsFailure())
    {
        // サーバ受信エラーが発生
        // ・カレント要求から削除
        // ・クライアントに通知
        // ・サーバはリトライしてもらう旨のResultを返す
        Result clientResult = result;

        {
            KScopedSchedulingLock schedLocker;
            NN_KERN_ASSERT(pRequest == m_pCurrentRequest);
            m_pCurrentRequest = nullptr;
            if (!m_RequestQueue.empty())
            {
                NotifyAvailable();
            }
        }

        {
            KScopedAutoObject<KSessionRequest> autoSessionRequestCloser(pRequest);

            KWritableEvent* pEvent = pRequest->GetEvent();
            if (pEvent)
            {
                // Client Async
                KProcess* pClient = &pClientThread->GetParent();
                KProcessPageTable* pToPageTable = &pClient->GetPageTable();

                if (clientResult.IsFailure())
                {
                    ReplyAsyncError(pClient, pClientMessage, clientBufferSize, clientResult);
                }

                // result は無視
                Result resultCheck = pToPageTable->UnlockForIpcUserBuffer(pClientMessage, clientBufferSize);
                if (resultCheck.IsFailure())
                {
                    NN_WARNING(false, "failed to unlock IPC user buffer");
                }
                pEvent->Signal();
            }
            else
            {
                KScopedSchedulingLock schedLocker;

                if (pClientThread->GetState() == KThread::STATE_WAIT)
                {
                    pClientThread->SetSyncedObject(nullptr, clientResult);
                    pClientThread->SetState(KThread::STATE_RUNNABLE);
                }
            }
        }

        // 受信指定リストが壊された場合は、ResultReceptionSpecificationListBroken() を返却
        // 受信指定リストが壊されなかった場合は、ResultNotFound() を返却してカーネル内でリトライする
        if (receiveListBroken)
        {
            result = nn::svc::ResultReceptionSpecificationListBroken();
        }
        else
        {
            result = nn::svc::ResultNotFound();
        }
    }

    return result;
#else   // if NN_KERN_HAS_MMU
    return nn::svc::ResultNotImplemented();
#endif  // if NN_KERN_HAS_MMU else
}

//! サーバーセッションからクライアントセッションに対してレスポンスを返します。
Result KServerSession::SendReply(uintptr_t pServerMessage, size_t serverBufferSize)
{
    // serverThread は必ず CurrntThreadです。
    NN_KERN_THIS_ASSERT();

#if NN_KERN_HAS_MMU
    KScopedLightLock locker(&m_Lock);

    KSessionRequest* pRequest;
    Result result;

    // Clinet Sessionを取得します
    // このタイミングで非シグナル状態からシグナル状態に変わることがあります。
    {
        KScopedSchedulingLock schedLocker;

        pRequest = m_pCurrentRequest;
        if (!pRequest)
        {
            return nn::svc::ResultInvalidState();
        }

        m_pCurrentRequest = nullptr;
        if (!m_RequestQueue.empty())
        {
            NotifyAvailable();
        }
    }

    KScopedAutoObject<KSessionRequest> autoCloser(pRequest);

    KThread* pClientThread = pRequest->GetThread();
    KWritableEvent* pEvent = pRequest->GetEvent();
    uintptr_t pClientMessage = pRequest->GetAddress();
    size_t clientBufferSize = pRequest->GetSize();

    bool isClosed = (!pClientThread || m_pParent->IsClientClosed());

    if (isClosed)
    {
        KProcess* pServer = pRequest->GetServerProcess();
        KProcess* pClient = pClientThread? &pClientThread->GetParent(): nullptr;

        KProcessPageTable* pToPageTable   = (pClient? &pClient->GetPageTable(): nullptr);
        KProcessPageTable* pFromPageTable = (pServer? &pServer->GetPageTable(): nullptr);

        Result resultCheck;
        result = CleanupServerHandle(pServerMessage, serverBufferSize);
        resultCheck = CleanupMap(pToPageTable, pFromPageTable, pRequest);
        if (result.IsSuccess())
        {
            result = resultCheck;
        }
    }
    else
    {
        //  サーバー → クライアントスレッド間でメッセージ交換します
        result = SendMessage(*pClientThread, pServerMessage, serverBufferSize, pClientMessage, clientBufferSize, pRequest, this);
    }


    Result clientResult = result;

    if (isClosed && result.IsSuccess())
    {
        clientResult = result = nn::svc::ResultSessionClosed();
    }
    else
    {
        // サーバはリプライ時に SessionClosed以外のエラーを返さない
        result = ResultSuccess();
    }

    if (pClientThread)
    {
        if (pEvent)
        {
            // Client Async
            KProcess* pClient = &pClientThread->GetParent();
            KProcessPageTable* pToPageTable = &pClient->GetPageTable();

            if (clientResult.IsFailure())
            {
                ReplyAsyncError(pClient, pClientMessage, clientBufferSize, clientResult);
            }

            // result は無視
            Result resultCheck = pToPageTable->UnlockForIpcUserBuffer(pClientMessage, clientBufferSize);
            if (resultCheck.IsFailure())
            {
                NN_WARNING(false, "failed to unlock IPC user buffer");
            }
            pEvent->Signal();
        }
        else
        {
            KScopedSchedulingLock schedLocker;

            if (pClientThread->GetState() == KThread::STATE_WAIT)
            {
                pClientThread->SetSyncedObject(nullptr, clientResult);
                pClientThread->SetState(KThread::STATE_RUNNABLE);
            }
        }
    }

    return result;
#else   // if NN_KERN_HAS_MMU
    NN_UNUSED(serverThread);
    NN_UNUSED(sendMessage);
    return nn::svc::ResultNotImplemented();
#endif  // if NN_KERN_HAS_MMU else
}

//!< クライアントセッションからリクエストが届いたことの通知
nn::Result KServerSession::OnRequest(KSessionRequest* pRequest)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT( KScheduler::IsSchedulerLocked() );

    if (m_pParent->IsServerClosed())
    {
        return nn::svc::ResultSessionClosed();
    }

    if (pRequest->GetEvent() == nullptr)
    {
        // リクエストスレッドを処理待ち状態にしつつ、リクエスト処理待ちキューに登録します。
        // →サーバーセッション(同期オブジェクト)がシグナル状態になります。
        KThread* pThread = pRequest->GetThread();
        if (pThread->IsTerminateRequested())
        {
            return nn::svc::ResultTerminateRequested();
        }
        pThread->SetState(KThread::STATE_WAIT);
    }

    NN_KERN_KTRACE_IPC_SEND(GetParent()->GetIndex(), pRequest->GetIndex());

    bool isEmpty = m_RequestQueue.empty();

    pRequest->Open();
    m_RequestQueue.push_back(*pRequest);

    if (isEmpty)
    {
        NotifyAvailable();
    }

    return ResultSuccess();
}

void KServerSession::Dump()
{
    KScopedLightLock locker(&m_Lock);
    {
        KScopedSchedulingLock schedLocker;
        NN_KERN_RELEASE_LOG("Dump Session %p\n", this);

        bool hasReq = false;
        if (m_pCurrentRequest)
        {
            KThread *pThread = m_pCurrentRequest->GetThread();
            NN_UNUSED(pThread);
            NN_KERN_RELEASE_LOG("    CurrentReq %p Thread=%p ID=%d\n", m_pCurrentRequest, pThread, static_cast<Bit32>(pThread? pThread->GetId(): 0xffffffff));
            hasReq = true;
        }

        for (RequestList::iterator it = m_RequestQueue.begin(); it != m_RequestQueue.end(); it++)
        {
            KThread *pThread = it->GetThread();
            NN_UNUSED(pThread);
            NN_KERN_RELEASE_LOG("    Req %p Thread=%p ID=%d\n", &*it, pThread, static_cast<Bit32>(pThread? pThread->GetId(): 0xffffffff));
            hasReq = true;
        }
        if (!hasReq)
        {
            NN_KERN_RELEASE_LOG("    Nothing\n");
        }
    }
}

}}
