﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include "kern_InterlockedSelect.h"
#include "kern_KThread.h"
#include "kern_KSpinLockMutex.h"
#include "kern_KHardwareTimerBase.h"
#include "kern_HardwareTimerSelect.h"
#include "kern_KInterruptTask.h"
#include "kern_InterruptNameSelect.h"
#include "kern_KThreadQueue.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_KSimpleLock.h"
#include "kern_KPriorityQueue.h"
#include "kern_KTrace.h"

namespace nn { namespace kern {

class KScheduler
    : public KInterruptTask
{
public:
    struct SchedulingCounter
    {
        bool        m_SchedulingRequired;
        bool        m_InterruptTaskThreadIsRunnable;
        bool        m_EnableIdleCount;
        uint64_t    m_IdleCount;
        KThread*    m_pHighestThread;
        void*       m_IdleStack;
    };

public:
    KScheduler();
    virtual ~KScheduler() {}
    KScheduler& operator=(const KScheduler&) = delete;
    KScheduler(const KScheduler&) = delete;

    void Initialize(KThread* pIdleThread);
    void Activate();

    uint64_t GetIdleCount() const { return m_SchedulingCounter.m_IdleCount; }

    KInterruptTask* OnInterrupt(int32_t interruptRequestNo)
    {
        NN_KERN_THIS_ASSERT();
        NN_UNUSED(interruptRequestNo);
        return GetSchedulingDummyTask();
    }
    virtual void DoInterruptTask(){}

    void SetInterruptTaskRunnable()
    {
        m_SchedulingCounter.m_InterruptTaskThreadIsRunnable = true;
        m_SchedulingCounter.m_SchedulingRequired = true;
    }
    void InterruptTaskThreadToRunnable();
    bool SwitchThread(KThread* pNextThread);
    void ForceScheduling()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
        NN_KERN_EQUAL_ASSERT(m_MyCore, GetCurrentCpuNo());
        m_SchedulingCounter.m_SchedulingRequired = true;
        Schedule();
    }
    KThread* GetPrevThread() const { return m_pPrevThread; }
    int64_t GetPrevSwitch() const { return m_PrevSwitch; }
    void RequestScheduling()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());

        m_SchedulingCounter.m_SchedulingRequired = true;

        if (CanSchedule())
        {
            Schedule();
        }
    }
    void RequestSchedulingOnInterrupt()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
        m_InInterruptHandler = true;
        RequestScheduling();
        m_InInterruptHandler = false;
    }
    void RescheduleOther() const;
    void RescheduleIfNeeded()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);

        {
            KDisableInterrupt di;

            GetCurrentThread().EnableDispatch();
            bool schedulingRequired = m_SchedulingCounter.m_SchedulingRequired;
            KCPU::DataMemoryBarrier();
            if (schedulingRequired)
            {
                Schedule();
            }
        }
    }
    KThread* GetIdleThread() const { return m_pIdleThread; }
    bool IsActive() const { return m_IsActive; }
    const KContext* SelectThread();
    void UpdateHighestThread(KThread* pThread);
    KThread* GetHighestThread() const { return m_SchedulingCounter.m_pHighestThread; }
public:
    static void OnThreadStateChanged(KThread* pThread, KThread::ThreadState state);
    static void OnThreadPriorityChanged(KThread* pThread, int32_t previousPriority);
    static void OnThreadCoreMaskChanged(KThread* pThread, const KAffinityMask& prevAffinity, int prevRunning);
    static bool CanSchedule() { return GetCurrentThread().GetDisableDispatchCount() == 0; }
    static bool IsSchedulerLocked() { return s_SchedulerMutex.IsLockedByMe(); }
    static void RotateReadyQueue(int coreNo, int priority);
    static void YieldWithCoreMigration();
    static void YieldWithoutCoreMigration();
    static void SwitchToAnyThread();
    static KInterruptTask* GetSchedulingDummyTask() { return reinterpret_cast<KInterruptTask*>(1); }
    static void ClearPrevThread(KThread* pThread);

    static void DisableScheduling()
    {
        NN_KERN_MIN_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
        GetCurrentThread().DisableDispatch();
    }
    static void EnableScheduling();
    static void EnableSchedulingAndScheduleAll();

    static void UpdateHighestThreads()
    {
        if (s_Update)
        {
            UpdateHighestThreadsImpl();
        }
    }

private:
    void Schedule(SchedulingCounter* pCounter);
    void Schedule()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_EQUAL_ASSERT(m_MyCore, GetCurrentCpuNo());
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
        GetCurrentThread().DisableDispatch();
        Schedule(&m_SchedulingCounter);
        GetCurrentThread().EnableDispatch();
    }
    friend class KScopedSchedulingLock;
    friend class KScopedSchedulingLockAndSleep;

    static bool s_Update;
    static KSpinLockMutex s_SchedulerMutex;
    static KPriorityQueue s_PriorityQueue;

    static void UpdateHighestThreadsImpl();
private:
    enum
    {
        CoreMigrationPriorityThreshold = 1, // 高優先度スレッドが動作するコアから移動させないための境界値
    };

    SchedulingCounter   m_SchedulingCounter;
    bool                m_InInterruptHandler;
    bool                m_IsActive;
    int                 m_MyCore;
    KThread*            m_pPrevThread;
    int64_t             m_PrevSwitch;
    KThread*            m_pIdleThread;
};

}}

