﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KResourceLimit.h"
#include "kern_KProcess.h"
#include <cstring>

namespace nn { namespace kern {

const nn::svc::Tick KResourceLimit::DefaultTimeout = nn::svc::Tick(TimeSpan::FromSeconds(10));

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KResourceLimit);

void KResourceLimit::Initialize()
{
    m_NumWaiter = 0;
    std::memset(m_Values, 0, sizeof(m_Values));
    std::memset(m_Limits, 0, sizeof(m_Limits));
    std::memset(m_Hints, 0, sizeof(m_Hints));
}

void KResourceLimit::Finalize()
{
}

Result KResourceLimit::SetLimitValue(nn::svc::LimitableResource resource, int64_t value)
{
    NN_KERN_THIS_ASSERT();
    KScopedLightLock locker(&m_Mutex);

    if (m_Values[resource] > value)
    {
        return nn::svc::ResultInvalidState();
    }

    m_Limits[resource] = value;
    NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
    NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);

    return ResultSuccess();
}

int64_t KResourceLimit::GetLimitValue(nn::svc::LimitableResource resource) const
{
    NN_KERN_THIS_ASSERT();
    int64_t value;

    {
        KScopedLightLock locker(&m_Mutex);
        value = m_Limits[resource];
        NN_KERN_ASSERT(value >= 0);
        NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
        NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);
    }

    return value;
}

int64_t KResourceLimit::GetCurrentValue(nn::svc::LimitableResource resource) const
{
    NN_KERN_THIS_ASSERT();
    int64_t value;

    {
        KScopedLightLock locker(&m_Mutex);
        value = m_Values[resource];
        NN_KERN_ASSERT(value >= 0);
        NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
        NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);
    }

    return value;
}

int64_t KResourceLimit::GetCurrentFreeValue(nn::svc::LimitableResource resource) const
{
    NN_KERN_THIS_ASSERT();
    int64_t value;

    {
        KScopedLightLock locker(&m_Mutex);
        NN_KERN_ASSERT(m_Values[resource] > 0);
        NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
        NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);
        value = m_Limits[resource] - m_Values[resource];
    }

    return value;
}

bool KResourceLimit::TestLimit(nn::svc::LimitableResource resource, int64_t value, int64_t timeout)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(value > 0);

    KScopedLightLock locker(&m_Mutex);

    NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);
    if (m_Hints[resource] >= m_Limits[resource])
    {
        NN_WARNING(false, "KResourceLimit::TestLimit failed. PID=%lld(%s) res= %d lim= %lld, cur= %lld hint= %lld req= %lld",
                GetCurrentProcess().GetId(),
                GetCurrentProcess().GetName(),
                resource, m_Limits[resource], m_Values[resource], m_Hints[resource], value);
        return false;
    }

    for (;;)
    {
        NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
        NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);

        if (m_Values[resource] >= m_Values[resource] + value)
        {
            break;
        }

        if (m_Values[resource] + value <= m_Limits[resource])
        {
            m_Values[resource] += value;
            m_Hints[resource] += value;
            return true;
        }

        if (m_Hints[resource] + value <= m_Limits[resource] && (timeout < 0 || Kernel::GetHardwareTimer().GetTick() < timeout))
        {
            m_NumWaiter++;
            m_Conditionvariable.Wait(&m_Mutex, timeout);
            m_NumWaiter--;
        }
        else
        {
            break;
        }
    }

    NN_WARNING(false, "KResourceLimit::TestLimit failed. PID=%lld(%s) res= %d lim= %lld, cur= %lld hint= %lld req= %lld",
            GetCurrentProcess().GetId(),
            GetCurrentProcess().GetName(),
            resource, m_Limits[resource], m_Values[resource], m_Hints[resource], value);

    return false;
}

bool KResourceLimit::TestLimit(nn::svc::LimitableResource resource, int64_t value)
{
    return TestLimit(resource, value, KHardwareTimer::GetTick() + DefaultTimeout);
}

void KResourceLimit::ReleaseLimit(nn::svc::LimitableResource resource, int64_t value)
{
    ReleaseLimit(resource, value, value);
}

void KResourceLimit::ReleaseLimit(nn::svc::LimitableResource resource, int64_t value, int64_t hint)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(value >= 0);
    NN_KERN_ASSERT(hint >= 0);

    KScopedLightLock locker(&m_Mutex);

    NN_KERN_ASSERT(m_Values[resource] <= m_Limits[resource]);
    NN_KERN_ASSERT(m_Hints[resource] <= m_Values[resource]);
    NN_KERN_ASSERT(value <= m_Values[resource]);
    NN_KERN_ASSERT(hint <= m_Hints[resource]);

    m_Values[resource] -= value;
    m_Hints[resource] -= hint;

    if (m_NumWaiter != 0)
    {
        m_Conditionvariable.Broadcast();
    }
}

}}
