﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_CPUSelect.h"
#include "kern_KThread.h"

namespace nn { namespace kern {

    class KPriorityQueue
    {
    private:
        typedef KThread::QueueEntry Entry;
        enum
        {
            NumPriority = (nn::svc::LowestThreadPriority - nn::svc::HighestThreadPriority + 1)
        };

        template <size_t N> class Bitmask
        {
        public:
            Bitmask()
            {
                for (size_t i = 0; i < sizeof(m_Flags) / sizeof(m_Flags[0]); i++)
                {
                    m_Flags[i] = 0;
                }
            }
            NN_FORCEINLINE void SetBit(int i)
            {
                uint32_t index = i / 64;
                uint32_t bit   = i % 64;
                m_Flags[index] |= (1ull << (63 - bit));
            }
            NN_FORCEINLINE void ClearBit(int i)
            {
                uint32_t index = i / 64;
                uint32_t bit   = i % 64;
                m_Flags[index] &= ~(1ull << (63 - bit));
            }
            NN_FORCEINLINE int CountLeadingZero() const
            {
                size_t i;
                for (i = 0; i < sizeof(m_Flags) / sizeof(m_Flags[0]); i++)
                {
                    if (m_Flags[i])
                    {
                        return 64 * i + __builtin_clzll(m_Flags[i]);
                    }
                }
                return 64 * i;
            }

            NN_FORCEINLINE int Next(int n) const
            {
                size_t i;
                for (i = (n + 1) / 64; i < sizeof(m_Flags) / sizeof(m_Flags[0]); i++)
                {
                    Bit64 flags = m_Flags[i];
                    if (((n + 1) % 64) != 0)
                    {
                        uint32_t bit = 63 - (n % 64);
                        flags &= ((1ull << bit) - 1);
                    }
                    if (flags)
                    {
                        return 64 * i + __builtin_clzll(flags);
                    }
                }
                return 64 * i;
            }


        private:
            Bit64 m_Flags[(N + 63) / 64];
        };

        class KPriorityQueueImpl
        {
        private:
            class KCoreQueue
            {
            public:
                KCoreQueue()
                {
                    for (int32_t coreNo = 0; coreNo < KCPU::NUM_CORE; ++coreNo)
                    {
                        m_Root[coreNo].Initialize();
                    }
                }

                NN_FORCEINLINE bool Enqueue(KThread* pAdd, int coreNo)
                {
                    KThread* pLast = m_Root[coreNo].pPrev;
                    Entry& lastE = (pLast != nullptr) ? pLast->m_PriorityQueueEntry[coreNo]: m_Root[coreNo];

                    pAdd->m_PriorityQueueEntry[coreNo].pPrev = pLast;
                    pAdd->m_PriorityQueueEntry[coreNo].pNext = nullptr;

                    lastE.pNext = pAdd;
                    m_Root[coreNo].pPrev = pAdd;

                    return (pLast == nullptr);
                }

                NN_FORCEINLINE bool EnqueueToTop(KThread* pAdd, int coreNo)
                {
                    KThread* pTop = m_Root[coreNo].pNext;
                    Entry& nextE = (pTop != nullptr) ? pTop->m_PriorityQueueEntry[coreNo]: m_Root[coreNo];

                    pAdd->m_PriorityQueueEntry[coreNo].pPrev = nullptr;
                    pAdd->m_PriorityQueueEntry[coreNo].pNext = pTop;

                    nextE.pPrev = pAdd;
                    m_Root[coreNo].pNext = pAdd;

                    return (pTop == nullptr);
                }

                NN_FORCEINLINE bool Remove(KThread* pRemove, int coreNo)
                {
                    KThread* pPrev = pRemove->m_PriorityQueueEntry[coreNo].pPrev;
                    KThread* pNext = pRemove->m_PriorityQueueEntry[coreNo].pNext;
                    Entry& prevE = (pPrev != nullptr) ? pPrev->m_PriorityQueueEntry[coreNo]: m_Root[coreNo];
                    Entry& nextE = (pNext != nullptr) ? pNext->m_PriorityQueueEntry[coreNo]: m_Root[coreNo];

                    prevE.pNext = pNext;
                    nextE.pPrev = pPrev;
                    return (m_Root[coreNo].pNext == nullptr);
                }

                NN_FORCEINLINE KThread* GetFront(int coreNo) const
                {
                    return m_Root[coreNo].pNext;
                }

            private:
                Entry m_Root[KCPU::NUM_CORE];
            };

        public:
            NN_FORCEINLINE void Enqueue(KThread* pThread, int priority, int coreNo)
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);
                if (NN_LIKELY(priority <= nn::svc::LowestThreadPriority))
                {
                    if (m_CoreQueue[priority].Enqueue(pThread, coreNo))
                    {
                        m_AvailablePriorityFlags[coreNo].SetBit(priority);
                    }
                }
            }

            NN_FORCEINLINE void EnqueueToTop(KThread* pThread, int priority, int coreNo)
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);
                if (NN_LIKELY(priority <= nn::svc::LowestThreadPriority))
                {
                    if (m_CoreQueue[priority].EnqueueToTop(pThread, coreNo))
                    {
                        m_AvailablePriorityFlags[coreNo].SetBit(priority);
                    }
                }
            }

            NN_FORCEINLINE void Remove(KThread* pThread, int priority, int coreNo)
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);
                if (NN_LIKELY(priority <= nn::svc::LowestThreadPriority))
                {
                    if (m_CoreQueue[priority].Remove(pThread, coreNo))
                    {
                        m_AvailablePriorityFlags[coreNo].ClearBit(priority);
                    }
                }
            }

            NN_FORCEINLINE void MoveToTop(KThread* pThread, int priority, int coreNo)
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);
                if (NN_LIKELY(priority <= nn::svc::LowestThreadPriority))
                {
                    m_CoreQueue[priority].Remove(pThread, coreNo);
                    m_CoreQueue[priority].EnqueueToTop(pThread, coreNo);
                }
            }

            NN_FORCEINLINE KThread* MoveToEnd(KThread* pThread, int priority, int coreNo)
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);
                if (NN_LIKELY(priority <= nn::svc::LowestThreadPriority))
                {
                    m_CoreQueue[priority].Remove(pThread, coreNo);
                    m_CoreQueue[priority].Enqueue(pThread, coreNo);
                    return m_CoreQueue[priority].GetFront(coreNo);
                }
                else
                {
                    return nullptr;
                }
            }

            NN_FORCEINLINE KThread* GetFront(int coreNo) const
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                int priority = m_AvailablePriorityFlags[coreNo].CountLeadingZero();
                if (priority <= nn::svc::LowestThreadPriority)
                {
                    return m_CoreQueue[priority].GetFront(coreNo);
                }
                else
                {
                    return nullptr;
                }
            }

            NN_FORCEINLINE KThread* GetFront(int priority, int coreNo) const
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                if (priority <= nn::svc::LowestThreadPriority)
                {
                    return m_CoreQueue[priority].GetFront(coreNo);
                }
                else
                {
                    return nullptr;
                }
            }

            NN_FORCEINLINE KThread* GetNext(int coreNo, KThread* pThread) const
            {
                NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
                KThread* pNext = pThread->m_PriorityQueueEntry[coreNo].pNext;
                if (pNext == nullptr)
                {
                    int priority = m_AvailablePriorityFlags[coreNo].Next(pThread->GetPriority());
                    if (priority <= nn::svc::LowestThreadPriority)
                    {
                        pNext = m_CoreQueue[priority].GetFront(coreNo);
                    }
                }
                return pNext;
            }

        private:
            KCoreQueue m_CoreQueue[NumPriority];
            Bitmask<NumPriority> m_AvailablePriorityFlags[KCPU::NUM_CORE];
        };


        KPriorityQueueImpl m_Fore;
        KPriorityQueueImpl m_Back;

        NN_FORCEINLINE void Enqueue(KThread* pThread, int priority)
        {
            NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);

            Bit64 affinity = pThread->GetAffinityMask().GetAffinityMask();
            int32_t foreCore = pThread->GetRunningProcessor();

            if (foreCore >= 0)
            {
                m_Fore.Enqueue(pThread, priority, foreCore);
                affinity &= ~(1ull << foreCore);
            }

            while (affinity)
            {
                int coreNo = __builtin_ctzll(affinity);
                affinity &= ~(1ull << coreNo);
                m_Back.Enqueue(pThread, priority, coreNo);
            }
        }

        NN_FORCEINLINE void EnqueueToTop(KThread* pThread, int priority)
        {
            NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);

            Bit64 affinity = pThread->GetAffinityMask().GetAffinityMask();
            int32_t foreCore = pThread->GetRunningProcessor();

            if (foreCore >= 0)
            {
                m_Fore.EnqueueToTop(pThread, priority, foreCore);
                affinity &= ~(1ull << foreCore);
            }

            while (affinity)
            {
                int coreNo = __builtin_ctzll(affinity);
                affinity &= ~(1ull << coreNo);
                m_Back.Enqueue(pThread, priority, coreNo);
            }
        }

        NN_FORCEINLINE void Remove(KThread* pThread, int priority)
        {
            NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority + 1);

            Bit64 affinity = pThread->GetAffinityMask().GetAffinityMask();
            int32_t foreCore = pThread->GetRunningProcessor();

            if (foreCore >= 0)
            {
                m_Fore.Remove(pThread, priority, foreCore);
                affinity &= ~(1ull << foreCore);
            }

            while (affinity)
            {
                int coreNo = __builtin_ctzll(affinity);
                affinity &= ~(1ull << coreNo);
                m_Back.Remove(pThread, priority, coreNo);
            }
        }

    public:

        NN_FORCEINLINE void PriorityChange(KThread* pThread, int prevPriority, bool isCurrent)
        {
            NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= prevPriority && prevPriority <= nn::svc::LowestThreadPriority + 1);
            int priority = pThread->GetPriority();
            Remove(pThread, prevPriority);
            if (isCurrent)
            {
                EnqueueToTop(pThread, priority);
            }
            else
            {
                Enqueue(pThread, priority);
            }
        }

        NN_FORCEINLINE void AffinityMaskChange(KThread* pThread, const KAffinityMask& prevAffinity, int prevRunning)
        {
            int priority = pThread->GetPriority();
            const KAffinityMask& affinity = pThread->GetAffinityMask();
            int32_t foreCore = pThread->GetRunningProcessor();

            for (int32_t coreNo = 0; coreNo < KCPU::NUM_CORE; ++coreNo)
            {
                if (prevAffinity.GetAffinity(coreNo))
                {
                    if (coreNo == prevRunning)
                    {
                        m_Fore.Remove(pThread, priority, coreNo);
                    }
                    else
                    {
                        m_Back.Remove(pThread, priority, coreNo);
                    }
                }
            }
            for (int32_t coreNo = 0; coreNo < KCPU::NUM_CORE; ++coreNo)
            {
                if (affinity.GetAffinity(coreNo))
                {
                    if (coreNo == foreCore)
                    {
                        m_Fore.Enqueue(pThread, priority, coreNo);
                    }
                    else
                    {
                        m_Back.Enqueue(pThread, priority, coreNo);
                    }
                }
            }
        }

        NN_FORCEINLINE void RunningCoreChange(KThread* pThread, int prevRunning)
        {
            int32_t foreCore = pThread->GetRunningProcessor();
            int priority = pThread->GetPriority();

            if (foreCore != prevRunning)
            {
                if (prevRunning >= 0)
                {
                    m_Fore.Remove(pThread, priority, prevRunning);
                }
                if (foreCore >= 0)
                {
                    m_Back.Remove(pThread, priority, foreCore);
                    m_Fore.Enqueue(pThread, priority, foreCore);
                }
                if (prevRunning >= 0)
                {
                    m_Back.Enqueue(pThread, priority, prevRunning);
                }
            }
        }

        NN_FORCEINLINE void RunningCoreChangeToTop(KThread* pThread, int prevRunning)
        {
            int32_t foreCore = pThread->GetRunningProcessor();
            int priority = pThread->GetPriority();

            if (foreCore != prevRunning)
            {
                if (prevRunning >= 0)
                {
                    m_Fore.Remove(pThread, priority, prevRunning);
                }
                if (foreCore >= 0)
                {
                    m_Back.Remove(pThread, priority, foreCore);
                    m_Fore.EnqueueToTop(pThread, priority, foreCore);
                }
                if (prevRunning >= 0)
                {
                    m_Back.Enqueue(pThread, priority, prevRunning);
                }
            }
        }

        KPriorityQueue() {}

        NN_FORCEINLINE void Enqueue(KThread* pAdd)
        {
            Enqueue(pAdd, pAdd->GetPriority());
        }

        NN_FORCEINLINE void Remove(KThread* pRemove)
        {
            Remove(pRemove, pRemove->GetPriority());
        }

        NN_FORCEINLINE KThread* GetNextSamePriority(int coreNo, KThread* pThread) const
        {
            return pThread->m_PriorityQueueEntry[coreNo].pNext;
        }

        NN_FORCEINLINE KThread* GetFront(int coreNo) const
        {
            return m_Fore.GetFront(coreNo);
        }

        NN_FORCEINLINE KThread* GetFront(int priority, int coreNo) const
        {
            return m_Fore.GetFront(priority, coreNo);
        }

        NN_FORCEINLINE KThread* GetNext(int coreNo, KThread* pThread) const
        {
            return m_Fore.GetNext(coreNo, pThread);
        }

        NN_FORCEINLINE KThread* GetFrontBack(int coreNo) const
        {
            return m_Back.GetFront(coreNo);
        }

        NN_FORCEINLINE KThread* GetFrontBack(int priority, int coreNo) const
        {
            return m_Back.GetFront(priority, coreNo);
        }

        NN_FORCEINLINE KThread* GetNextBack(int coreNo, KThread* pThread) const
        {
            return m_Back.GetNext(coreNo, pThread);
        }

        NN_FORCEINLINE void MoveToTop(KThread* pThread)
        {
            int32_t foreCore = pThread->GetRunningProcessor();
            int priority = pThread->GetPriority();
            m_Fore.MoveToTop(pThread, priority, foreCore);
        }

        NN_FORCEINLINE KThread* MoveToEnd(KThread* pThread)
        {
            int32_t foreCore = pThread->GetRunningProcessor();
            int priority = pThread->GetPriority();
            return m_Fore.MoveToEnd(pThread, priority, foreCore);
        }
    };
}}

