﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"
#include "kern_KUseSlabAllocator.h"
#include "kern_Result.h"
#include "kern_KTaggedAddress.h"
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace kern {

class KBlockInfo : public nn::util::IntrusiveListBaseNode<KBlockInfo>
{
public:
    KBlockInfo(){}

    void Initialize(KVirtualAddress addr, size_t numPages)
    {
        NN_KERN_THIS_ASSERT();
        m_Addr = addr;
        m_NumPages = numPages;
    }

    KVirtualAddress GetBlockAddr() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Addr;
    }

    size_t GetNumPages() const
    {
        NN_KERN_THIS_ASSERT();
        return m_NumPages;
    }

    bool IsSamePages(const KBlockInfo& rhs) const
    {
        return (this->m_Addr     == rhs.m_Addr)
            && (this->m_NumPages == rhs.m_NumPages);
    }

    bool IsLowerThan(KVirtualAddress addr) const;

    bool TryConcatenate(KVirtualAddress addr, size_t numPages);

private:
    KVirtualAddress m_Addr;
    size_t m_NumPages;
};

class KBlockInfoManager;
class KPageGroup
{
public:
    typedef nn::util::IntrusiveList<KBlockInfo, nn::util::IntrusiveListBaseNodeTraits<KBlockInfo>> BlockInfoList;
    KPageGroup() : m_pManager(nullptr)
    {
        NN_KERN_THIS_ASSERT();
    }

    ~KPageGroup()
    {
        NN_KERN_THIS_ASSERT();
    }

    void Initialize(KBlockInfoManager* pManager);
    void Finalize();
    Result AddBlock(KVirtualAddress addr, size_t numPages);

    BlockInfoList::const_iterator GetBlockBeginIter() const { return m_BlockList.begin();  }
    BlockInfoList::const_iterator GetBlockEndIter()   const { return m_BlockList.end();    }

    void Open() const;
    void Close() const;
    bool IsSamePages(const KPageGroup& rhs) const;

    size_t GetTotalNumPages() const;

private:
    BlockInfoList m_BlockList;
    KBlockInfoManager* m_pManager;
};

class KScopedPageGroup
{
private:
    KPageGroup m_PageGroup;

public:
    explicit KScopedPageGroup(KBlockInfoManager* pManager)
    {
        m_PageGroup.Initialize(pManager);
    }
    ~KScopedPageGroup()
    {
        m_PageGroup.Finalize();
    }
    KPageGroup& GetPageGroup() { return m_PageGroup; }
};




}}

