﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_KUseSlabAllocator.h"
#include "kern_KObjectContainer.h"
#include "kern_KLightMutex.h"
#include "kern_KAutoObject.h"

namespace nn { namespace kern {

class KObjectName :
    public KUseSlabAllocator<KObjectName>,
    public nn::util::IntrusiveListBaseNode<KObjectName>
{
public:
    static const int32_t NAME_LENGTH = 12;      //!< オブジェクトの名前の長さ

private:
    typedef nn::util::IntrusiveList<KObjectName, nn::util::IntrusiveListBaseNodeTraits<KObjectName>> NameList;

private:
    char            m_Name[NAME_LENGTH];    //!< オブジェクトの名前
    KAutoObject*    m_pObject;              //!< 関連付けるオブジェクト

    static NameList     s_List;             //!< 名前とオブジェクトの組み合わせリスト
    static KLightMutex  s_Mutex;            //!< リストを操作する際の同期オブジェクト

public:
    static Result       Name(KAutoObject* pObj, const char* name);
    static KAutoObject* Find(const char* name);
    static Result       Delete(KAutoObject* pObj, const char* name);

    /*!
        @brief      名前に対応する Object を取得します。

        @param[in]  name    オブジェクトに対応する名前

        @return     名前に対応する Object のポインタを返します。
                    型を間違えているか、一致するものがなれば NULL を返します。

    */
    template<typename T>
    static T* Find(const char* name)
    {
        KAutoObject* pObject = Find(name);
        if( pObject == NULL )
        {
            return NULL;
        }

        T* pT = pObject->DynamicCast<T*>();
        if( pT == NULL )
        {
            pObject->Close();
        }

        return pT;
    }

    template<typename T>
    static Result Delete(const char* name)
    {
        KAutoObject* pObject = Find(name);
        if (!pObject)
        {
            return nn::svc::ResultNotFound();
        }
        KScopedAutoObject<KAutoObject> autoCloser(pObject);

        T* pT = pObject->DynamicCast<T*>();
        if (!pT)
        {
            return nn::svc::ResultNotFound();
        }

        return Delete(pObject, name);
    }

private:
    static KAutoObject* FindImpl(const char* name);

    /*!
        @brief      KObjectName に対して pObj と名前を対応付けます。

        @param[in]  pObj    登録するオブジェクト
        @param[in]  name    オブジェクトに対応する名前

    */
    void            Initialize(KAutoObject* pObj, const char* name);

    /*!
        @brief      名前が等しいかどうか判定します。

        @param[in]  name    名前

        @return     名前が等しければ true を返します。

    */
    bool            Is(const char* name) const;

    KAutoObject*    GetObject() const { return m_pObject; }
};

}}

