﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Result.h"
#include "kern_KLightMutex.h"
#include "kern_KAutoObject.h"
#include "kern_KRbTree.h"

/*! @file

    @brief      KObjectContainer のクラス定義です。

    コンテナに対して、KAutoObjectの追加、削除をするオブジェクトです。

*/
namespace nn { namespace kern {

class KProcess;
struct KObjectContainerCompare
{
    int operator() (const KAutoObjectWithList& a, const KAutoObjectWithList& b)
    {
        Bit64 aid = a.GetId();
        Bit64 bid = b.GetId();

        if (aid < bid)
        {
            return -1;
        }
        if (bid < aid)
        {
            return 1;
        }
        return 0;
    }
};

class KObjectContainer
{
public:
    typedef IntrusiveRbTree<KAutoObjectWithList, IntrusiveRbTreeMemberNodeTraits<KAutoObjectWithList, &KAutoObjectWithList::m_Node>, KObjectContainerCompare> ObjectList;
    typedef ObjectList::iterator ObjectListIterator;

    class ListAccessor
        : public KScopedLightLock
    {
    private:
        ObjectList& m_List;

    public:
        explicit ListAccessor(KObjectContainer* pContainer)
            : KScopedLightLock(&pContainer->m_Mutex)
            , m_List(pContainer->m_ObjectList)
        {}

        ObjectListIterator GetBegin() const
        {
            return m_List.begin();
        }
        ObjectListIterator GetEnd() const
        {
            return m_List.end();
        }
        ObjectListIterator Find(ObjectList::const_reference ref) const
        {
            return m_List.find(ref);
        }
    };

    friend class ListAccessor;


private:
    ObjectList      m_ObjectList;           //!< コンテナ
    KLightMutex     m_Mutex;                //!< コンテナの排他制御オブジェクト

public:
    KObjectContainer();
    ~KObjectContainer();
    KObjectContainer& operator=(const KObjectContainer&) = delete;
    KObjectContainer(const KObjectContainer&) = delete;

    void Initialize() {}
    void Finalize() {}

    /*!
        @brief      コンテナにオブジェクトを登録します。

        @param[in]  pObj     対象のオブジェクト

    */
    Result              Register(KAutoObjectWithList* pObj);

    /*!
        @brief      コンテナからオブジェクトを削除します。

        @param[in]  pObj     対象のオブジェクト

        @return     常に成功を返します

    */
    Result              Unregister(KAutoObjectWithList* pObj);

    int32_t                 GetOwnedNum(KProcess* pOwner);

private:
    /*!
        @brief      コンテナにオブジェクトを登録します。

        @param[in]  pObj     対象のオブジェクト

        @return     常に成功を返します

    */
    Result              RegisterImpl(KAutoObjectWithList* pObj);
};



}}

