﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_KMemoryBlock.h"
#include "kern_KTaggedAddress.h"
#include "kern_KPageTableManager.h"

namespace nn {
    namespace kern {
class KMemoryBlockManagerUpdateBuffer
{
private:
    KMemoryBlock* m_pBlocks[2];
    size_t m_Index;
    KMemoryBlockResourceManager* m_pMemoryBlockManager;

public:
    explicit KMemoryBlockManagerUpdateBuffer(Result* pResult, KMemoryBlockResourceManager* pMemoryBlockManager): m_Index(0), m_pMemoryBlockManager(pMemoryBlockManager)
    {
        for (size_t i = 0; i < sizeof(m_pBlocks) / sizeof(*m_pBlocks); i++)
        {
            m_pBlocks[i] = nullptr;
        }
        for (size_t i = 0; i < sizeof(m_pBlocks) / sizeof(*m_pBlocks); i++)
        {
            m_pBlocks[i] = m_pMemoryBlockManager->Allocate();
            if (!m_pBlocks[i])
            {
                *pResult = nn::svc::ResultOutOfResource();
                return;
            }
        }
        *pResult = ResultSuccess();
    }

    ~KMemoryBlockManagerUpdateBuffer()
    {
        for (size_t i = 0; i < sizeof(m_pBlocks) / sizeof(*m_pBlocks); i++)
        {
            if (m_pBlocks[i])
            {
                m_pMemoryBlockManager->Free(static_cast<KMemoryBlock*>(m_pBlocks[i]));
            }
        }
    }

    KMemoryBlock* Allocate()
    {
        NN_KERN_ABORT_UNLESS(m_Index < sizeof(m_pBlocks) / sizeof(*m_pBlocks));
        NN_KERN_ABORT_UNLESS(m_pBlocks[m_Index]);
        KMemoryBlock* pBlock = m_pBlocks[m_Index];
        m_pBlocks[m_Index++] = nullptr;
        return pBlock;
    }

    void Free(KMemoryBlock* pBlock)
    {
        NN_KERN_ABORT_UNLESS(pBlock);
        NN_KERN_ABORT_UNLESS(m_Index <= sizeof(m_pBlocks) / sizeof(*m_pBlocks));
        if (m_Index == 0)
        {
            m_pMemoryBlockManager->Free(static_cast<KMemoryBlock*>(pBlock));
        }
        else
        {
            m_pBlocks[--m_Index] = pBlock;
        }
    }
};

struct KMemoryBlockCompare
{
    int operator() (const KMemoryBlock& a, const KMemoryBlock& b)
    {
        KProcessAddress addr = a.GetBaseAddress();

        if (addr < b.GetBaseAddress())
        {
            return -1;
        }
        if (addr <= b.GetLastAddress())
        {
            return 0;
        }
        return 1;
    }
};
class KMemoryBlockManager
{
public:
    typedef IntrusiveRbTree<KMemoryBlock, IntrusiveRbTreeBaseNodeTraits<KMemoryBlock>, KMemoryBlockCompare> MemoryBlockList;
private:
    MemoryBlockList             m_MemoryBlockList;
    KProcessAddress             m_Begin;
    KProcessAddress             m_End;

public:
    Result  Initialize(KProcessAddress begin, KProcessAddress end, KMemoryBlockResourceManager* pMemoryBlockManager);
    void    Finalize(KMemoryBlockResourceManager* pMemoryBlockManager);

    void    Update(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages, KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute);
    void    UpdateLock(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages, void (KMemoryBlock::*pLock)(KMemoryPermission newPermission),KMemoryPermission newPermission);

    void    UpdateIfMatch(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages,
            KMemoryState testState, KMemoryPermission testPermission, KMemoryAttribute testAttribute,
            KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute);

    const KMemoryBlock* FindByAddress(KProcessAddress addr) const;
    KProcessAddress FindFreeAreaInRegion(KProcessAddress regionBegin, size_t regionNumPages, size_t numPages, size_t align, size_t alignOffset, size_t guardPages) const;

    KMemoryBlockManager::MemoryBlockList::iterator FindIterator(KProcessAddress addr) const;
    KMemoryBlockManager::MemoryBlockList::const_iterator EndIterator() const { return m_MemoryBlockList.end(); }

    void    Dump() const;
    bool    Check() const;
};

    }
}

