﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_BaseType.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_Assert.h"
#include "kern_KTaggedAddress.h"
#include "kern_KUseSlabAllocator.h"
#include "kern_KRbTree.h"

namespace nn { namespace kern {

enum KMemoryState: uint32_t
{
    KMemoryState_Mask               = (0xFFu <<  0),
    KMemoryState_FlagsProtect       = (1u << 8),        // PROTECT 可能
    KMemoryState_FlagsDebug         = (1u << 9),        // デバッガが特権アクセス可能
    KMemoryState_FlagsIpc           = (1u << 10),       // IPC 可能
    KMemoryState_FlagsNonDeviceIpc  = (1u << 11),       // IPC 可能
    KMemoryState_FlagsNonSecureIpc  = (1u << 12),       // IPC 可能
    KMemoryState_FlagsMapped        = (1u << 13),       // 何かがマッピングされている
    KMemoryState_FlagsCode          = (1u << 14),       // EX / RO
    KMemoryState_FlagsAlias         = (1u << 15),       // Alias可能
    KMemoryState_FlagsDll           = (1u << 16),       // Dll Alias
    KMemoryState_FlagsTransfer      = (1u << 17),       // Transfer可能
    KMemoryState_FlagsQueryPhys     = (1u << 18),       // QueryPhys可能
    KMemoryState_FlagsDeviceShared  = (1u << 19),       // Device Address Space Map 可能
    KMemoryState_FlagsDeviceSharedAligned  = (1u << 20),       // Device Address Space Map 可能
    KMemoryState_FlagsIpcBuffer     = (1u << 21),       // IPC User Buffer可能
    KMemoryState_FlagsMemory        = (1u << 22),       // 通常メモリ(Kernel Heap アドレス)
    KMemoryState_FlagsMapProcess    = (1u << 23),       // MapProcessMemory可能 (Code, CodeData)
    KMemoryState_FlagsUncache       = (1u << 24),       // 非キャッシュ可能
    KMemoryState_FlagsCodeMemory    = (1u << 25),       // コードメモリ化可能


    KMemoryState_DataFlags        =                                        KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc | KMemoryState_FlagsIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned
                                                                         | KMemoryState_FlagsAlias | KMemoryState_FlagsProtect | KMemoryState_FlagsTransfer
                                                                         | KMemoryState_FlagsIpcBuffer | KMemoryState_FlagsUncache,
    KMemoryState_CodeFlags        =                                        KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc | KMemoryState_FlagsIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned
                                                                         | KMemoryState_FlagsCode | KMemoryState_FlagsDebug,


    KMemoryState_Free             = nn::svc::MemoryState_Free,
    KMemoryState_Io               = nn::svc::MemoryState_Io              | KMemoryState_FlagsMapped,
    KMemoryState_Static           = nn::svc::MemoryState_Static          | KMemoryState_FlagsMapped | KMemoryState_FlagsQueryPhys,
    KMemoryState_Normal           = nn::svc::MemoryState_Normal          | KMemoryState_DataFlags | KMemoryState_FlagsCodeMemory,
    KMemoryState_Shared           = nn::svc::MemoryState_Shared          | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory,
    KMemoryState_GeneratedCode    = nn::svc::MemoryState_GeneratedCode   | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsDebug,
    KMemoryState_CodeOut          = nn::svc::MemoryState_CodeOut         | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory,
    KMemoryState_Code             = nn::svc::MemoryState_Code            | KMemoryState_CodeFlags | KMemoryState_FlagsMapProcess,
    KMemoryState_CodeData         = nn::svc::MemoryState_CodeData        | KMemoryState_DataFlags | KMemoryState_FlagsMapProcess | KMemoryState_FlagsCodeMemory,
    KMemoryState_AliasCode        = nn::svc::MemoryState_AliasCode       | KMemoryState_CodeFlags | KMemoryState_FlagsMapProcess | KMemoryState_FlagsDll,
    KMemoryState_AliasCodeData    = nn::svc::MemoryState_AliasCodeData   | KMemoryState_DataFlags | KMemoryState_FlagsMapProcess | KMemoryState_FlagsCodeMemory| KMemoryState_FlagsDll,
    KMemoryState_Ipc              = nn::svc::MemoryState_Ipc             | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc | KMemoryState_FlagsIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned,
    KMemoryState_NonSecureIpc     = nn::svc::MemoryState_NonSecureIpc    | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned,
    KMemoryState_NonDeviceIpc     = nn::svc::MemoryState_NonDeviceIpc    | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc
                                                                         | KMemoryState_FlagsDeviceShared,
    KMemoryState_Stack            = nn::svc::MemoryState_Stack           | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc | KMemoryState_FlagsIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned,
    KMemoryState_ThreadLocal      = nn::svc::MemoryState_ThreadLocal     | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory,
    KMemoryState_Transfered       = nn::svc::MemoryState_Transfered      | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc | KMemoryState_FlagsIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned
                                                                         | KMemoryState_FlagsUncache,
    KMemoryState_SharedTransfered = nn::svc::MemoryState_SharedTransfered| KMemoryState_FlagsMapped | KMemoryState_FlagsMemory | KMemoryState_FlagsQueryPhys
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc
                                                                         | KMemoryState_FlagsDeviceShared | KMemoryState_FlagsDeviceSharedAligned,
    KMemoryState_SharedCode       = nn::svc::MemoryState_SharedCode      | KMemoryState_FlagsMapped | KMemoryState_FlagsMemory
                                                                         | KMemoryState_FlagsNonDeviceIpc | KMemoryState_FlagsNonSecureIpc,
    KMemoryState_Kernel           = nn::svc::MemoryState_Kernel          | KMemoryState_FlagsMapped,

    KMemoryState_None           = 0,
    KMemoryState_All            = 0xffffffff,
};

enum KMemoryAttribute: uint8_t
{
    KMemoryAttribute_Mask           = 0xff,
    KMemoryAttribute_None           = 0,
    KMemoryAttribute_Locked         = nn::svc::MemoryAttribute_Locked,
    KMemoryAttribute_IpcLocked      = nn::svc::MemoryAttribute_IpcLocked,
    KMemoryAttribute_DeviceShared   = nn::svc::MemoryAttribute_DeviceShared,
    KMemoryAttribute_Uncached       = nn::svc::MemoryAttribute_Uncached,

    KMemoryAttribute_All            = 0xff,
    KMemoryAttribute_SetMask        = KMemoryAttribute_Uncached,
};

enum KMemoryPermission: uint8_t
{
    KMemoryPermission_UserMask          = 0x7,
    KMemoryPermission_None              = 0,
    KMemoryPermission_KernelRead        = (1u << 3),
    KMemoryPermission_KernelWrite       = (1u << 4),
    KMemoryPermission_KernelExecute     = (1u << 5),
    KMemoryPermission_KernelReadWrite   = KMemoryPermission_KernelRead | KMemoryPermission_KernelWrite,
    KMemoryPermission_KernelReadExecute = KMemoryPermission_KernelRead | KMemoryPermission_KernelExecute,
    KMemoryPermission_UserRead          = nn::svc::MemoryPermission_Read    | KMemoryPermission_KernelRead,
    KMemoryPermission_UserWrite         = nn::svc::MemoryPermission_Write   | KMemoryPermission_KernelWrite,
    KMemoryPermission_UserExecute       = nn::svc::MemoryPermission_Execute,
    KMemoryPermission_UserReadWrite     = KMemoryPermission_UserWrite   | KMemoryPermission_UserRead,
    KMemoryPermission_UserReadExecute   = KMemoryPermission_UserExecute | KMemoryPermission_UserRead,

    KMemoryPermission_IpcLockChangeMask = KMemoryPermission_UserReadWrite,
    KMemoryPermission_All               = 0xff,
};

inline KMemoryPermission ConvertToKMemoryPermission(nn::svc::MemoryPermission permission)
{
    return static_cast<KMemoryPermission>((permission & KMemoryPermission_UserMask) | ((permission & (KMemoryPermission_UserReadWrite)) << 3) | KMemoryPermission_KernelRead);
}

struct KMemoryInfo
{
    uintptr_t           baseAddress;
    size_t              size;
    KMemoryState        state;
    KMemoryPermission   permission;
    KMemoryAttribute    attribute;
    KMemoryPermission   permissionOrig;
    uint16_t            ipcLockCount;
    uint16_t            deviceSharedCount;

    nn::svc::MemoryInfo ConvertToSvcMemoryInfo() const
    {
        nn::svc::MemoryInfo osInfo;
        osInfo.baseAddress  = this->baseAddress;
        osInfo.size         = this->size;
        osInfo.state        = static_cast<nn::svc::MemoryState>(this->state & KMemoryState_Mask);
        osInfo.permission   = static_cast<nn::svc::MemoryPermission>(this->permission & KMemoryPermission_UserMask);
        osInfo.attribute    = static_cast<nn::svc::MemoryAttribute>(this->attribute & KMemoryAttribute_Mask);
        osInfo.ipcCount     = this->ipcLockCount;
        osInfo.deviceCount  = this->deviceSharedCount;
        return osInfo;
    }
};

class KMemoryBlock :
    public IntrusiveRbTreeBaseNode<KMemoryBlock>
{
public:
    KMemoryBlock(){}
    ~KMemoryBlock(){}

    void Initialize(KProcessAddress addr, size_t numPages, KMemoryState memoryState, KMemoryPermission permission, KMemoryAttribute attribute)
    {
        NN_KERN_THIS_ASSERT();
        m_Addr              = addr;
        m_NumPages          = numPages;
        m_Permission        = permission;
        m_MemoryState       = memoryState;
        m_Attribute         = attribute;
        m_PermissionOrig    = KMemoryPermission_None;
        m_IpcLockCount      = 0;
        m_DeviceSharedCount = 0;
    }

    void Update(KMemoryState memoryState, KMemoryPermission permission, KMemoryAttribute attribute)
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ASSERT(m_PermissionOrig == KMemoryPermission_None);
        NN_KERN_ASSERT((m_Attribute & KMemoryAttribute_IpcLocked) == 0);
        m_Permission  = permission;
        m_MemoryState = memoryState;
        m_Attribute   = static_cast<KMemoryAttribute>(attribute | (m_Attribute & (KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared)));
    }

    KProcessAddress GetBaseAddress() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Addr;
    }

    KProcessAddress GetLastAddress() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Addr + m_NumPages * NN_KERN_FINEST_PAGE_SIZE - 1;
    }

    size_t GetNumPages() const
    {
        NN_KERN_THIS_ASSERT();
        return m_NumPages;
    }

    bool Overlaps(KProcessAddress addr, size_t numPages) const;
    bool Includes(KProcessAddress addr, size_t numPages) const;
    bool Includes(KProcessAddress addr) const;
    bool IsIncluded(KProcessAddress addr, size_t numPages) const;
    KMemoryInfo GetMemoryInfo() const;

    bool Is(KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute) const
    {
        NN_KERN_THIS_ASSERT();
        return (this->m_Permission  == permission)
            && (this->m_MemoryState == state)
            && ((this->m_Attribute | KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared) == (attribute | KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared));
    }
    bool IsSameProperty(const KMemoryBlock& rhs) const
    {
        NN_KERN_THIS_ASSERT();
        return (this->m_Permission        == rhs.m_Permission)
            && (this->m_MemoryState       == rhs.m_MemoryState)
            && (this->m_Attribute         == rhs.m_Attribute)
            && (this->m_PermissionOrig    == rhs.m_PermissionOrig)
            && (this->m_IpcLockCount      == rhs.m_IpcLockCount)
            && (this->m_DeviceSharedCount == rhs.m_DeviceSharedCount);
    }

    void Split(KMemoryBlock* pBlock, KProcessAddress addr);
    void Add(size_t numPages);

    void IpcLock(KMemoryPermission newPermission)
    {
        NN_KERN_ASSERT((m_Attribute & KMemoryAttribute_IpcLocked) == KMemoryAttribute_IpcLocked || m_IpcLockCount == 0);
        uint16_t after = ++m_IpcLockCount;
        NN_KERN_ABORT_UNLESS(after > 0);
        if (after == 1)
        {
            NN_KERN_ASSERT(m_PermissionOrig == KMemoryPermission_None);
            NN_KERN_ASSERT((m_Permission | newPermission) == m_Permission);
            NN_KERN_ASSERT(((m_Permission & KMemoryPermission_UserExecute) != KMemoryPermission_UserExecute) || (newPermission == KMemoryPermission_UserRead));
            m_PermissionOrig = m_Permission;
            m_Permission = static_cast<KMemoryPermission>((newPermission & KMemoryPermission_IpcLockChangeMask) | (m_PermissionOrig & ~KMemoryPermission_IpcLockChangeMask));
        }
        m_Attribute = static_cast<KMemoryAttribute>(m_Attribute | KMemoryAttribute_IpcLocked);
    }

    void IpcUnlock(KMemoryPermission newPermission)
    {
        NN_UNUSED(newPermission);
        NN_KERN_ASSERT((m_Attribute & KMemoryAttribute_IpcLocked) == KMemoryAttribute_IpcLocked);
        uint16_t prev = m_IpcLockCount--;
        NN_KERN_ABORT_UNLESS(prev > 0);
        if (prev == 1)
        {
            NN_KERN_ASSERT(m_PermissionOrig != KMemoryPermission_None);
            m_Permission = m_PermissionOrig;
            m_PermissionOrig = KMemoryPermission_None;
            m_Attribute = static_cast<KMemoryAttribute>(m_Attribute & ~KMemoryAttribute_IpcLocked);
        }
    }

    void DeviceShared(KMemoryPermission newPermission)
    {
        NN_UNUSED(newPermission);
        NN_KERN_ASSERT((m_Attribute & KMemoryAttribute_DeviceShared) == KMemoryAttribute_DeviceShared || m_DeviceSharedCount == 0);
        m_Attribute = static_cast<KMemoryAttribute>(m_Attribute | KMemoryAttribute_DeviceShared);
        uint16_t after = ++m_DeviceSharedCount;
        NN_KERN_ABORT_UNLESS(after > 0);
    }

    void DeviceUnshared(KMemoryPermission newPermission)
    {
        NN_UNUSED(newPermission);
        NN_KERN_ASSERT((m_Attribute & KMemoryAttribute_DeviceShared) == KMemoryAttribute_DeviceShared);
        uint16_t prev = m_DeviceSharedCount--;
        NN_KERN_ABORT_UNLESS(prev > 0);
        if (prev == 1)
        {
            m_Attribute = static_cast<KMemoryAttribute>(m_Attribute & ~KMemoryAttribute_DeviceShared);
        }
    }

private:
    KProcessAddress     m_Addr;
    size_t              m_NumPages;
    KMemoryState        m_MemoryState;
    uint16_t            m_IpcLockCount;
    uint16_t            m_DeviceSharedCount;

    KMemoryPermission   m_Permission;
    KMemoryPermission   m_PermissionOrig;
    KMemoryAttribute    m_Attribute;
};

}}

