﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_Assert.h"
#include "kern_KSlabAllocator.h"

// nw4r/misc.hより移植
// TORIAEZUここに置いた
#ifdef NW4R_COMPILER_WITH_W64            // 64bitCPUの場合はこちら
typedef __w64 unsigned long IntPtr;
typedef __w64 signed long PtrDiff;
#else                                    // 32bitCPUの場合
typedef unsigned long IntPtr;
typedef signed long PtrDiff;
#endif

// TORIAEZUここに置いた
#ifndef NW4R_ASSERT
#define NW4R_ASSERT(exp)
#endif
#ifndef NW4R_NULL_ASSERT
#define NW4R_NULL_ASSERT(x)
#endif

namespace nn { namespace kern {

namespace old_lib {

/* 前方宣言。 ***************************************************************/

class LinkedListNode;

namespace detail {

class LinkedListImpl;

} // namespace detail

/* イテレータ ***************************************************************/

namespace detail {

template <
    typename T,
    typename Difference = PtrDiff,
    typename Pointer    = T*,
    typename Reference  = T& >
class Iterator
{
public:
    typedef T           value_type;
    typedef Difference  difference_type;
    typedef Pointer     pointer;
    typedef Reference   reference;
};


template <typename TIt>
class ReverseIterator
    : public detail::Iterator<
    typename TIt::value_type,
    typename TIt::difference_type,
    typename TIt::pointer,
    typename TIt::reference>
{
private:  /****************************************************************/

    typedef detail::Iterator<
        typename TIt::value_type,
        typename TIt::difference_type,
        typename TIt::pointer,
        typename TIt::reference >
    Base;

public:  /*****************************************************************/

    typedef ReverseIterator<TIt> Self;
    typedef typename Base::pointer     pointer;
    typedef typename Base::reference   reference;

    /**************************************************************************/

    explicit ReverseIterator() {}
    explicit ReverseIterator(TIt it) : mCurrent(it) {}
    template<typename T>
    explicit ReverseIterator(const ReverseIterator<T>& rhs ) : mCurrent( rhs.GetBase() ) {}

    TIt       GetBase()    const { return mCurrent; }
    reference operator*()  const
    {
        TIt tmp = mCurrent;
        return (*--tmp);
    }
    pointer   operator->() const { return (&**this); }
    Self& operator++()
    {
        (void)--mCurrent;
        return *this;
    }
    Self  operator++(int)
    {
        Self tmp = *this;
        (void)--mCurrent;
        return tmp;
    }
    Self& operator--()
    {
        (void)++mCurrent;
        return *this;
    }
    Self  operator--(int)
    {
        Self tmp = *this;
        (void)++mCurrent;
        return tmp;
    }

    friend bool operator==(const Self &r1,const Self &r2) { return r1.mCurrent == r2.mCurrent; }
    friend bool operator!=(const Self &r1,const Self &r2) { return !(r1==r2); }

private:  /****************************************************************/

    TIt mCurrent;
};

} // namespace detail

/****************************************************************************
双方向リンクリストノード
*****************************************************************************/

class LinkedListNode
{
public:  /*****************************************************************/

    typedef LinkedListNode Self;

    /**************************************************************************/

    explicit LinkedListNode()
    : mNext(NULL),
      mPrev(NULL)
    {}
    LinkedListNode& operator=(const LinkedListNode&) = delete;
    LinkedListNode(const LinkedListNode&) = delete;

    /**************************************************************************/

    Self* GetNext() const {return mNext;}
    Self* GetPrev() const {return mPrev;}

    /* 演算子。 ***************************************************************/

    friend bool operator==(const Self &r1,const Self &r2) { return &r1 == &r2; }
    friend bool operator!=(const Self &r1,const Self &r2) { return !(r1 == r2); }

private:  /****************************************************************/
    Self* mNext;
    Self* mPrev;

    /**************************************************************************/

    friend class detail::LinkedListImpl;  // mNext,mPrevを使用するため。
};

namespace detail {

/****************************************************************************
双方向リンクリスト実装クラス
*****************************************************************************/

class LinkedListImpl
{
public:  /*****************************************************************/

    typedef LinkedListImpl        Self;
    typedef unsigned long       size_type;
    typedef PtrDiff             difference_type;

    // リンクリストノードクラス。
    typedef LinkedListNode        Node;
    typedef Node                value_type;
    typedef value_type*         pointer;
    typedef const value_type*   const_pointer;
    typedef value_type&         reference;
    typedef const value_type&   const_reference;

    /**************************************************************************/

    /**************************************************************************/
    //*/ イテレータ。

    class Iterator;
    class ConstIterator;

    class Iterator
        : public detail::Iterator<value_type>
    {
    public:  /***************************************************************/

        typedef Iterator TIt;
        typedef detail::Iterator<value_type> TBaseIt;

        /************************************************************************/

        explicit Iterator()
        : mPointer(NULL)
        {}

        /************************************************************************/

        reference operator*() const
        {
            NW4R_NULL_ASSERT(mPointer);
            return *mPointer;
        }
        pointer operator->()  const
        {
            return mPointer;
        }

        TIt &operator++()
        {
            mPointer=mPointer->GetNext();
            return *this;
        }
        TIt operator++(int)
        {
            const TIt it(*this);
            (void)++*this;
            return it;
        }
        TIt &operator--()
        {
            mPointer=mPointer->GetPrev();
            return *this;
        }
        TIt operator--(int)
        {
            const TIt it(*this);
            (void)--*this;
            return it;
        }

        friend bool operator==(const TIt& it1, const TIt& it2) { return it1.mPointer == it2.mPointer; }
        friend bool operator!=(const TIt& it1, const TIt& it2) { return !(it1 == it2); }

    private:  /**************************************************************/

        explicit Iterator(pointer p) : mPointer(p) {}

        /************************************************************************/

        pointer mPointer;

        /************************************************************************/

        friend class LinkedListImpl;   // コンストラクタを使用するため。
        friend class ConstIterator;  // mPointerを使用するため。
    };
    typedef Iterator Iterator_alias_;  // ConstIterator内部で使用。

    class ConstIterator
        : public detail::Iterator<value_type>
    {
    public:  /***************************************************************/

        typedef ConstIterator TIt;
        typedef detail::Iterator<value_type> TBaseIt;
        typedef const_pointer pointer;
        typedef const_reference reference;

        /************************************************************************/

        explicit ConstIterator()
        : mPointer(NULL)
        {}
        explicit ConstIterator(Iterator_alias_ it) : mPointer(it.mPointer) {}

        /************************************************************************/

        reference operator*() const
        {
            NW4R_NULL_ASSERT(mPointer);
            return *mPointer;
        }
        pointer operator->()  const { return mPointer; }

        TIt &operator++()
        {
            mPointer = mPointer->GetNext();
            return *this;
        }
        TIt operator++(int)
        {
            const TIt it(*this);
            (void)++*this;
            return it;
        }
        TIt &operator--()
        {
            mPointer = mPointer->GetPrev();
            return *this;
        }
        TIt operator--(int)
        {
            const TIt it(*this);
            (void)--*this;
            return it;
        }

        friend bool operator==(const TIt& it1, const TIt& it2) { return it1.mPointer == it2.mPointer; }
        friend bool operator!=(const TIt& it1, const TIt& it2) { return !(it1 == it2);}

    private:  /**************************************************************/

        explicit ConstIterator(pointer p) : mPointer(p) {}

        /************************************************************************/

        pointer mPointer;

        /************************************************************************/

        friend class LinkedListImpl;  // コンストラクタを使用するため。
    };

    typedef detail::ReverseIterator<Iterator>      ReverseIterator;
    typedef detail::ReverseIterator<ConstIterator> ConstReverseIterator;

    /**************************************************************************/
    //*/ コンストラクタ/デストラクタ。

    explicit LinkedListImpl() { Initialize_(); }
    ~LinkedListImpl();
    LinkedListImpl& operator=(const LinkedListImpl&) = delete;
    LinkedListImpl(const LinkedListImpl&) = delete;

    /**************************************************************************/
    //*/ 状態の問い合わせ、イテレータの取得など。

    size_type       GetSize() const {return mSize;}
    bool            IsEmpty() const {return mSize==0;}

    Iterator        GetBeginIter()       {return Iterator(mNode.GetNext());}
    ConstIterator   GetBeginIter() const {return ConstIterator(mNode.GetNext());}
    Iterator        GetEndIter()         {return Iterator(&mNode);}
    ConstIterator   GetEndIter()   const {return ConstIterator(const_cast<Node*>(&mNode));}

    ReverseIterator      GetBeginReverseIter()       {return ReverseIterator(GetEndIter());}
    ConstReverseIterator GetBeginReverseIter() const {return ConstReverseIterator(GetEndIter());}
    ReverseIterator      GetEndReverseIter()         {return ReverseIterator(GetBeginIter());}
    ConstReverseIterator GetEndReverseIter()   const {return ConstReverseIterator(GetBeginIter());}

    reference GetFront()
    {
        NW4R_ASSERT(!IsEmpty());
        return *GetBeginIter();
    }
    const_reference GetFront() const
    {
        NW4R_ASSERT(!IsEmpty());
        return *GetBeginIter();
    }
    reference GetBack()
    {
        NW4R_ASSERT(!IsEmpty());
        return *--GetEndIter();
    }
    const_reference GetBack() const
    {
        NW4R_ASSERT(!IsEmpty());
        return *--GetEndIter();
    }

    /**************************************************************************/
    //*/ 要素の挿入/削除。

    void PushFront(pointer p) {(void)Insert(GetBeginIter(),p);}
    void PushBack(pointer p)  {(void)Insert(GetEndIter(),p);}
    Iterator PopFront()           {return Erase(GetBeginIter());}
    Iterator PopBack()            {return Erase(--GetEndIter());}

    Iterator Insert(Iterator it,pointer p);

    Iterator Erase(pointer p);
    Iterator Erase(const Iterator& it);
    Iterator Erase(const Iterator& itFirst, const Iterator& itLast);

    void Clear();

    /**************************************************************************/
    //*/ 要素の順序の変更。

    void Reverse();

    /**************************************************************************/
    //*/ 要素とノードとの変換。

    static Iterator GetIteratorFromPointer(pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return Iterator(p);
    }
    static ConstIterator GetIteratorFromPointer(const_pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return ConstIterator(p);
    }

private:  /****************************************************************/

    void Initialize_()
    {
        mSize=0;
        mNode.mNext = &mNode;
        mNode.mPrev = &mNode;
    }

    /**************************************************************************/

    size_type   mSize;  // 現在の要素数。
    Node        mNode;  // ノードリンクリストの[先端-1]兼[終端]。
};

} // namespace detail


template < typename T, PtrDiff TNOffset >
class LinkedList : private detail::LinkedListImpl
{
private:  /****************************************************************/

    typedef detail::LinkedListImpl Base;

public:  /*****************************************************************/

    typedef LinkedList Self;
    using Base::Node;

    using Base::size_type;
    using Base::difference_type;

    typedef T                  value_type;
    typedef value_type*        pointer;
    typedef const value_type*  const_pointer;
    typedef value_type&        reference;
    typedef const value_type&  const_reference;

private:  /****************************************************************/

    typedef Base::Iterator      TIt_base_;
    typedef Base::ConstIterator TItC_base_;

public:  /*****************************************************************/

    /**************************************************************************/
    //*/ イテレータ。

    class Iterator;
    class ConstIterator;

    class Iterator
        : public detail::Iterator<value_type>
    {
    public:  /***************************************************************/

        typedef Iterator TIt;
        typedef detail::Iterator<value_type> TBaseIt;

        /************************************************************************/

        explicit Iterator() {}

        /************************************************************************/

        reference operator*()  const
        {
            pointer p=operator->();
            NW4R_NULL_ASSERT(p);
            return *p;
        }
        pointer   operator->() const { return GetPointerFromNode(it_.operator->()); }

        TIt &operator++()
        {
            (void)++it_;
            return *this;
        }
        TIt operator++(int)
        {
            const TIt it(*this);
            (void)++*this;
            return it;
        }
        TIt &operator--()
        {
            (void)--it_;
            return *this;
        }
        TIt operator--(int)
        {
            const TIt it(*this);
            (void)--*this;
            return it;
        }

        friend bool operator==(const TIt& it1, const TIt& it2) { return it1.it_ == it2.it_; }
        friend bool operator!=(const TIt& it1, const TIt& it2) { return !(it1 == it2); }

    private:  /**************************************************************/

        explicit Iterator(TIt_base_ it) : it_(it) {}

        /************************************************************************/

        TIt_base_ it_;

        /************************************************************************/

//        friend Self;  // コンストラクタを使用するため。
        friend class ConstIterator;  // it_を使用するため。
    };

    typedef Iterator Iterator_alias_;  // ConstIterator内部で使用。

    class ConstIterator
        : public detail::Iterator<value_type>
    {
    public:  /***************************************************************/

        typedef ConstIterator TIt;
        typedef detail::Iterator<value_type> TBaseIt;
        typedef const_pointer pointer;
        typedef const_reference reference;

        /************************************************************************/

        explicit ConstIterator() {}
        explicit ConstIterator(Iterator_alias_ it) : it_(it.it_) {}

        /************************************************************************/

        reference operator*()  const
        {
            pointer p = operator->();
            NW4R_NULL_ASSERT(p);
            return *p;
        }
        pointer   operator->() const { return GetPointerFromNode(it_.operator->()); }

        TIt &operator++()
        {
            (void)++it_;
            return *this;
        }
        TIt operator++(int)
        {
            const TIt it(*this);
            (void)++*this;
            return it;
        }
        TIt &operator--()
        {
            (void)--it_;
            return *this;
        }
        TIt operator--(int)
        {
            const TIt it(*this);
            (void)--*this;
            return it;
        }

        friend bool operator==(const TIt& it1, const TIt& it2) { return it1.it_ == it2.it_; }
        friend bool operator!=(const TIt& it1, const TIt& it2) { return !(it1 == it2); }

    protected:  /************************************************************/

        explicit ConstIterator(TItC_base_ it) : it_(it) {}

        /************************************************************************/

        TItC_base_ it_;

        /************************************************************************/

//        friend Self;  // コンストラクタを使用するため。
    };

    typedef detail::ReverseIterator<Iterator> ReverseIterator;
    typedef detail::ReverseIterator<ConstIterator> ConstReverseIterator;

    /**************************************************************************/
    //*/ コンストラクタ/デストラクタ。

    explicit LinkedList() {}

    /**************************************************************************/
    //*/ 状態の問い合わせ、イテレータの取得など。

    using Base::GetSize;
    using Base::IsEmpty;

    Iterator       GetBeginIter()       { return Iterator(Base::GetBeginIter());}
    ConstIterator  GetBeginIter() const { return ConstIterator(const_cast<Self*>(this)->GetBeginIter());}
    Iterator       GetEndIter()         { return Iterator(Base::GetEndIter());}
    ConstIterator  GetEndIter()   const { return ConstIterator(const_cast<Self*>(this)->GetEndIter());}

    ReverseIterator       GetBeginReverseIter()       { return ReverseIterator(GetEndIter());}
    ConstReverseIterator  GetBeginReverseIter() const { return ConstReverseIterator(GetEndIter());}
    ReverseIterator       GetEndReverseIter()         { return ReverseIterator(GetBeginIter());}
    ConstReverseIterator  GetEndReverseIter()   const { return ConstReverseIterator(GetBeginIter());}

    reference GetFront()
    {
        NW4R_ASSERT(!IsEmpty());
        return *GetBeginIter();
    }
    const_reference GetFront() const
    {
        NW4R_ASSERT(!IsEmpty());
        return *GetBeginIter();
    }
    reference GetBack()
    {
        NW4R_ASSERT(!IsEmpty());
        return *--GetEndIter();
    }
    const_reference GetBack() const
    {
        NW4R_ASSERT(!IsEmpty());
        return *--GetEndIter();
    }

    /**************************************************************************/
    //*/ 要素の挿入/削除。

    Iterator Insert(const Iterator& it,pointer p) { return Iterator(Base::Insert(it.it_,GetNodeFromPointer(p)));}
    void     PushFront(pointer p)          { (void)Insert(GetBeginIter(),p);}
    void     PushBack(pointer p)           { (void)Insert(GetEndIter(),p);}

    using Base::PopFront;
    using Base::PopBack;

    Iterator Erase(const Iterator& it)                      { return Iterator(Base::Erase(it.it_));}
    Iterator Erase(const Iterator& itFirst, const Iterator& itLast) { return Iterator(Base::Erase(itFirst.it_,itLast.it_));}
    Iterator Erase(pointer p)                        { return Iterator(Base::Erase(GetNodeFromPointer(p)));}

    using Base::Clear;

    /**************************************************************************/
    //*/ 要素の順序の変更。

    using Base::Reverse;

    /**************************************************************************/
    //*/ 要素とノードとの変換。

    static Iterator GetIteratorFromPointer(Node *p)
    {
        NW4R_NULL_ASSERT(p);
        return Iterator(Base::GetIteratorFromPointer(p));
    }
    static ConstIterator GetIteratorFromPointer(const Node *p)
    {
        NW4R_NULL_ASSERT(p);
        return ConstIterator(Base::GetIteratorFromPointer(p));
    }
    static Iterator GetIteratorFromPointer(pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return GetIteratorFromPointer(GetNodeFromPointer(p));
    }
    static ConstIterator GetIteratorFromPointer(const_pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return GetIteratorFromPointer(GetNodeFromPointer(p));
    }

    static Node* GetNodeFromPointer(pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return reinterpret_cast<Node*>(reinterpret_cast<IntPtr>(p) + TNOffset);
    }
    static const Node* GetNodeFromPointer(const_pointer p)
    {
        NW4R_NULL_ASSERT(p);
        return reinterpret_cast<const Node*>(reinterpret_cast<IntPtr>(p) + TNOffset);
    }
    static pointer GetPointerFromNode(Node* p)
    {
        NW4R_NULL_ASSERT(p);
        return reinterpret_cast<pointer>(reinterpret_cast<IntPtr>(p) - TNOffset);
    }
    static const_pointer GetPointerFromNode(const Node* p)
    {
        NW4R_NULL_ASSERT(p);
        return reinterpret_cast<const_pointer>( reinterpret_cast<IntPtr>(p) - TNOffset);
    }
};

} // old_lib

// LinkedListNode のサイズを固定にするために、リストの item はポインタのみに限定する。
// ポインタは template で型をつけるべきだが、スラブアロケータで共通に初期化したいので
// template 化しない。


struct KLinkedListNode
{
public:
    old_lib::LinkedListNode m_LinkedListNode;
    void* m_pItem;

    explicit KLinkedListNode() : m_pItem(NULL)
    {
        NN_KERN_THIS_ASSERT();
    }
    ~KLinkedListNode(){}

    void Initialize(void* pItem)
    {
        NN_KERN_THIS_ASSERT();
        m_pItem = pItem;
    }

    static void InitializeSlab(void* pMem, size_t memSize)
    {
        s_NodeAllocator.Initialize(pMem, memSize);
    }

    static uintptr_t GetSlabAddr()           { return s_NodeAllocator.GetSlabAddr(); }
    static size_t    GetSlabSize()           { return s_NodeAllocator.GetSlabSize(); }
    static size_t    GetNumRemain()          { return s_NodeAllocator.GetNumRemain(); }
    static size_t    GetPeakNum()            { return s_NodeAllocator.GetPeakNum(); }
    static size_t    GetObjSize()            { return s_NodeAllocator.GetObjSize(); }


    static KLinkedListNode* Allocate()
    {
        return s_NodeAllocator.AllocateFromSlab();
    }

    static void Free(KLinkedListNode* pObj)
    {
        s_NodeAllocator.FreeToSlab(pObj);
    }

    static KSlabAllocator<KLinkedListNode> s_NodeAllocator;
};

template<typename T>
class KLinkedList :
    private old_lib::detail::LinkedListImpl
{
    static const size_t NODE_OFFSET = offsetof(KLinkedListNode, m_LinkedListNode);

    typedef old_lib::detail::LinkedListImpl Base;
    typedef Base::Iterator BaseIterator;
    typedef Base::ConstIterator BaseConstIterator;

    typedef Base::Node BaseNode;

public:

    class Iterator;
    class ConstIterator;

    class Iterator
    {
    public:
        explicit Iterator(){}
        ~Iterator(){}

        T&   operator*()  const
        {
            T* pItem = operator->();
            return *pItem;
        }
        T*   operator->() const { return static_cast<T*>(GetNodeFromBaseNode(m_It.operator->())->m_pItem);}

        Iterator& operator++()
        {
            ++m_It;
            return *this;
        }
        Iterator operator++(int)
        {
            const Iterator it(*this);
            (void)++*this;
            return it;
        }
        Iterator& operator--()
        {
            --m_It;
            return *this;
        }
        Iterator operator--(int)
        {
            const Iterator it(*this);
            (void)--*this;
            return it;
        }

        T*   GetCurrentItem() const { return operator->();}

        friend bool operator==(const Iterator& a, const Iterator& b) { return a.m_It == b.m_It; }
        friend bool operator!=(const Iterator& c, const Iterator& d) { return !(c == d); }

    private:
        explicit Iterator(BaseIterator it) : m_It(it) {}
        BaseIterator m_It;

        friend class KLinkedList<T>;
        friend class ConstIterator;
    };

    class ConstIterator
    {
    public:
        explicit ConstIterator() {}
        explicit ConstIterator(Iterator it) : m_It(it.m_It) {}

        const T&   operator*()  const
        {
            const T* pItem = operator->();
            return *pItem;
        }
        const T*   operator->() const { return static_cast<T*>(GetNodeFromBaseNode(m_It.operator->())->m_pItem); }

        ConstIterator& operator++()
        {
            ++m_It;
            return *this;
        }
        ConstIterator operator++(int)
        {
            const ConstIterator it(*this);
            (void)++*this; return it;
        }
        ConstIterator& operator--()
        {
            --m_It;
            return *this;
        }
        ConstIterator operator--(int)
        {
            const ConstIterator it(*this);
            (void)--*this;
            return it;
        }

        const T*   GetCurrentItem() const { return operator->();}

        friend bool operator==(const ConstIterator& a, const ConstIterator& b) { return a.m_It == b.m_It; }
        friend bool operator!=(const ConstIterator& c, const ConstIterator& d) { return !(c == d); }

    protected:

        explicit ConstIterator(BaseConstIterator it) : m_It(it) {}
        BaseConstIterator m_It;

        friend class KLinkedList<T>;
    };

    typedef old_lib::detail::ReverseIterator<Iterator> ReverseIterator;
    typedef old_lib::detail::ReverseIterator<ConstIterator> ConstReverseIterator;

    using Base::GetSize;
    using Base::IsEmpty;

    Iterator       GetBeginIter()       { return Iterator(Base::GetBeginIter());}
    ConstIterator  GetBeginIter() const { return ConstIterator(const_cast<KLinkedList<T>*>(this)->GetBeginIter());}
    Iterator       GetEndIter()         { return Iterator(Base::GetEndIter());}
    ConstIterator  GetEndIter()   const { return ConstIterator(const_cast<KLinkedList<T>*>(this)->GetEndIter());}

    ReverseIterator       GetBeginReverseIter()       { return ReverseIterator(GetEndIter());}
    ConstReverseIterator  GetBeginReverseIter() const { return ConstReverseIterator(GetEndIter());}
    ReverseIterator       GetEndReverseIter()         { return ReverseIterator(GetBeginIter());}
    ConstReverseIterator  GetEndReverseIter()   const { return ConstReverseIterator(GetBeginIter());}


    T* GetFront()
    {
        return GetBeginIter().GetCurrentItem();
    }
    const T* GetFront() const
    {
        return GetBeginIter().GetCurrentItem();
    }
    T* GetBack()
    {
        return (--GetEndIter()).GetCurrentItem();
    }
    const T* GetBack() const
    {
        return (--GetEndIter()).GetCurrentItem();
    }


    Iterator Insert(const Iterator& it, T* pItem)
    {
        KLinkedListNode* pNode = AllocateLinkedListNode(pItem);
        return Iterator(Base::Insert(it.m_It, &(pNode->m_LinkedListNode)));
    }

    void     PushFront(T* pItem)          { (void)Insert(GetBeginIter(), pItem);}
    void     PushBack(T* pItem)           { (void)Insert(GetEndIter(), pItem);}

    Iterator PopFront()                 { return Erase(GetBeginIter());}
    Iterator PopBack()                  { return Erase(GetEndIter());}

    Iterator Erase(const Iterator& it)
    {
        KLinkedListNode* pDeadNode  = GetNodeFromBaseNode(it.m_It.operator->());
        Iterator returnIt = Iterator(Base::Erase(it.m_It));
        FreeLinkedListNode(pDeadNode);

        return returnIt;
    }

    Iterator Erase(const Iterator& itFirst, const Iterator& itLast)
    {
        Iterator it = itFirst;
        while(it != itLast)
        {
            it = Erase(it);
        }
        return it;
    }

    void MoveToFront(const Iterator& it)
    {
        BaseNode* pBaseNode = it.m_It.operator->();
        Base::Erase(it.m_It);
        Base::PushFront(pBaseNode);
    }
    void MoveToBack(const Iterator& it)
    {
        BaseNode* pBaseNode = it.m_It.operator->();
        Base::Erase(it.m_It);
        Base::PushBack(pBaseNode);
    }

    // TODO: O(n) なので注意。
    bool Remove(T* pItem)
    {
        bool isRemoved = false;

        for(Iterator it = GetBeginIter(); it != GetEndIter(); )
        {
            if(it.GetCurrentItem() == pItem)
            {
                it = Erase(it);
                isRemoved = true;
                continue;
            }
            ++it;
        }

        return isRemoved;
    }
    bool RemoveOne(T* pItem)
    {
        const Iterator end = GetEndIter();

        for(Iterator it = GetBeginIter(); it != end; ++it)
        {
            if( it.GetCurrentItem() == pItem )
            {
                Erase(it);
                return true;
            }
        }

        return false;
    }

    void Clear()
    {
        Erase(GetBeginIter(), GetEndIter());
    }

    using Base::Reverse;



private:
    static KLinkedListNode* AllocateLinkedListNode(T* pItem)
    {
        KLinkedListNode* pNode = KLinkedListNode::Allocate();
        NN_KERN_ABORT_UNLESS(pNode);
        pNode->Initialize(pItem);
        return pNode;
    }

    static void FreeLinkedListNode(KLinkedListNode* deadLink)
    {
        KLinkedListNode::Free(deadLink);
    }

    // NintendoWare LinkedList.h からのコピペ。
    static Iterator GetIteratorFromBaseNode(BaseNode* pBaseNode)
    {
        return Iterator(Base::GetIteratorFromPointer(pBaseNode));
    }
    static ConstIterator GetIteratorFromBaseNode(const BaseNode* pBaseNode)
    {
        return ConstIterator(Base::GetIteratorFromPointer(pBaseNode));
    }
    static Iterator GetIteratorFromNode(KLinkedListNode* pNode)
    {
        return GetIteratorFromBaseNode(&(pNode->m_LinkedListNode));
    }
    static const Iterator GetIteratorFromNode(const KLinkedListNode* pNode)
    {
        return GetIteratorFromBaseNode(&(pNode->m_LinkedListNode));
    }

    static KLinkedListNode* GetNodeFromBaseNode(BaseNode* pBaseNode)
    {
        return reinterpret_cast<KLinkedListNode*>(reinterpret_cast<uintptr_t>(pBaseNode) - NODE_OFFSET);
    }
    static const KLinkedListNode* GetNodeFromBaseNode(const BaseNode* pBaseNode)
    {
        return reinterpret_cast<KLinkedListNode*>( reinterpret_cast<uintptr_t>(pBaseNode) - NODE_OFFSET);
    }
};

}}

