﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KLightServerSession.h"
#include "kern_KLightSession.h"
#include "kern_KScheduler.h"
#include "kern_Panic.h"
#include "kern_Utility.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KLightServerSession);

void KLightServerSession::Destroy()
{
    NN_KERN_THIS_ASSERT();

    // 同期待ちで待機しているリクエストを処理します
    CleanupRequest();

    m_pParent->OnServerClosed();
    // このタイミングで既に this が破棄されている可能性があるので
    // 以降、this にアクセスしてはいけない。
}

void KLightServerSession::OnClientClosed()
{
    NN_KERN_THIS_ASSERT();

    CleanupRequest();
}

void KLightServerSession::CleanupRequest()
{
    KThread* pCurrentRequest = nullptr;
    {
        KScopedSchedulingLock locker;
        if (m_pCurrentRequest)
        {
            if (!m_pCurrentRequest->IsTerminateRequested())
            {
                NN_KERN_ASSERT(m_pCurrentRequest->GetState() == KThread::STATE_WAIT);
                NN_KERN_ASSERT(m_pCurrentRequest == m_RequestQueue.GetFront());
                m_RequestQueue.WakupThread(m_pCurrentRequest);
                m_pCurrentRequest->SetSyncedObject(nullptr, nn::svc::ResultSessionClosed());
            }
            pCurrentRequest = m_pCurrentRequest;
            m_pCurrentRequest = nullptr;
            m_pServerThread = nullptr;
        }

        while (m_RequestQueue.IsNotEmpty())
        {
            KThread* pClient = m_RequestQueue.WakupFrontThread();
            pClient->SetSyncedObject(nullptr, nn::svc::ResultSessionClosed());
        }

        while (m_ServerQueue.IsNotEmpty())
        {
            m_ServerQueue.WakupFrontThread();
        }
    }
    if (pCurrentRequest)
    {
        pCurrentRequest->Close();
    }
}

Result KLightServerSession::ReplyAndReceive(Bit32* pData)
{
    KThread* pServerThread = &GetCurrentThread();
    pServerThread->SetLightSessionData(pData);

    KThread* pCurrentRequest = nullptr;
    if (pData[0] & KLightSession::ReplyFlag)
    {
        KScopedSchedulingLock locker;

        if (m_pParent->IsClientClosed() || (m_pParent->IsServerClosed()))
        {
            return nn::svc::ResultSessionClosed();
        }

        if (!m_pCurrentRequest)
        {
            return nn::svc::ResultInvalidState();
        }

        if (m_pServerThread != pServerThread)
        {
            return nn::svc::ResultInvalidState();
        }

        if (!m_pCurrentRequest->IsTerminateRequested())
        {
            NN_KERN_ASSERT(m_pCurrentRequest->GetState() == KThread::STATE_WAIT);
            NN_KERN_ASSERT(m_pCurrentRequest == m_RequestQueue.GetFront());
            ::std::memcpy(m_pCurrentRequest->GetLightSessionData(), pServerThread->GetLightSessionData(), KLightSession::DataSize);
            m_RequestQueue.WakupThread(m_pCurrentRequest);
        }

        pCurrentRequest = m_pCurrentRequest;
        m_pCurrentRequest = nullptr;
        m_pServerThread = nullptr;
    }

    if (pCurrentRequest)
    {
        pCurrentRequest->Close();
    }

    bool setCanCancel = false;
    for (;;)
    {
        KScopedSchedulingLock locker;

        if (m_ServerQueue.IsNotEmpty() || m_pServerThread)
        {
            return nn::svc::ResultInvalidState();
        }

        if (setCanCancel)
        {
            pServerThread->ClearCanCancel();
            setCanCancel = false;
        }

        if (m_pParent->IsClientClosed() || (m_pParent->IsServerClosed()))
        {
            return nn::svc::ResultSessionClosed();
        }

        if (!m_pCurrentRequest && m_RequestQueue.IsNotEmpty())
        {
            m_pCurrentRequest = m_RequestQueue.GetFront();
            pCurrentRequest = m_pCurrentRequest;
            pCurrentRequest->Open();
            m_pServerThread = pServerThread;
            break;
        }
        else
        {
            if (m_ServerQueue.SleepThread(pServerThread))
            {
                if (pServerThread->IsWaitCanceled())
                {
                    m_ServerQueue.WakupThread(pServerThread);
                    pServerThread->ClearWaitCanceled();
                    return nn::svc::ResultCancelled();
                }
                pServerThread->SetCanCancel();
                setCanCancel = true;
            }
            else
            {
                return nn::svc::ResultTerminateRequested();
            }
        }
    }

    ::std::memcpy(pServerThread->GetLightSessionData(), pCurrentRequest->GetLightSessionData(), KLightSession::DataSize);
    return ResultSuccess();
}

//!< クライアントセッションからリクエストが届いたことの通知
nn::Result KLightServerSession::OnRequest(KThread* pRequestSource)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT( KScheduler::IsSchedulerLocked() );
    Result result;

    if (m_pParent->IsServerClosed())
    {
        result = nn::svc::ResultSessionClosed();
    }
    else
    {
        if (m_RequestQueue.SleepThread(pRequestSource))
        {
            if (!m_pCurrentRequest)
            {
                m_ServerQueue.WakupFrontThread();
            }
            result = ResultSuccess();
        }
        else
        {
            result = nn::svc::ResultTerminateRequested();
        }
    }

    return result;
}


}}
