﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_Assert.h"
#include "kern_KInterruptTask.h"
#include "kern_Platform.h"
#include "kern_KThread.h"

namespace nn { namespace kern {

class KThread;
class KScheduler;
class KProcess;

// INFO: 現在の実装ではこのクラスのキューはスレッドセーフではない。
//       スケジューラ側での同期が必要。

/*!
    @brief     割り込みタスクを管理するクラスです。

*/
class KInterruptTaskManager
{
private:
    /*!
        @brief     割り込みタスクのキューを管理するクラスです。

    */
    class Queue
    {
    private:
        KInterruptTask* m_head;     //! 割り込みキューのリンクリスト先頭
        KInterruptTask* m_tail;     //! 割り込みキューのリンクリスト最後尾
    public:
        Queue() : m_head(0), m_tail(0) {}

        /*!
            @brief     キューの最後尾に KInterruptTask を追加します。

            @param[in]  task   追加対象の KInterruptTask

        */
        void            Enqueue(KInterruptTask* task);

        /*!
            @brief     キューの先頭から KInterruptTask を 1 つ取り出します。

            @return     キューの先頭にあった KInterruptTask

        */
        void            Dequeue();

        /*!
            @brief     キューの先頭にある KInterruptTask を 1 つ取得します。

            キューから取り除きません。

            @return     キューの先頭にある KInterruptTask

        */
        KInterruptTask* GetFront() const { return m_head; }

        /*!
            @brief     キューは空かどうか取得します。

            @return    キューが空であれば true

        */
        bool            IsEmpty() const { return m_head == 0; }

        /*!
            @brief     キューをクリアします。

        */
        void            Clear() { m_head = m_tail = 0; }
    };

private:
    Queue       m_Queue;            //!< 割り込みタスクのキューです
    KThread*    m_pThread;          //!< 遅延呼び出しスレッド

public:

    /*!
        @brief     割り込みタスクマネージャを初期化します。

    */
    void        Initialize();

    /*!
        @brief     キューにタスクを追加します。

        この関数は KInterruptManager::OnInterruptRequest 内から割り込み禁止状態のまま
        呼び出されるべきです。

    */
    void        EnqueueTask(KInterruptTask* task);

    /*!
        @brief     割り込みの遅延呼び出しスレッドを取得します。

        @return    割り込みの遅延呼び出しスレッドのポインタ

    */
    KThread*    GetThread() { return m_pThread; }

private:
    /*!
        @brief     割り込みの遅延呼び出しメインループ

        この関数は ThreadFunc から呼び出されるべきです。

    */
    void        ThreadFuncBody();

    /*!
        @brief     スレッドエントリー

    */
    static void ThreadFunc(uintptr_t param);
};

}}

