﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KInterruptTaskManager.h"
#include "kern_KMemoryManager.h"
#include "kern_KScheduler.h"
#include "kern_KProcess.h"
#include "kern_Utility.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

/*!
    @brief     キューの最後尾に KInterruptTask を追加します

    @param[in]  task   追加対象の KInterruptTask

*/
void KInterruptTaskManager::Queue::Enqueue(KInterruptTask* task)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERTMSG(task->GetNextTask() == 0, "%p %p", task, task->GetNextTask());
    NN_KERN_ASSERT(task != m_head);
    NN_KERN_ASSERT(task != m_tail);

    if (m_tail)
    {
        m_tail->SetNextTask(task);
        m_tail = task;
    }
    else
    {
        m_head = m_tail = task;
    }
    #ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
    task->SetNextTask(reinterpret_cast<KInterruptTask*>(1));
    #endif
}

/*!
    @brief     キューの先頭から KInterruptTask を1つ取り出します

    @return     キューの先頭にあった KInterruptTask

*/
void KInterruptTaskManager::Queue::Dequeue()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(m_head);
    NN_KERN_ASSERT(m_tail);
    #ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
    NN_KERN_ASSERT(m_tail->GetNextTask() == reinterpret_cast<KInterruptTask*>(1));
    #endif

    KInterruptTask* head = m_head;
    if (m_head == m_tail)
    {
        m_head = m_tail = 0;
    }
    else
    {
        m_head = head->GetNextTask();
    }
    #ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
    head->SetNextTask(0);
    #endif
}

/*!
    @brief     初期化します

*/
void KInterruptTaskManager::Initialize()
{
    // TODO: エラー処理
    NN_KERN_THIS_ASSERT();
    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_ThreadCountMax, 1));

    // 割り込みの遅延呼び出しスレッドを生成します
    m_pThread = KThread::Create();
    InitializeThreadForRealTime(
        m_pThread,
        &ThreadFunc,
        reinterpret_cast<uintptr_t>(this) );
    KThread::Register(m_pThread);

    m_pThread->Run();

    NN_LOG("    TaskThread ID=%lld, Core=%d, Affinity=%llx, State=%d, Priority=%d\n",
        m_pThread->GetId(), m_pThread->GetRunningProcessor(),
        m_pThread->GetAffinityMask().GetAffinityMask(),
        m_pThread->GetState(), m_pThread->GetPriority());
}

/*!
    @brief     スレッドエントリー

*/
void KInterruptTaskManager::ThreadFunc(uintptr_t param)
{
    reinterpret_cast<KInterruptTaskManager*>(param)->ThreadFuncBody();
}

/*!
    @brief     割り込みの遅延呼び出しメインループです

    このループはリアルタイム優先度で動作します。
    EnqueueTask される度に起床し、キューに詰まれたタスクを処理します。

*/
void KInterruptTaskManager::ThreadFuncBody()
{
    NN_KERN_THIS_ASSERT();

    for(;;)
    {
        KInterruptTask* task;

        {
            KDisableInterrupt di;

            task = m_Queue.GetFront();
            if (task == nullptr)
            {
                GetCurrentThread().DisableDispatch();
                m_pThread->SetState(KThread::STATE_WAIT);
                GetCurrentThread().EnableDispatch();
                GetCoreScheduler().ForceScheduling();
                continue;
            }

            m_Queue.Dequeue();
        }

        task->DoInterruptTask();
    }
}

/*!
    @brief     キューにタスクを追加します。

    この関数は KInterruptManager::OnInterruptRequest 内から割り込み禁止状態のまま
    呼び出されるべきです。

*/
void KInterruptTaskManager::EnqueueTask(KInterruptTask* task)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT( ! KInterruptManager::IsInterruptEnabled() );

    m_Queue.Enqueue(task);
    GetCoreScheduler().SetInterruptTaskRunnable();
}

}}
